/*
 * LPFAdj.c - Define the low pass filter adjustment table.
 *
 * Copyright (C) 2025 Martin Guy <martinwguy@gmail.com>
 * based on dolbybcsoftwaredecode by Richard Evans 2018.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation. See COPYING for details.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "LPFAdj.h"

#include <stdint.h>
#include <math.h>

#define LPFadjTabSize   800

static uint16_t const LPFadjTab[LPFadjTabSize] = {
  17289,17258,17226,17196,17166,17134,17103,17074,17042,17018,
  16983,16952,16921,16892,16864,16833,16803,16773,16744,16727,
  16692,16656,16628,16601,16585,16550,16514,16487,16457,16436,
  16401,16374,16346,16319,16291,16264,16236,16209,16181,16164,
  16129,16102,16074,16050,16022,15996,15969,15944,15917,15901,
  15866,15842,15815,15790,15765,15739,15715,15690,15665,15654,
  15619,15591,15566,15543,15520,15495,15470,15448,15423,15410,
  15376,15354,15329,15306,15340,15360,15325,15276,15207,15166,
  15141,15126,15103,15081,15062,15038,15016,14996,14972,14963,
  14930,14909,14888,14867,14849,14827,14805,14785,14764,14799,
  14820,14785,14736,14667,14626,14601,14606,14588,14567,14558,
  14529,14511,14492,14473,14454,14437,14418,14400,14381,14404,
  14369,14327,14302,14293,14328,14349,14314,14265,14206,14171,
  14150,14155,14138,14122,14108,14090,14073,14057,14041,14061,
  14026,13994,13975,13964,13951,13933,13918,13904,13887,13909,
  13874,13844,13826,13814,13800,13787,13772,13759,13743,13748,
  13717,13698,13689,13679,13664,13651,13640,13626,13612,13610,
  13586,13576,13563,13550,13585,13606,13571,13522,13491,13472,
  13467,13456,13445,13434,13424,13412,13400,13389,13379,13377,
  13357,13349,13338,13327,13317,13309,13297,13288,13278,13313,
  13334,13299,13250,13221,13204,13214,13205,13197,13187,13180,
  13170,13162,13154,13146,13141,13133,13123,13114,13108,13122,
  13093,13076,13078,13072,13068,13057,13051,13045,13038,13042,
  13026,13021,13013,13007,13004,12995,12989,12985,12979,13014,
  13035,13000,12957,12931,12915,12940,12936,12937,12940,12975,
  12996,12961,12912,12883,12866,12901,12922,12895,12879,12914,
  12935,12900,12879,12878,12876,12872,12868,12867,12863,12867,
  12859,12858,12855,12854,12889,12910,12875,12848,12832,12859,
  12875,12843,12824,12843,12843,12841,12840,12840,12840,12851,
  12840,12842,12841,12841,12843,12844,12843,12843,12844,12879,
  12900,12865,12844,12851,12853,12854,12855,12860,12860,12863,
  12864,12868,12868,12871,12877,12877,12880,12882,12886,12921,
  12942,12907,12886,12906,12941,12962,12927,12906,12924,12939,
  12932,12938,12944,12947,12955,12958,12962,12968,12973,12990,
  12985,12991,12996,13002,13008,13015,13021,13029,13034,13069,
  13090,13057,13037,13069,13088,13086,13091,13100,13108,13143,
  13164,13132,13113,13148,13169,13166,13175,13186,13193,13214,
  13212,13222,13231,13241,13276,13297,13271,13255,13290,13339,
  13368,13385,13350,13329,13358,13375,13380,13392,13403,13424,
  13428,13441,13452,13465,13479,13492,13502,13516,13529,13564,
  13585,13570,13583,13597,13613,13625,13639,13655,13669,13694,
  13698,13715,13728,13744,13762,13776,13791,13807,13822,13840,
  13855,13872,13888,13907,13942,13963,13957,13977,13993,14020,
  14036,14048,14065,14083,14104,14120,14140,14159,14178,14213,
  14234,14237,14256,14280,14300,14318,14339,14361,14380,14415,
  14436,14444,14466,14488,14513,14534,14555,14578,14600,14635,
  14656,14670,14693,14719,14744,14766,14790,14814,14839,14874,
  14895,14916,14940,14966,15002,15024,15044,15071,15097,15133,
  15155,15179,15206,15234,15264,15290,15318,15349,15376,15414,
  15437,15466,15493,15523,15556,15586,15614,15646,15676,15718,
  15743,15771,15803,15837,15871,15901,15934,15969,16000,16043,
  16069,16104,16136,16172,16209,16241,16276,16312,16349,16401,
  16432,16458,16494,16535,16592,16626,16646,16681,16723,16774,
  16805,16840,16880,16920,16964,17003,17041,17083,17125,17184,
  17219,17251,17294,17338,17383,17424,17469,17512,17557,17620,
  17658,17695,17738,17785,17835,17881,17926,17973,18021,18088,
  18128,18167,18216,18266,18319,18367,18417,18471,18520,18585,
  18624,18679,18729,18784,18861,18907,18943,18993,19055,19113,
  19167,19225,19278,19341,19399,19454,19513,19574,19630,19706,
  19752,19813,19874,19934,19999,20059,20123,20185,20248,20336,
  20389,20444,20509,20577,20647,20710,20777,20846,20915,20996,
  21050,21124,21194,21265,21340,21412,21482,21554,21630,21736,
  21800,21855,21929,22013,22131,22202,22245,22305,22389,22499,
  22565,22646,22724,22807,22892,22977,23058,23145,23230,23349,
  23420,23493,23580,23672,23765,23851,23941,24036,24126,24252,
  24328,24411,24504,24600,24697,24796,24893,24994,25093,25218,
  25296,25400,25503,25609,25714,25812,25925,26034,26137,26270,
  26358,26471,26582,26695,26853,26948,27038,27154,27272,27437,
  27536,27630,27751,27872,28002,28121,28245,28375,28500,28631,
  28757,28892,29020,29153,29291,29423,29558,29694,29831,30023,
  30138,30256,30399,30548,30694,30835,30985,31137,31285,31465,
  31588,31744,31900,32060,32223,32380,32538,32701,32865,33065,
  33199,33369,33538,33712,33956,34102,34235,34417,34591,34788,
  34957,35147,35328,35517,35714,35898,36092,36287,36482,36755,
  36919,37088,37290,37497,37711,37920,38128,38348,38578,38900,
  39093,39216,39388,39629,39906,40137,40370,40607,40840,41166,
  41362,41573,41817,42069,42331,42580,42834,43104,43363,43663,
  43894,44175,44442,44722,45018,45282,45570,45861,46149,46526,
  46752,47045,47350,47665,47978,48288,48602,48929,49249,49586,
  49913,50253,50585,50930,51288,51638,51985,52345,52706,55450
};


double LPFadj(double FCRatio)
{
  int32_t TabLPs, TabHPs;
  uint16_t LowVal, HigVal;
  double DifVal;

  /* Set table positions */
  TabLPs = round((FCRatio * 1000.0) - 0.5);
  if (TabLPs < 1)             TabLPs = 1;
  if (TabLPs > LPFadjTabSize) TabLPs = LPFadjTabSize;
  if (TabLPs < LPFadjTabSize) TabHPs = TabLPs + 1;
  else                        TabHPs = TabLPs;

  /* Set values from table */
  LowVal = LPFadjTab[TabLPs-1];
  HigVal = LPFadjTab[TabHPs-1];

  /* Work out exact position above low value */
  if (FCRatio > 0.001)
    DifVal = 1000 * FCRatio - TabLPs;
  else
    DifVal = 0.0;

  /* Work out how much to add to LowVal */
  DifVal *= HigVal - LowVal;

  /* Return the result */
  return ((LowVal + DifVal) / 10000.0);
}
