/***************************************************************************
    qgsattributesformproperties.cpp
    ---------------------
    begin                : August 2017
    copyright            : (C) 2017 by David Signer
    email                : david at opengis dot ch
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qgsactionmanager.h"
#include "qgsaddtaborgroup.h"
#include "qgsattributesformproperties.h"
#include "moc_qgsattributesformproperties.cpp"
#include "qgsattributetypedialog.h"
#include "qgsattributeformcontaineredit.h"
#include "qgsattributewidgetedit.h"
#include "qgsattributesforminitcode.h"
#include "qgsqmlwidgetwrapper.h"
#include "qgshtmlwidgetwrapper.h"
#include "qgsapplication.h"
#include "qgscodeeditor.h"
#include "qgscodeeditorhtml.h"
#include "qgsexpressioncontextutils.h"
#include "qgssettingsregistrycore.h"
#include "qgstextwidgetwrapper.h"
#include "qgsgui.h"
#include "qgseditorwidgetregistry.h"
#include "qgscodeeditorexpression.h"
#include "qgsfieldcombobox.h"
#include "qgsexpressionfinder.h"
#include "qgsexpressionbuilderdialog.h"
#include "qgshelp.h"
#include "qgsxmlutils.h"

#ifdef ENABLE_MODELTEST
#include "modeltest.h"
#endif

const QgsSettingsEntryBool *QgsAttributesFormProperties::settingShowAliases = new QgsSettingsEntryBool( QStringLiteral( "show-aliases" ), sTreeAttributesForm, false, QStringLiteral( "Whether to show aliases (true) or names (false) in both the Available Widgets and the Form Layout panels." ) );

QgsAttributesFormProperties::QgsAttributesFormProperties( QgsVectorLayer *layer, QWidget *parent )
  : QWidget( parent )
  , mLayer( layer )
{
  if ( !layer )
    return;

  setupUi( this );

  mEditorLayoutComboBox->addItem( tr( "Autogenerate" ), QVariant::fromValue( Qgis::AttributeFormLayout::AutoGenerated ) );
  mEditorLayoutComboBox->addItem( tr( "Drag and Drop Designer" ), QVariant::fromValue( Qgis::AttributeFormLayout::DragAndDrop ) );
  mEditorLayoutComboBox->addItem( tr( "Provide ui-file" ), QVariant::fromValue( Qgis::AttributeFormLayout::UiFile ) );
  mShowAliasesButton->setChecked( settingShowAliases->value() );

  // available widgets tree
  QGridLayout *availableWidgetsWidgetLayout = new QGridLayout;
  mAvailableWidgetsView = new QgsAttributesAvailableWidgetsView( mLayer, this );
  availableWidgetsWidgetLayout->addWidget( mAvailableWidgetsView );
  availableWidgetsWidgetLayout->setContentsMargins( 0, 0, 0, 0 );
  mAvailableWidgetsWidget->setLayout( availableWidgetsWidgetLayout );
  mAvailableWidgetsView->setContextMenuPolicy( Qt::CustomContextMenu );

  mAvailableWidgetsModel = new QgsAttributesAvailableWidgetsModel( mLayer, QgsProject().instance(), this );
  mAvailableWidgetsProxyModel = new QgsAttributesFormProxyModel( this );
  mAvailableWidgetsProxyModel->setAttributesFormSourceModel( mAvailableWidgetsModel );
  mAvailableWidgetsProxyModel->setRecursiveFilteringEnabled( true );
  mAvailableWidgetsView->setModel( mAvailableWidgetsProxyModel );

#ifdef ENABLE_MODELTEST
  new ModelTest( mAvailableWidgetsProxyModel, this );
#endif

  // form layout tree
  QGridLayout *formLayoutWidgetLayout = new QGridLayout;
  mFormLayoutView = new QgsAttributesFormLayoutView( mLayer, this );
  formLayoutWidgetLayout->addWidget( mFormLayoutView );
  formLayoutWidgetLayout->setContentsMargins( 0, 0, 0, 0 );
  mFormLayoutWidget->setLayout( formLayoutWidgetLayout );

  mFormLayoutModel = new QgsAttributesFormLayoutModel( mLayer, QgsProject().instance(), this );
  mFormLayoutProxyModel = new QgsAttributesFormProxyModel( this );
  mFormLayoutProxyModel->setAttributesFormSourceModel( mFormLayoutModel );
  mFormLayoutProxyModel->setRecursiveFilteringEnabled( true );
  mFormLayoutView->setModel( mFormLayoutProxyModel );

#ifdef ENABLE_MODELTEST
  new ModelTest( mFormLayoutProxyModel, this );
#endif

  connect( mAvailableWidgetsView->selectionModel(), &QItemSelectionModel::selectionChanged, this, &QgsAttributesFormProperties::onAttributeSelectionChanged );
  connect( mFormLayoutView->selectionModel(), &QItemSelectionModel::selectionChanged, this, &QgsAttributesFormProperties::onFormLayoutSelectionChanged );

  connect( mAvailableWidgetsView, &QWidget::customContextMenuRequested, this, &QgsAttributesFormProperties::onContextMenuRequested );

  connect( mAddContainerButton, &QAbstractButton::clicked, this, &QgsAttributesFormProperties::addContainer );
  connect( mRemoveLayoutItemButton, &QAbstractButton::clicked, this, &QgsAttributesFormProperties::removeTabOrGroupButton );
  connect( mInvertSelectionButton, &QAbstractButton::clicked, this, &QgsAttributesFormProperties::onInvertSelectionButtonClicked );
  connect( mShowAliasesButton, &QAbstractButton::toggled, this, &QgsAttributesFormProperties::toggleShowAliases );
  connect( mEditorLayoutComboBox, static_cast<void ( QComboBox::* )( int )>( &QComboBox::currentIndexChanged ), this, &QgsAttributesFormProperties::mEditorLayoutComboBox_currentIndexChanged );
  connect( pbnSelectEditForm, &QToolButton::clicked, this, &QgsAttributesFormProperties::pbnSelectEditForm_clicked );
  connect( mTbInitCode, &QPushButton::clicked, this, &QgsAttributesFormProperties::mTbInitCode_clicked );

  connect( mSearchLineEdit, &QgsFilterLineEdit::textChanged, this, &QgsAttributesFormProperties::updateFilteredItems );

  connect( mLayer, &QgsVectorLayer::updatedFields, this, [this] {
    if ( !mBlockUpdates )
      updatedFields();
  } );

  // Context menu and children actions
  mAvailableWidgetsContextMenu = new QMenu( this );
  mActionCopyWidgetConfiguration = new QAction( tr( "Copy widget configuration" ), this );
  mActionPasteWidgetConfiguration = new QAction( tr( "Paste widget configuration" ), this );

  connect( mActionCopyWidgetConfiguration, &QAction::triggered, this, &QgsAttributesFormProperties::copyWidgetConfiguration );
  connect( mActionPasteWidgetConfiguration, &QAction::triggered, this, &QgsAttributesFormProperties::pasteWidgetConfiguration );

  mAvailableWidgetsContextMenu->addAction( mActionCopyWidgetConfiguration );
  mAvailableWidgetsContextMenu->addAction( mActionPasteWidgetConfiguration );

  mMessageBar = new QgsMessageBar();
  mMessageBar->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Fixed );
  gridLayout->addWidget( mMessageBar, 0, 0 );

  // Assign initial size to splitter widgets. By doing so, we can
  // show an eventual horizontal scrollbar in the right-hand side panel
  splitter->setSizes( { widget->minimumSizeHint().width(), 600 } );
}

void QgsAttributesFormProperties::init()
{
  initAvailableWidgetsView();
  initFormLayoutView();

  initLayoutConfig();
  initInitPython();
  initSuppressCombo();
}

void QgsAttributesFormProperties::initAvailableWidgetsView()
{
  mAvailableWidgetsView->setSortingEnabled( false );
  mAvailableWidgetsView->setSelectionBehavior( QAbstractItemView::SelectRows );
  mAvailableWidgetsView->setSelectionMode( QAbstractItemView::SelectionMode::ExtendedSelection );
  mAvailableWidgetsView->setAcceptDrops( false );
  mAvailableWidgetsView->setDragDropMode( QAbstractItemView::DragDropMode::DragOnly );

  mAvailableWidgetsModel->populate();
  mAvailableWidgetsModel->setShowAliases( settingShowAliases->value() );
  mAvailableWidgetsView->expandAll();
}

void QgsAttributesFormProperties::initFormLayoutView()
{
  // tabs and groups info
  mFormLayoutView->setSortingEnabled( false );
  mFormLayoutView->setSelectionBehavior( QAbstractItemView::SelectRows );
  mFormLayoutView->setSelectionMode( QAbstractItemView::SelectionMode::ExtendedSelection );
  mFormLayoutView->setAcceptDrops( true );
  mFormLayoutView->setDragDropMode( QAbstractItemView::DragDropMode::DragDrop );
  mFormLayoutView->setDefaultDropAction( Qt::MoveAction );

  mFormLayoutView->selectionModel()->clear();
  mFormLayoutModel->populate();
  mFormLayoutModel->setShowAliases( settingShowAliases->value() );
  mFormLayoutView->expandAll();
}


void QgsAttributesFormProperties::initSuppressCombo()
{
  if ( QgsSettingsRegistryCore::settingsDigitizingDisableEnterAttributeValuesDialog->value() )
  {
    mFormSuppressCmbBx->addItem( tr( "Hide Form on Add Feature (global settings)" ), QVariant::fromValue( Qgis::AttributeFormSuppression::Default ) );
  }
  else
  {
    mFormSuppressCmbBx->addItem( tr( "Show Form on Add Feature (global settings)" ), QVariant::fromValue( Qgis::AttributeFormSuppression::Default ) );
  }
  mFormSuppressCmbBx->addItem( tr( "Hide Form on Add Feature" ), QVariant::fromValue( Qgis::AttributeFormSuppression::On ) );
  mFormSuppressCmbBx->addItem( tr( "Show Form on Add Feature" ), QVariant::fromValue( Qgis::AttributeFormSuppression::Off ) );

  mFormSuppressCmbBx->setCurrentIndex( mFormSuppressCmbBx->findData( QVariant::fromValue( mLayer->editFormConfig().suppress() ) ) );
}

void QgsAttributesFormProperties::initAvailableWidgetsActions( const QList< QgsAction > actions )
{
  mAvailableWidgetsModel->populateLayerActions( actions );
}

QgsExpressionContext QgsAttributesFormProperties::createExpressionContext() const
{
  QgsExpressionContext context;
  context.appendScopes( QgsExpressionContextUtils::globalProjectLayerScopes( mLayer ) );
  return context;
}

void QgsAttributesFormProperties::initLayoutConfig()
{
  mEditorLayoutComboBox->setCurrentIndex( mEditorLayoutComboBox->findData( QVariant::fromValue( mLayer->editFormConfig().layout() ) ) );

  mEditorLayoutComboBox_currentIndexChanged( mEditorLayoutComboBox->currentIndex() );

  const QgsEditFormConfig cfg = mLayer->editFormConfig();
  mEditFormLineEdit->setText( cfg.uiForm() );
}

void QgsAttributesFormProperties::initInitPython()
{
  const QgsEditFormConfig cfg = mLayer->editFormConfig();

  mInitCodeSource = cfg.initCodeSource();
  mInitFunction = cfg.initFunction();
  mInitFilePath = cfg.initFilePath();
  mInitCode = cfg.initCode();

  if ( mInitCode.isEmpty() )
  {
    mInitCode.append( tr( "# -*- coding: utf-8 -*-\n\"\"\"\n"
                          "QGIS forms can have a Python function that is called when the form is\n"
                          "opened.\n"
                          "\n"
                          "Use this function to add extra logic to your forms.\n"
                          "\n"
                          "Enter the name of the function in the \"Python Init function\"\n"
                          "field.\n"
                          "An example follows:\n"
                          "\"\"\"\n"
                          "from qgis.PyQt.QtWidgets import QWidget\n\n"
                          "def my_form_open(dialog, layer, feature):\n"
                          "    geom = feature.geometry()\n"
                          "    control = dialog.findChild(QWidget, \"MyLineEdit\")\n" ) );
  }
}

void QgsAttributesFormProperties::loadAttributeTypeDialog()
{
  if ( mAvailableWidgetsView->selectionModel()->selectedRows( 0 ).count() != 1 )
    return;

  const QModelIndex index = mAvailableWidgetsView->firstSelectedIndex();

  const QgsAttributesFormData::FieldConfig cfg = mAvailableWidgetsModel->data( index, QgsAttributesFormModel::ItemFieldConfigRole ).value< QgsAttributesFormData::FieldConfig >();
  const QString fieldName = mAvailableWidgetsModel->data( index, QgsAttributesFormModel::ItemNameRole ).toString();
  const int fieldIndex = mLayer->fields().indexOf( fieldName );

  if ( fieldIndex < 0 )
    return;

  mAttributeTypeDialog = new QgsAttributeTypeDialog( mLayer, fieldIndex, mAttributeTypeFrame );

  loadAttributeTypeDialogFromConfiguration( cfg );

  mAttributeTypeDialog->setDefaultValueExpression( mLayer->defaultValueDefinition( fieldIndex ).expression() );
  mAttributeTypeDialog->setApplyDefaultValueOnUpdate( mLayer->defaultValueDefinition( fieldIndex ).applyOnUpdate() );

  mAttributeTypeDialog->layout()->setContentsMargins( 0, 0, 0, 0 );
  mAttributeTypeFrame->layout()->setContentsMargins( 0, 0, 0, 0 );

  mAttributeTypeFrame->layout()->addWidget( mAttributeTypeDialog );
}

void QgsAttributesFormProperties::loadAttributeTypeDialogFromConfiguration( const QgsAttributesFormData::FieldConfig &config )
{
  const QgsFieldConstraints constraints = config.mFieldConstraints;

  mAttributeTypeDialog->setAlias( config.mAlias );
  mAttributeTypeDialog->setDataDefinedProperties( config.mDataDefinedProperties );
  mAttributeTypeDialog->setComment( config.mComment );
  mAttributeTypeDialog->setFieldEditable( config.mEditable );
  mAttributeTypeDialog->setLabelOnTop( config.mLabelOnTop );
  mAttributeTypeDialog->setReuseLastValues( config.mReuseLastValues );
  mAttributeTypeDialog->setNotNull( constraints.constraints() & QgsFieldConstraints::ConstraintNotNull );
  mAttributeTypeDialog->setNotNullEnforced( constraints.constraintStrength( QgsFieldConstraints::ConstraintNotNull ) == QgsFieldConstraints::ConstraintStrengthHard );
  mAttributeTypeDialog->setUnique( constraints.constraints() & QgsFieldConstraints::ConstraintUnique );
  mAttributeTypeDialog->setUniqueEnforced( constraints.constraintStrength( QgsFieldConstraints::ConstraintUnique ) == QgsFieldConstraints::ConstraintStrengthHard );
  mAttributeTypeDialog->setSplitPolicy( config.mSplitPolicy );
  mAttributeTypeDialog->setDuplicatePolicy( config.mDuplicatePolicy );
  mAttributeTypeDialog->setMergePolicy( config.mMergePolicy );

  QgsFieldConstraints::Constraints providerConstraints = QgsFieldConstraints::Constraints();
  if ( constraints.constraintOrigin( QgsFieldConstraints::ConstraintNotNull ) == QgsFieldConstraints::ConstraintOriginProvider )
    providerConstraints |= QgsFieldConstraints::ConstraintNotNull;
  if ( constraints.constraintOrigin( QgsFieldConstraints::ConstraintUnique ) == QgsFieldConstraints::ConstraintOriginProvider )
    providerConstraints |= QgsFieldConstraints::ConstraintUnique;
  if ( constraints.constraintOrigin( QgsFieldConstraints::ConstraintExpression ) == QgsFieldConstraints::ConstraintOriginProvider )
    providerConstraints |= QgsFieldConstraints::ConstraintExpression;
  mAttributeTypeDialog->setProviderConstraints( providerConstraints );

  mAttributeTypeDialog->setConstraintExpression( constraints.constraintExpression() );
  mAttributeTypeDialog->setConstraintExpressionDescription( constraints.constraintDescription() );
  mAttributeTypeDialog->setConstraintExpressionEnforced( constraints.constraintStrength( QgsFieldConstraints::ConstraintExpression ) == QgsFieldConstraints::ConstraintStrengthHard );

  // Make sure the widget is refreshed, even if
  // the new widget type matches the current one
  mAttributeTypeDialog->setEditorWidgetConfig( config.mEditorWidgetConfig );
  mAttributeTypeDialog->setEditorWidgetType( config.mEditorWidgetType, true );
}

void QgsAttributesFormProperties::storeAttributeTypeDialog()
{
  if ( !mAttributeTypeDialog )
    return;

  if ( mAttributeTypeDialog->fieldIdx() < 0 || mAttributeTypeDialog->fieldIdx() >= mLayer->fields().count() )
    return;

  QgsAttributesFormData::FieldConfig cfg;

  cfg.mComment = mLayer->fields().at( mAttributeTypeDialog->fieldIdx() ).comment();
  cfg.mEditable = mAttributeTypeDialog->fieldEditable();
  cfg.mLabelOnTop = mAttributeTypeDialog->labelOnTop();
  cfg.mReuseLastValues = mAttributeTypeDialog->reuseLastValues();
  cfg.mAlias = mAttributeTypeDialog->alias();
  cfg.mDataDefinedProperties = mAttributeTypeDialog->dataDefinedProperties();

  QgsFieldConstraints constraints;
  if ( mAttributeTypeDialog->notNull() )
  {
    constraints.setConstraint( QgsFieldConstraints::ConstraintNotNull );
  }
  else if ( mAttributeTypeDialog->notNullFromProvider() )
  {
    constraints.setConstraint( QgsFieldConstraints::ConstraintNotNull, QgsFieldConstraints::ConstraintOriginProvider );
  }

  if ( mAttributeTypeDialog->unique() )
  {
    constraints.setConstraint( QgsFieldConstraints::ConstraintUnique );
  }
  else if ( mAttributeTypeDialog->uniqueFromProvider() )
  {
    constraints.setConstraint( QgsFieldConstraints::ConstraintUnique, QgsFieldConstraints::ConstraintOriginProvider );
  }

  if ( !mAttributeTypeDialog->constraintExpression().isEmpty() )
  {
    constraints.setConstraint( QgsFieldConstraints::ConstraintExpression );
  }

  constraints.setConstraintExpression( mAttributeTypeDialog->constraintExpression(), mAttributeTypeDialog->constraintExpressionDescription() );

  constraints.setConstraintStrength( QgsFieldConstraints::ConstraintNotNull, mAttributeTypeDialog->notNullEnforced() ? QgsFieldConstraints::ConstraintStrengthHard : QgsFieldConstraints::ConstraintStrengthSoft );
  constraints.setConstraintStrength( QgsFieldConstraints::ConstraintUnique, mAttributeTypeDialog->uniqueEnforced() ? QgsFieldConstraints::ConstraintStrengthHard : QgsFieldConstraints::ConstraintStrengthSoft );
  constraints.setConstraintStrength( QgsFieldConstraints::ConstraintExpression, mAttributeTypeDialog->constraintExpressionEnforced() ? QgsFieldConstraints::ConstraintStrengthHard : QgsFieldConstraints::ConstraintStrengthSoft );

  // The call to mLayer->setDefaultValueDefinition will possibly emit updatedFields
  // which will set mAttributeTypeDialog to nullptr so we need to store any value before calling it
  cfg.mFieldConstraints = constraints;
  cfg.mEditorWidgetType = mAttributeTypeDialog->editorWidgetType();
  cfg.mEditorWidgetConfig = mAttributeTypeDialog->editorWidgetConfig();
  cfg.mSplitPolicy = mAttributeTypeDialog->splitPolicy();
  cfg.mDuplicatePolicy = mAttributeTypeDialog->duplicatePolicy();
  cfg.mMergePolicy = mAttributeTypeDialog->mergePolicy();

  const int fieldIndex = mAttributeTypeDialog->fieldIdx();
  mLayer->setDefaultValueDefinition( fieldIndex, QgsDefaultValue( mAttributeTypeDialog->defaultValueExpression(), mAttributeTypeDialog->applyDefaultValueOnUpdate() ) );

  const QString fieldName = mLayer->fields().at( fieldIndex ).name();

  QModelIndex index = mAvailableWidgetsModel->fieldModelIndex( fieldName );
  if ( index.isValid() )
  {
    mAvailableWidgetsModel->setData( index, QVariant::fromValue<QgsAttributesFormData::FieldConfig>( cfg ), QgsAttributesFormModel::ItemFieldConfigRole );
    mAvailableWidgetsModel->setData( index, mAttributeTypeDialog->alias(), QgsAttributesFormModel::ItemDisplayRole );
  }

  // Save alias to each matching field item in Form Layout model
  mFormLayoutModel->updateAliasForFieldItems( fieldName, mAttributeTypeDialog->alias() );
}

void QgsAttributesFormProperties::storeAttributeWidgetEdit()
{
  if ( !mAttributeWidgetEdit )
    return;

  if ( mFormLayoutView->selectionModel()->selectedRows().count() != 1 )
    return;

  QModelIndex index = mFormLayoutView->firstSelectedIndex();
  storeAttributeWidgetEdit( index );
}

void QgsAttributesFormProperties::storeAttributeWidgetEdit( const QModelIndex &index )
{
  if ( !mAttributeWidgetEdit )
    return;

  if ( !index.isValid() )
    return;

  auto itemData = index.data( QgsAttributesFormLayoutModel::ItemDataRole ).value< QgsAttributesFormData::AttributeFormItemData >();
  mAttributeWidgetEdit->updateItemData( itemData );

  if ( index.data( QgsAttributesFormModel::ItemTypeRole ) == QgsAttributesFormData::Relation )
  {
    QgsAttributesFormData::RelationEditorConfiguration config = mAttributeWidgetEdit->updatedRelationConfiguration();
    itemData.setRelationEditorConfiguration( config );
    mFormLayoutModel->setData( index, config.label, QgsAttributesFormLayoutModel::ItemDisplayRole );
  }
  mFormLayoutModel->setData( index, itemData, QgsAttributesFormLayoutModel::ItemDataRole );
}

void QgsAttributesFormProperties::loadAttributeWidgetEdit()
{
  if ( mFormLayoutView->selectionModel()->selectedRows().count() != 1 )
    return;

  const QModelIndex currentIndex = mFormLayoutView->firstSelectedIndex();
  const QgsAttributesFormData::AttributeFormItemData itemData = currentIndex.data( QgsAttributesFormModel::ItemDataRole ).value< QgsAttributesFormData::AttributeFormItemData >();
  mAttributeWidgetEdit = new QgsAttributeWidgetEdit( itemData, this );
  if ( currentIndex.data( QgsAttributesFormModel::ItemTypeRole ) == QgsAttributesFormData::Relation )
  {
    mAttributeWidgetEdit->setRelationSpecificWidget( itemData.relationEditorConfiguration(), currentIndex.data( QgsAttributesFormModel::ItemIdRole ).toString() );
  }
  mAttributeTypeFrame->layout()->setContentsMargins( 0, 0, 0, 0 );
  mAttributeTypeFrame->layout()->addWidget( mAttributeWidgetEdit );
}

void QgsAttributesFormProperties::loadInfoWidget( const QString &infoText )
{
  mInfoTextWidget = new QLabel( infoText );
  mAttributeTypeFrame->layout()->setContentsMargins( 0, 0, 0, 0 );
  mAttributeTypeFrame->layout()->addWidget( mInfoTextWidget );
}

void QgsAttributesFormProperties::storeAttributeContainerEdit()
{
  if ( !mAttributeContainerEdit )
    return;

  if ( mFormLayoutView->selectionModel()->selectedRows().count() != 1 )
    return;

  const QModelIndex currentIndex = mFormLayoutView->firstSelectedIndex();
  storeAttributeContainerEdit( currentIndex );
}

void QgsAttributesFormProperties::storeAttributeContainerEdit( const QModelIndex &index )
{
  if ( !mAttributeContainerEdit )
    return;

  if ( !index.isValid() )
    return;

  auto itemData = index.data( QgsAttributesFormModel::ItemDataRole ).value< QgsAttributesFormData::AttributeFormItemData >();
  QString containerName;

  mAttributeContainerEdit->updateItemData( itemData, containerName );
  mFormLayoutModel->setData( index, itemData, QgsAttributesFormLayoutModel::ItemDataRole );
  mFormLayoutModel->setData( index, containerName, QgsAttributesFormLayoutModel::ItemNameRole );
}

void QgsAttributesFormProperties::loadAttributeContainerEdit()
{
  if ( mFormLayoutView->selectionModel()->selectedRows().count() != 1 )
    return;

  const QModelIndex currentIndex = mFormLayoutView->firstSelectedIndex();
  const QgsAttributesFormData::AttributeFormItemData itemData = currentIndex.data( QgsAttributesFormModel::ItemDataRole ).value< QgsAttributesFormData::AttributeFormItemData >();
  mAttributeContainerEdit = new QgsAttributeFormContainerEdit( itemData, mLayer, this );
  mAttributeContainerEdit->setTitle( currentIndex.data( QgsAttributesFormModel::ItemNameRole ).toString() );
  mAttributeContainerEdit->setUpContainerTypeComboBox( !currentIndex.parent().isValid(), itemData.containerType() );

  mAttributeContainerEdit->registerExpressionContextGenerator( this );
  mAttributeContainerEdit->layout()->setContentsMargins( 0, 0, 0, 0 );
  mAttributeTypeFrame->layout()->setContentsMargins( 0, 0, 0, 0 );
  mAttributeTypeFrame->layout()->addWidget( mAttributeContainerEdit );
}

void QgsAttributesFormProperties::onAttributeSelectionChanged( const QItemSelection &, const QItemSelection & )
{
  disconnect( mFormLayoutView->selectionModel(), &QItemSelectionModel::selectionChanged, this, &QgsAttributesFormProperties::onFormLayoutSelectionChanged );

  QModelIndex index;
  if ( mFormLayoutView->selectionModel()->selectedRows( 0 ).count() == 1 )
  {
    // Go to the form layout view and store the single-selected index, as
    // it will be used to store its current settings before being deselected
    index = mFormLayoutView->firstSelectedIndex();
  }

  loadAttributeSpecificEditor( mAvailableWidgetsView, mFormLayoutView, index );
  connect( mFormLayoutView->selectionModel(), &QItemSelectionModel::selectionChanged, this, &QgsAttributesFormProperties::onFormLayoutSelectionChanged );
}

void QgsAttributesFormProperties::onFormLayoutSelectionChanged( const QItemSelection &, const QItemSelection &deselected )
{
  // when the selection changes in the DnD layout, sync the main tree
  disconnect( mAvailableWidgetsView->selectionModel(), &QItemSelectionModel::selectionChanged, this, &QgsAttributesFormProperties::onAttributeSelectionChanged );
  QModelIndex index;
  if ( deselected.indexes().count() == 1 )
  {
    index = mFormLayoutProxyModel->mapToSource( deselected.indexes().at( 0 ) );
  }
  else if ( deselected.indexes().count() == 0 && mFormLayoutView->selectionModel()->selectedIndexes().count() == 2 )
  {
    // There was 1 selected, it was not deselected, but instead a new item was added to selection
    index = mFormLayoutView->firstSelectedIndex();
  }

  loadAttributeSpecificEditor( mFormLayoutView, mAvailableWidgetsView, index );
  connect( mAvailableWidgetsView->selectionModel(), &QItemSelectionModel::selectionChanged, this, &QgsAttributesFormProperties::onAttributeSelectionChanged );
}

void QgsAttributesFormProperties::loadAttributeSpecificEditor( QgsAttributesFormBaseView *emitter, QgsAttributesFormBaseView *receiver, QModelIndex &deselectedFormLayoutIndex )
{
  const Qgis::AttributeFormLayout layout = mEditorLayoutComboBox->currentData().value<Qgis::AttributeFormLayout>();

  if ( layout == Qgis::AttributeFormLayout::DragAndDrop )
  {
    storeAttributeWidgetEdit( deselectedFormLayoutIndex );
    storeAttributeContainerEdit( deselectedFormLayoutIndex );
  }
  if ( mAttributeTypeDialog )
  {
    storeAttributeTypeDialog();
  }

  clearAttributeTypeFrame();

  if ( emitter->selectionModel()->selectedRows( 0 ).count() != 1 )
  {
    receiver->clearSelection();
  }
  else
  {
    const QModelIndex index = emitter->firstSelectedIndex();
    const auto indexType = static_cast< QgsAttributesFormData::AttributesFormItemType >( index.data( QgsAttributesFormModel::ItemTypeRole ).toInt() );
    switch ( indexType )
    {
      case QgsAttributesFormData::Relation:
      {
        receiver->selectFirstMatchingItem( QgsAttributesFormData::Relation, index.data( QgsAttributesFormModel::ItemIdRole ).toString() );
        if ( layout == Qgis::AttributeFormLayout::DragAndDrop )
        {
          loadAttributeWidgetEdit();
        }
        else
        {
          loadInfoWidget( tr( "This configuration is available in the Drag and Drop Designer" ) );
        }
        break;
      }
      case QgsAttributesFormData::Field:
      {
        receiver->selectFirstMatchingItem( QgsAttributesFormData::Field, index.data( QgsAttributesFormModel::ItemNameRole ).toString() );
        if ( layout == Qgis::AttributeFormLayout::DragAndDrop )
        {
          loadAttributeWidgetEdit();
        }
        loadAttributeTypeDialog();
        break;
      }
      case QgsAttributesFormData::Container:
      {
        receiver->clearSelection();
        loadAttributeContainerEdit();
        break;
      }
      case QgsAttributesFormData::Action:
      {
        receiver->selectFirstMatchingItem( QgsAttributesFormData::Action, index.data( QgsAttributesFormModel::ItemIdRole ).toString() );
        const QgsAction action { mLayer->actions()->action( index.data( QgsAttributesFormModel::ItemIdRole ).toString() ) };
        loadInfoWidget( action.html() );
        break;
      }
      case QgsAttributesFormData::QmlWidget:
      case QgsAttributesFormData::HtmlWidget:
      case QgsAttributesFormData::TextWidget:
      case QgsAttributesFormData::SpacerWidget:
      {
        if ( layout != Qgis::AttributeFormLayout::DragAndDrop )
        {
          loadInfoWidget( tr( "This configuration is available with double-click in the Drag and Drop Designer" ) );
        }
        else
        {
          loadInfoWidget( tr( "This configuration is available with double-click in the Form Layout panel" ) );
        }
        receiver->clearSelection();
        break;
      }
      case QgsAttributesFormData::WidgetType:
      {
        receiver->clearSelection();
        break;
      }
    }
  }
}

void QgsAttributesFormProperties::clearAttributeTypeFrame()
{
  if ( mAttributeWidgetEdit )
  {
    mAttributeTypeFrame->layout()->removeWidget( mAttributeWidgetEdit );
    mAttributeWidgetEdit->deleteLater();
    mAttributeWidgetEdit = nullptr;
  }
  if ( mAttributeTypeDialog )
  {
    mAttributeTypeFrame->layout()->removeWidget( mAttributeTypeDialog );
    mAttributeTypeDialog->deleteLater();
    mAttributeTypeDialog = nullptr;
  }
  if ( mAttributeContainerEdit )
  {
    mAttributeTypeFrame->layout()->removeWidget( mAttributeContainerEdit );
    mAttributeContainerEdit->deleteLater();
    mAttributeContainerEdit = nullptr;
  }
  if ( mInfoTextWidget )
  {
    mAttributeTypeFrame->layout()->removeWidget( mInfoTextWidget );
    mInfoTextWidget->deleteLater();
    mInfoTextWidget = nullptr;
  }
}

void QgsAttributesFormProperties::onInvertSelectionButtonClicked( bool checked )
{
  Q_UNUSED( checked )
  for ( int i = 0; i < mFormLayoutProxyModel->rowCount(); ++i )
  {
    QModelIndex index = mFormLayoutProxyModel->index( i, 0 );
    mFormLayoutView->selectionModel()->select( index, QItemSelectionModel::Toggle );
  }
}

void QgsAttributesFormProperties::toggleShowAliases( bool checked )
{
  settingShowAliases->setValue( checked );
  mAvailableWidgetsModel->setShowAliases( checked );
  mFormLayoutModel->setShowAliases( checked );
}

void QgsAttributesFormProperties::addContainer()
{
  QList<QgsAddAttributeFormContainerDialog::ContainerPair> existingContainerList = mFormLayoutModel->listOfContainers();

  QModelIndex currentItem;
  if ( mFormLayoutView->selectionModel()->selectedRows().count() > 0 )
    currentItem = mFormLayoutView->firstSelectedIndex();

  QgsAddAttributeFormContainerDialog dialog( mLayer, existingContainerList, currentItem, this );

  if ( !dialog.exec() )
    return;

  const QString name = dialog.name();
  QModelIndex parentContainerItem = dialog.parentContainerItem();

  mFormLayoutModel->addContainer( parentContainerItem, name, dialog.columnCount(), dialog.containerType() );
  if ( parentContainerItem.isValid() )
    mFormLayoutView->setExpanded( parentContainerItem, true );
}

void QgsAttributesFormProperties::removeTabOrGroupButton()
{
  // deleting an item may delete any number of nested child items -- so we delete
  // them one at a time and then see if there's any selection left
  while ( true )
  {
    const QModelIndexList items = mFormLayoutView->selectionModel()->selectedRows();
    if ( items.empty() )
      break;

    const QModelIndex item = mFormLayoutProxyModel->mapToSource( items.at( 0 ) );
    mFormLayoutModel->removeRow( item.row(), item.parent() );
  }
}

void QgsAttributesFormProperties::mEditorLayoutComboBox_currentIndexChanged( int )
{
  // Refresh the right panel. Save selection to recover it later.
  const QItemSelection selection = mAvailableWidgetsView->selectionModel()->selection();
  if ( selection.count() > 0 )
  {
    mAvailableWidgetsView->selectionModel()->clear();
  }

  if ( mFormLayoutView->selectionModel()->selectedRows().count() > 0 )
  {
    mFormLayoutView->selectionModel()->clear(); // Get rid of e.g., container selection
  }

  const Qgis::AttributeFormLayout layout = mEditorLayoutComboBox->currentData().value<Qgis::AttributeFormLayout>();
  switch ( layout )
  {
    case Qgis::AttributeFormLayout::AutoGenerated:
      mFormLayoutWidget->setVisible( false );
      mTreeViewHorizontalSpacer->changeSize( 0, 20, QSizePolicy::Fixed, QSizePolicy::Fixed );
      mUiFileFrame->setVisible( false );
      mAddContainerButton->setVisible( false );
      mRemoveLayoutItemButton->setVisible( false );
      mInvertSelectionButton->setVisible( false );
      break;

    case Qgis::AttributeFormLayout::DragAndDrop:
      mFormLayoutWidget->setVisible( true );
      mTreeViewHorizontalSpacer->changeSize( 6, 20, QSizePolicy::Fixed, QSizePolicy::Fixed );
      mUiFileFrame->setVisible( false );
      mAddContainerButton->setVisible( true );
      mRemoveLayoutItemButton->setVisible( true );
      mInvertSelectionButton->setVisible( true );
      break;

    case Qgis::AttributeFormLayout::UiFile:
      // ui file
      mFormLayoutWidget->setVisible( false );
      mTreeViewHorizontalSpacer->changeSize( 0, 20, QSizePolicy::Fixed, QSizePolicy::Fixed );
      mUiFileFrame->setVisible( true );
      mAddContainerButton->setVisible( false );
      mRemoveLayoutItemButton->setVisible( false );
      mInvertSelectionButton->setVisible( false );
      break;
  }

  // Get the selection back so that we refresh the right panel
  if ( selection.count() > 0 )
  {
    mAvailableWidgetsView->selectionModel()->select( selection, QItemSelectionModel::Select );
  }
}

void QgsAttributesFormProperties::mTbInitCode_clicked()
{
  QgsAttributesFormInitCode attributesFormInitCode;

  attributesFormInitCode.setCodeSource( mInitCodeSource );
  attributesFormInitCode.setInitCode( mInitCode );
  attributesFormInitCode.setInitFilePath( mInitFilePath );
  attributesFormInitCode.setInitFunction( mInitFunction );

  if ( !attributesFormInitCode.exec() )
    return;

  mInitCodeSource = attributesFormInitCode.codeSource();
  mInitCode = attributesFormInitCode.initCode();
  mInitFilePath = attributesFormInitCode.initFilePath();
  mInitFunction = attributesFormInitCode.initFunction();
}

void QgsAttributesFormProperties::pbnSelectEditForm_clicked()
{
  QgsSettings myQSettings;
  const QString lastUsedDir = myQSettings.value( QStringLiteral( "style/lastUIDir" ), QDir::homePath() ).toString();
  const QString uifilename = QFileDialog::getOpenFileName( this, tr( "Select edit form" ), lastUsedDir, tr( "UI file" ) + " (*.ui)" );

  if ( uifilename.isNull() )
    return;

  const QFileInfo fi( uifilename );
  myQSettings.setValue( QStringLiteral( "style/lastUIDir" ), fi.path() );
  mEditFormLineEdit->setText( uifilename );
}

void QgsAttributesFormProperties::store()
{
  storeAttributeWidgetEdit();
  storeAttributeContainerEdit();
  storeAttributeTypeDialog();
}

void QgsAttributesFormProperties::apply()
{
  mBlockUpdates++;
  store();

  QgsEditFormConfig editFormConfig = mLayer->editFormConfig();

  const QModelIndex fieldContainer = mAvailableWidgetsModel->fieldContainer();
  QModelIndex index;

  for ( int i = 0; i < mAvailableWidgetsModel->rowCount( fieldContainer ); i++ )
  {
    index = mAvailableWidgetsModel->index( i, 0, fieldContainer );
    const QgsAttributesFormData::FieldConfig cfg = index.data( QgsAttributesFormModel::ItemFieldConfigRole ).value<QgsAttributesFormData::FieldConfig>();

    const QString fieldName = index.data( QgsAttributesFormModel::ItemNameRole ).toString();
    const int idx = mLayer->fields().indexOf( fieldName );

    //continue in case field does not exist anymore
    if ( idx < 0 )
      continue;

    editFormConfig.setReadOnly( idx, !cfg.mEditable );
    editFormConfig.setLabelOnTop( idx, cfg.mLabelOnTop );
    editFormConfig.setReuseLastValue( idx, cfg.mReuseLastValues );

    if ( cfg.mDataDefinedProperties.count() > 0 )
    {
      editFormConfig.setDataDefinedFieldProperties( fieldName, cfg.mDataDefinedProperties );
    }

    mLayer->setEditorWidgetSetup( idx, QgsEditorWidgetSetup( cfg.mEditorWidgetType, cfg.mEditorWidgetConfig ) );

    const QgsFieldConstraints constraints = cfg.mFieldConstraints;
    mLayer->setConstraintExpression( idx, constraints.constraintExpression(), constraints.constraintDescription() );
    if ( constraints.constraints() & QgsFieldConstraints::ConstraintNotNull )
    {
      mLayer->setFieldConstraint( idx, QgsFieldConstraints::ConstraintNotNull, constraints.constraintStrength( QgsFieldConstraints::ConstraintNotNull ) );
    }
    else
    {
      mLayer->removeFieldConstraint( idx, QgsFieldConstraints::ConstraintNotNull );
    }
    if ( constraints.constraints() & QgsFieldConstraints::ConstraintUnique )
    {
      mLayer->setFieldConstraint( idx, QgsFieldConstraints::ConstraintUnique, constraints.constraintStrength( QgsFieldConstraints::ConstraintUnique ) );
    }
    else
    {
      mLayer->removeFieldConstraint( idx, QgsFieldConstraints::ConstraintUnique );
    }
    if ( constraints.constraints() & QgsFieldConstraints::ConstraintExpression )
    {
      mLayer->setFieldConstraint( idx, QgsFieldConstraints::ConstraintExpression, constraints.constraintStrength( QgsFieldConstraints::ConstraintExpression ) );
    }
    else
    {
      mLayer->removeFieldConstraint( idx, QgsFieldConstraints::ConstraintExpression );
    }

    mLayer->setFieldAlias( idx, cfg.mAlias );
    mLayer->setFieldSplitPolicy( idx, cfg.mSplitPolicy );
    mLayer->setFieldDuplicatePolicy( idx, cfg.mDuplicatePolicy );
    mLayer->setFieldMergePolicy( idx, cfg.mMergePolicy );
  }

  // // tabs and groups
  editFormConfig.clearTabs();

  for ( int t = 0; t < mFormLayoutModel->rowCount(); t++ )
  {
    QModelIndex index = mFormLayoutModel->index( t, 0 );
    QgsAttributeEditorElement *editorElement { mFormLayoutModel->createAttributeEditorWidget( index, nullptr ) };
    if ( editorElement )
      editFormConfig.addTab( editorElement );
  }

  editFormConfig.setUiForm( mEditFormLineEdit->text() );

  editFormConfig.setLayout( mEditorLayoutComboBox->currentData().value<Qgis::AttributeFormLayout>() );

  editFormConfig.setInitCodeSource( mInitCodeSource );
  editFormConfig.setInitFunction( mInitFunction );
  editFormConfig.setInitFilePath( mInitFilePath );
  editFormConfig.setInitCode( mInitCode );

  editFormConfig.setSuppress( mFormSuppressCmbBx->currentData().value<Qgis::AttributeFormSuppression>() );

  // write the legacy config of relation widgets to support settings read by the API
  const QModelIndex relationContainer = mAvailableWidgetsModel->relationContainer();

  for ( int i = 0; i < mAvailableWidgetsModel->rowCount( relationContainer ); i++ )
  {
    const QModelIndex relationIndex = mAvailableWidgetsModel->index( i, 0, relationContainer );

    const QgsAttributesFormData::AttributeFormItemData itemData = relationIndex.data( QgsAttributesFormModel::ItemDataRole ).value<QgsAttributesFormData::AttributeFormItemData>();
    const auto indexType = static_cast< QgsAttributesFormData::AttributesFormItemType >( relationIndex.data( QgsAttributesFormModel::ItemTypeRole ).toInt() );
    const QString indexId = relationIndex.data( QgsAttributesFormModel::ItemIdRole ).toString();

    const QModelIndex layoutIndex = mFormLayoutModel->firstRecursiveMatchingModelIndex( indexType, indexId );
    if ( layoutIndex.isValid() )
    {
      QVariantMap config;

      const QgsAttributesFormData::AttributeFormItemData tabIndexData = layoutIndex.data( QgsAttributesFormModel::ItemDataRole ).value<QgsAttributesFormData::AttributeFormItemData>();
      config[QStringLiteral( "nm-rel" )] = tabIndexData.relationEditorConfiguration().nmRelationId;
      config[QStringLiteral( "force-suppress-popup" )] = tabIndexData.relationEditorConfiguration().forceSuppressFormPopup;

      editFormConfig.setWidgetConfig( indexId, config );
      break;
    }
  }

  mLayer->setEditFormConfig( editFormConfig );
  mBlockUpdates--;
}


QgsAttributesFormBaseView::QgsAttributesFormBaseView( QgsVectorLayer *layer, QWidget *parent )
  : QTreeView( parent )
  , mLayer( layer )
{
}

QModelIndex QgsAttributesFormBaseView::firstSelectedIndex() const
{
  if ( selectionModel()->selectedRows( 0 ).count() == 0 )
    return QModelIndex();

  return mModel->mapToSource( selectionModel()->selectedRows( 0 ).at( 0 ) );
}

QgsExpressionContext QgsAttributesFormBaseView::createExpressionContext() const
{
  QgsExpressionContext expContext;
  expContext << QgsExpressionContextUtils::globalScope()
             << QgsExpressionContextUtils::projectScope( QgsProject::instance() );

  if ( mLayer )
    expContext << QgsExpressionContextUtils::layerScope( mLayer );

  expContext.appendScope( QgsExpressionContextUtils::formScope() );
  return expContext;
}

void QgsAttributesFormBaseView::selectFirstMatchingItem( const QgsAttributesFormData::AttributesFormItemType &itemType, const QString &itemId )
{
  // To be used with Relations, fields and actions
  const auto *model = static_cast< QgsAttributesFormModel * >( mModel->sourceModel() );
  QModelIndex index = mModel->mapFromSource( model->firstRecursiveMatchingModelIndex( itemType, itemId ) );

  if ( index.isValid() )
  {
    // TODO: compare with eventual single selected index, if they match, avoid calling next line
    selectionModel()->setCurrentIndex( index, QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows );
  }
  else
  {
    selectionModel()->clearSelection();
  }
}

void QgsAttributesFormBaseView::setFilterText( const QString &text )
{
  mModel->setFilterText( text );
}


QgsAttributesAvailableWidgetsView::QgsAttributesAvailableWidgetsView( QgsVectorLayer *layer, QWidget *parent )
  : QgsAttributesFormBaseView( layer, parent )
{
}

void QgsAttributesAvailableWidgetsView::setModel( QAbstractItemModel *model )
{
  mModel = qobject_cast<QgsAttributesFormProxyModel *>( model );
  if ( !mModel )
    return;

  QTreeView::setModel( mModel );
}

QgsAttributesAvailableWidgetsModel *QgsAttributesAvailableWidgetsView::availableWidgetsModel() const
{
  return static_cast< QgsAttributesAvailableWidgetsModel * >( mModel->sourceModel() );
}


QgsAttributesFormLayoutView::QgsAttributesFormLayoutView( QgsVectorLayer *layer, QWidget *parent )
  : QgsAttributesFormBaseView( layer, parent )
{
  connect( this, &QTreeView::doubleClicked, this, &QgsAttributesFormLayoutView::onItemDoubleClicked );
}

void QgsAttributesFormLayoutView::setModel( QAbstractItemModel *model )
{
  mModel = qobject_cast<QgsAttributesFormProxyModel *>( model );
  if ( !mModel )
    return;

  QTreeView::setModel( mModel );

  const auto *formLayoutModel = static_cast< QgsAttributesFormLayoutModel * >( mModel->sourceModel() );
  connect( formLayoutModel, &QgsAttributesFormLayoutModel::externalItemDropped, this, &QgsAttributesFormLayoutView::handleExternalDroppedItem );
  connect( formLayoutModel, &QgsAttributesFormLayoutModel::internalItemDropped, this, &QgsAttributesFormLayoutView::handleInternalDroppedItem );
}


void QgsAttributesFormLayoutView::handleExternalDroppedItem( QModelIndex &index )
{
  selectionModel()->setCurrentIndex( mModel->mapFromSource( index ), QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows );

  const auto itemType = static_cast< QgsAttributesFormData::AttributesFormItemType >( index.data( QgsAttributesFormModel::ItemTypeRole ).toInt() );

  if ( itemType == QgsAttributesFormData::QmlWidget
       || itemType == QgsAttributesFormData::HtmlWidget
       || itemType == QgsAttributesFormData::TextWidget
       || itemType == QgsAttributesFormData::SpacerWidget )
  {
    onItemDoubleClicked( mModel->mapFromSource( index ) );
  }
}

void QgsAttributesFormLayoutView::handleInternalDroppedItem( QModelIndex &index )
{
  selectionModel()->clearCurrentIndex();
  const auto itemType = static_cast< QgsAttributesFormData::AttributesFormItemType >( index.data( QgsAttributesFormModel::ItemTypeRole ).toInt() );
  if ( itemType == QgsAttributesFormData::Container )
  {
    expandRecursively( mModel->mapFromSource( index ) );
  }
}

void QgsAttributesFormLayoutView::dragEnterEvent( QDragEnterEvent *event )
{
  const QMimeData *data = event->mimeData();

  if ( data->hasFormat( QStringLiteral( "application/x-qgsattributesformavailablewidgetsrelement" ) )
       || data->hasFormat( QStringLiteral( "application/x-qgsattributesformlayoutelement" ) ) )
  {
    // Inner drag and drop actions are always MoveAction
    if ( event->source() == this )
    {
      event->setDropAction( Qt::MoveAction );
    }
  }
  else
  {
    event->ignore();
  }

  QTreeView::dragEnterEvent( event );
}

/**
 * Is called when mouse is moved over attributes tree before a
 * drop event.
 */
void QgsAttributesFormLayoutView::dragMoveEvent( QDragMoveEvent *event )
{
  const QMimeData *data = event->mimeData();

  if ( data->hasFormat( QStringLiteral( "application/x-qgsattributesformavailablewidgetsrelement" ) )
       || data->hasFormat( QStringLiteral( "application/x-qgsattributesformlayoutelement" ) ) )
  {
    // Inner drag and drop actions are always MoveAction
    if ( event->source() == this )
    {
      event->setDropAction( Qt::MoveAction );
    }
  }
  else
  {
    event->ignore();
  }

  QTreeView::dragMoveEvent( event );
}

void QgsAttributesFormLayoutView::dropEvent( QDropEvent *event )
{
  if ( !( event->mimeData()->hasFormat( QStringLiteral( "application/x-qgsattributesformavailablewidgetsrelement" ) )
          || event->mimeData()->hasFormat( QStringLiteral( "application/x-qgsattributesformlayoutelement" ) ) ) )
    return;

  if ( event->source() == this )
  {
    event->setDropAction( Qt::MoveAction );
  }

  QTreeView::dropEvent( event );
}

void QgsAttributesFormLayoutView::onItemDoubleClicked( const QModelIndex &index )
{
  QModelIndex sourceIndex = mModel->mapToSource( index );
  QgsAttributesFormData::AttributeFormItemData itemData = sourceIndex.data( QgsAttributesFormModel::ItemDataRole ).value<QgsAttributesFormData::AttributeFormItemData>();
  const auto itemType = static_cast<QgsAttributesFormData::AttributesFormItemType>( sourceIndex.data( QgsAttributesFormModel::ItemTypeRole ).toInt() );
  const QString itemName = sourceIndex.data( QgsAttributesFormModel::ItemNameRole ).toString();

  QGroupBox *baseData = new QGroupBox( tr( "Base configuration" ) );

  QFormLayout *baseLayout = new QFormLayout();
  baseData->setLayout( baseLayout );
  QCheckBox *showLabelCheckbox = new QCheckBox( QStringLiteral( "Show label" ) );
  showLabelCheckbox->setChecked( itemData.showLabel() );
  baseLayout->addRow( showLabelCheckbox );
  QWidget *baseWidget = new QWidget();
  baseWidget->setLayout( baseLayout );

  switch ( itemType )
  {
    case QgsAttributesFormData::Action:
    case QgsAttributesFormData::Container:
    case QgsAttributesFormData::WidgetType:
    case QgsAttributesFormData::Relation:
    case QgsAttributesFormData::Field:
      break;

    case QgsAttributesFormData::QmlWidget:
    {
      QDialog dlg;
      dlg.setObjectName( "QML Form Configuration Widget" );
      QgsGui::enableAutoGeometryRestore( &dlg );
      dlg.setWindowTitle( tr( "Configure QML Widget" ) );

      QVBoxLayout *mainLayout = new QVBoxLayout( &dlg );
      QSplitter *qmlSplitter = new QSplitter();
      QWidget *qmlConfigWiget = new QWidget();
      QVBoxLayout *layout = new QVBoxLayout( qmlConfigWiget );
      layout->setContentsMargins( 0, 0, 0, 0 );
      mainLayout->addWidget( qmlSplitter );
      qmlSplitter->addWidget( qmlConfigWiget );
      layout->addWidget( baseWidget );

      QLineEdit *title = new QLineEdit( itemName );

      //qmlCode
      QgsCodeEditor *qmlCode = new QgsCodeEditor( this );
      qmlCode->setEditingTimeoutInterval( 250 );
      qmlCode->setText( itemData.qmlElementEditorConfiguration().qmlCode );

      QgsQmlWidgetWrapper *qmlWrapper = new QgsQmlWidgetWrapper( mLayer, nullptr, this );
      QgsFeature previewFeature;
      mLayer->getFeatures().nextFeature( previewFeature );

      //update preview on text change
      connect( qmlCode, &QgsCodeEditor::editingTimeout, this, [=] {
        qmlWrapper->setQmlCode( qmlCode->text() );
        qmlWrapper->reinitWidget();
        qmlWrapper->setFeature( previewFeature );
      } );

      //templates
      QComboBox *qmlObjectTemplate = new QComboBox();
      qmlObjectTemplate->addItem( tr( "Free Text…" ) );
      qmlObjectTemplate->addItem( tr( "Rectangle" ) );
      qmlObjectTemplate->addItem( tr( "Pie Chart" ) );
      qmlObjectTemplate->addItem( tr( "Bar Chart" ) );
      connect( qmlObjectTemplate, qOverload<int>( &QComboBox::activated ), qmlCode, [=]( int index ) {
        qmlCode->clear();
        switch ( index )
        {
          case 0:
          {
            qmlCode->setText( QString() );
            break;
          }
          case 1:
          {
            qmlCode->setText( QStringLiteral( "import QtQuick 2.0\n"
                                              "\n"
                                              "Rectangle {\n"
                                              "    width: 100\n"
                                              "    height: 100\n"
                                              "    color: \"steelblue\"\n"
                                              "    Text{ text: \"A rectangle\" }\n"
                                              "}\n" ) );
            break;
          }
          case 2:
          {
            qmlCode->setText( QStringLiteral( "import QtQuick 2.0\n"
                                              "import QtCharts 2.0\n"
                                              "\n"
                                              "ChartView {\n"
                                              "    width: 400\n"
                                              "    height: 400\n"
                                              "\n"
                                              "    PieSeries {\n"
                                              "        id: pieSeries\n"
                                              "        PieSlice { label: \"First slice\"; value: 25 }\n"
                                              "        PieSlice { label: \"Second slice\"; value: 45 }\n"
                                              "        PieSlice { label: \"Third slice\"; value: 30 }\n"
                                              "    }\n"
                                              "}\n" ) );
            break;
          }
          case 3:
          {
            qmlCode->setText( QStringLiteral( "import QtQuick 2.0\n"
                                              "import QtCharts 2.0\n"
                                              "\n"
                                              "ChartView {\n"
                                              "    title: \"Bar series\"\n"
                                              "    width: 600\n"
                                              "    height:400\n"
                                              "    legend.alignment: Qt.AlignBottom\n"
                                              "    antialiasing: true\n"
                                              "    ValueAxis{\n"
                                              "        id: valueAxisY\n"
                                              "        min: 0\n"
                                              "        max: 15\n"
                                              "    }\n"
                                              "\n"
                                              "    BarSeries {\n"
                                              "        id: mySeries\n"
                                              "        axisY: valueAxisY\n"
                                              "        axisX: BarCategoryAxis { categories: [\"2007\", \"2008\", \"2009\", \"2010\", \"2011\", \"2012\" ] }\n"
                                              "        BarSet { label: \"Bob\"; values: [2, 2, 3, 4, 5, 6] }\n"
                                              "        BarSet { label: \"Susan\"; values: [5, 1, 2, 4, 1, 7] }\n"
                                              "        BarSet { label: \"James\"; values: [3, 5, 8, 13, 5, 8] }\n"
                                              "    }\n"
                                              "}\n" ) );
            break;
          }
          default:
            break;
        }
      } );

      QgsFieldExpressionWidget *expressionWidget = new QgsFieldExpressionWidget;
      expressionWidget->setButtonVisible( false );
      expressionWidget->registerExpressionContextGenerator( this );
      expressionWidget->setLayer( mLayer );
      QToolButton *addFieldButton = new QToolButton();
      addFieldButton->setIcon( QgsApplication::getThemeIcon( QStringLiteral( "/symbologyAdd.svg" ) ) );

      QToolButton *editExpressionButton = new QToolButton();
      editExpressionButton->setIcon( QgsApplication::getThemeIcon( QStringLiteral( "/mIconExpression.svg" ) ) );
      editExpressionButton->setToolTip( tr( "Insert/Edit Expression" ) );

      connect( addFieldButton, &QAbstractButton::clicked, this, [=] {
        QString expression = expressionWidget->expression().trimmed().replace( '"', QLatin1String( "\\\"" ) );
        if ( !expression.isEmpty() )
          qmlCode->insertText( QStringLiteral( "expression.evaluate(\"%1\")" ).arg( expression ) );
      } );

      connect( editExpressionButton, &QAbstractButton::clicked, this, [=] {
        QString expression = QgsExpressionFinder::findAndSelectActiveExpression( qmlCode, QStringLiteral( "expression\\.evaluate\\(\\s*\"(.*?)\\s*\"\\s*\\)" ) );
        expression.replace( QLatin1String( "\\\"" ), QLatin1String( "\"" ) );
        QgsExpressionContext context = createExpressionContext();
        QgsExpressionBuilderDialog exprDlg( mLayer, expression, this, QStringLiteral( "generic" ), context );

        exprDlg.setWindowTitle( tr( "Insert Expression" ) );
        if ( exprDlg.exec() == QDialog::Accepted && !exprDlg.expressionText().trimmed().isEmpty() )
        {
          QString expression = exprDlg.expressionText().trimmed().replace( '"', QLatin1String( "\\\"" ) );
          if ( !expression.isEmpty() )
            qmlCode->insertText( QStringLiteral( "expression.evaluate(\"%1\")" ).arg( expression ) );
        }
      } );

      layout->addWidget( new QLabel( tr( "Title" ) ) );
      layout->addWidget( title );
      QGroupBox *qmlCodeBox = new QGroupBox( tr( "QML Code" ) );
      qmlCodeBox->setLayout( new QVBoxLayout );
      qmlCodeBox->layout()->addWidget( qmlObjectTemplate );
      QWidget *expressionWidgetBox = new QWidget();
      qmlCodeBox->layout()->addWidget( expressionWidgetBox );
      expressionWidgetBox->setLayout( new QHBoxLayout );
      expressionWidgetBox->layout()->setContentsMargins( 0, 0, 0, 0 );
      expressionWidgetBox->layout()->addWidget( expressionWidget );
      expressionWidgetBox->layout()->addWidget( addFieldButton );
      expressionWidgetBox->layout()->addWidget( editExpressionButton );
      expressionWidgetBox->layout()->addWidget( editExpressionButton );
      layout->addWidget( qmlCodeBox );
      layout->addWidget( qmlCode );
      QScrollArea *qmlPreviewBox = new QgsScrollArea();
      qmlPreviewBox->setMinimumWidth( 200 );
      qmlPreviewBox->setWidget( qmlWrapper->widget() );
      //emit to load preview for the first time
      emit qmlCode->editingTimeout();
      qmlSplitter->addWidget( qmlPreviewBox );
      qmlSplitter->setChildrenCollapsible( false );
      qmlSplitter->setHandleWidth( 6 );
      qmlSplitter->setSizes( QList<int>() << 1 << 1 );

      QDialogButtonBox *buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel | QDialogButtonBox::Help );

      connect( buttonBox, &QDialogButtonBox::accepted, &dlg, &QDialog::accept );
      connect( buttonBox, &QDialogButtonBox::rejected, &dlg, &QDialog::reject );
      connect( buttonBox, &QDialogButtonBox::helpRequested, &dlg, [=] {
        QgsHelp::openHelp( QStringLiteral( "working_with_vector/vector_properties.html#other-widgets" ) );
      } );

      mainLayout->addWidget( buttonBox );

      if ( dlg.exec() )
      {
        QgsAttributesFormData::QmlElementEditorConfiguration qmlEdCfg;
        qmlEdCfg.qmlCode = qmlCode->text();
        itemData.setQmlElementEditorConfiguration( qmlEdCfg );
        itemData.setShowLabel( showLabelCheckbox->isChecked() );

        mModel->sourceModel()->setData( sourceIndex, itemData, QgsAttributesFormModel::ItemDataRole );
        mModel->sourceModel()->setData( sourceIndex, title->text(), QgsAttributesFormModel::ItemNameRole );
      }
    }
    break;

    case QgsAttributesFormData::HtmlWidget:
    {
      QDialog dlg;
      dlg.setObjectName( "HTML Form Configuration Widget" );
      QgsGui::enableAutoGeometryRestore( &dlg );
      dlg.setWindowTitle( tr( "Configure HTML Widget" ) );

      QVBoxLayout *mainLayout = new QVBoxLayout( &dlg );
      QSplitter *htmlSplitter = new QSplitter();
      QWidget *htmlConfigWiget = new QWidget();
      QVBoxLayout *layout = new QVBoxLayout( htmlConfigWiget );
      layout->setContentsMargins( 0, 0, 0, 0 );
      mainLayout->addWidget( htmlSplitter );
      htmlSplitter->addWidget( htmlConfigWiget );
      htmlSplitter->setChildrenCollapsible( false );
      htmlSplitter->setHandleWidth( 6 );
      htmlSplitter->setSizes( QList<int>() << 1 << 1 );
      layout->addWidget( baseWidget );

      QLineEdit *title = new QLineEdit( itemName );

      //htmlCode
      QgsCodeEditorHTML *htmlCode = new QgsCodeEditorHTML();
      htmlCode->setSizePolicy( QSizePolicy::Policy::Expanding, QSizePolicy::Policy::Expanding );
      htmlCode->setText( itemData.htmlElementEditorConfiguration().htmlCode );

      QgsHtmlWidgetWrapper *htmlWrapper = new QgsHtmlWidgetWrapper( mLayer, nullptr, this );
      QgsFeature previewFeature;
      mLayer->getFeatures().nextFeature( previewFeature );

      //update preview on text change
      connect( htmlCode, &QgsCodeEditorHTML::textChanged, this, [=] {
        htmlWrapper->setHtmlCode( htmlCode->text() );
        htmlWrapper->reinitWidget();
        htmlWrapper->setFeature( previewFeature );
      } );

      QgsFieldExpressionWidget *expressionWidget = new QgsFieldExpressionWidget;
      expressionWidget->setButtonVisible( false );
      expressionWidget->registerExpressionContextGenerator( this );
      expressionWidget->setLayer( mLayer );
      QToolButton *addFieldButton = new QToolButton();
      addFieldButton->setIcon( QgsApplication::getThemeIcon( QStringLiteral( "/symbologyAdd.svg" ) ) );

      QToolButton *editExpressionButton = new QToolButton();
      editExpressionButton->setIcon( QgsApplication::getThemeIcon( QStringLiteral( "/mIconExpression.svg" ) ) );
      editExpressionButton->setToolTip( tr( "Insert/Edit Expression" ) );

      connect( addFieldButton, &QAbstractButton::clicked, this, [=] {
        QString expression = expressionWidget->expression().trimmed().replace( '"', QLatin1String( "\\\"" ) );
        if ( !expression.isEmpty() )
          htmlCode->insertText( QStringLiteral( "<script>document.write(expression.evaluate(\"%1\"));</script>" ).arg( expression ) );
      } );

      connect( editExpressionButton, &QAbstractButton::clicked, this, [=] {
        QString expression = QgsExpressionFinder::findAndSelectActiveExpression( htmlCode, QStringLiteral( "<script>\\s*document\\.write\\(\\s*expression\\.evaluate\\(\\s*\"(.*?)\\s*\"\\s*\\)\\s*\\)\\s*;?\\s*</script>" ) );
        expression.replace( QLatin1String( "\\\"" ), QLatin1String( "\"" ) );
        QgsExpressionContext context = createExpressionContext();
        QgsExpressionBuilderDialog exprDlg( mLayer, expression, this, QStringLiteral( "generic" ), context );

        exprDlg.setWindowTitle( tr( "Insert Expression" ) );
        if ( exprDlg.exec() == QDialog::Accepted && !exprDlg.expressionText().trimmed().isEmpty() )
        {
          QString expression = exprDlg.expressionText().trimmed().replace( '"', QLatin1String( "\\\"" ) );
          if ( !expression.isEmpty() )
            htmlCode->insertText( QStringLiteral( "<script>document.write(expression.evaluate(\"%1\"));</script>" ).arg( expression ) );
        }
      } );

      layout->addWidget( new QLabel( tr( "Title" ) ) );
      layout->addWidget( title );
      QGroupBox *expressionWidgetBox = new QGroupBox( tr( "HTML Code" ) );
      layout->addWidget( expressionWidgetBox );
      expressionWidgetBox->setLayout( new QHBoxLayout );
      expressionWidgetBox->layout()->addWidget( expressionWidget );
      expressionWidgetBox->layout()->addWidget( addFieldButton );
      expressionWidgetBox->layout()->addWidget( editExpressionButton );
      layout->addWidget( htmlCode );
      QScrollArea *htmlPreviewBox = new QgsScrollArea();
      htmlPreviewBox->setLayout( new QGridLayout );
      htmlPreviewBox->setMinimumWidth( 200 );
      htmlPreviewBox->layout()->addWidget( htmlWrapper->widget() );
      //emit to load preview for the first time
      emit htmlCode->textChanged();
      htmlSplitter->addWidget( htmlPreviewBox );
      htmlSplitter->setChildrenCollapsible( false );
      htmlSplitter->setHandleWidth( 6 );
      htmlSplitter->setSizes( QList<int>() << 1 << 1 );

      QDialogButtonBox *buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel | QDialogButtonBox::Help );

      connect( buttonBox, &QDialogButtonBox::accepted, &dlg, &QDialog::accept );
      connect( buttonBox, &QDialogButtonBox::rejected, &dlg, &QDialog::reject );
      connect( buttonBox, &QDialogButtonBox::helpRequested, &dlg, [=] {
        QgsHelp::openHelp( QStringLiteral( "working_with_vector/vector_properties.html#other-widgets" ) );
      } );

      mainLayout->addWidget( buttonBox );

      if ( dlg.exec() )
      {
        QgsAttributesFormData::HtmlElementEditorConfiguration htmlEdCfg;
        htmlEdCfg.htmlCode = htmlCode->text();
        itemData.setHtmlElementEditorConfiguration( htmlEdCfg );
        itemData.setShowLabel( showLabelCheckbox->isChecked() );

        mModel->sourceModel()->setData( sourceIndex, itemData, QgsAttributesFormModel::ItemDataRole );
        mModel->sourceModel()->setData( sourceIndex, title->text(), QgsAttributesFormModel::ItemNameRole );
      }
      break;
    }

    case QgsAttributesFormData::TextWidget:
    {
      QDialog dlg;
      dlg.setObjectName( "Text Form Configuration Widget" );
      QgsGui::enableAutoGeometryRestore( &dlg );
      dlg.setWindowTitle( tr( "Configure Text Widget" ) );

      QVBoxLayout *mainLayout = new QVBoxLayout( &dlg );
      QSplitter *textSplitter = new QSplitter();
      QWidget *textConfigWiget = new QWidget();
      QVBoxLayout *layout = new QVBoxLayout( textConfigWiget );
      layout->setContentsMargins( 0, 0, 0, 0 );
      mainLayout->addWidget( textSplitter );
      textSplitter->addWidget( textConfigWiget );
      layout->addWidget( baseWidget );

      QLineEdit *title = new QLineEdit( itemName );

      QgsCodeEditorHTML *text = new QgsCodeEditorHTML();
      text->setSizePolicy( QSizePolicy::Policy::Expanding, QSizePolicy::Policy::Expanding );
      text->setText( itemData.textElementEditorConfiguration().text );

      QgsTextWidgetWrapper *textWrapper = new QgsTextWidgetWrapper( mLayer, nullptr, this );
      QgsFeature previewFeature;
      mLayer->getFeatures().nextFeature( previewFeature );

      //update preview on text change
      connect( text, &QgsCodeEditorExpression::textChanged, this, [=] {
        textWrapper->setText( text->text() );
        textWrapper->reinitWidget();
        textWrapper->setFeature( previewFeature );
      } );

      QgsFieldExpressionWidget *expressionWidget = new QgsFieldExpressionWidget;
      expressionWidget->setButtonVisible( false );
      expressionWidget->registerExpressionContextGenerator( this );
      expressionWidget->setLayer( mLayer );
      QToolButton *addFieldButton = new QToolButton();
      addFieldButton->setIcon( QgsApplication::getThemeIcon( QStringLiteral( "/symbologyAdd.svg" ) ) );

      QToolButton *editExpressionButton = new QToolButton();
      editExpressionButton->setIcon( QgsApplication::getThemeIcon( QStringLiteral( "/mIconExpression.svg" ) ) );
      editExpressionButton->setToolTip( tr( "Insert/Edit Expression" ) );

      connect( addFieldButton, &QAbstractButton::clicked, this, [=] {
        QString expression = expressionWidget->expression().trimmed();
        if ( !expression.isEmpty() )
          text->insertText( QStringLiteral( "[%%1%]" ).arg( expression ) );
      } );
      connect( editExpressionButton, &QAbstractButton::clicked, this, [=] {
        QString expression = QgsExpressionFinder::findAndSelectActiveExpression( text );

        QgsExpressionContext context = createExpressionContext();
        QgsExpressionBuilderDialog exprDlg( mLayer, expression, this, QStringLiteral( "generic" ), context );

        exprDlg.setWindowTitle( tr( "Insert Expression" ) );
        if ( exprDlg.exec() == QDialog::Accepted && !exprDlg.expressionText().trimmed().isEmpty() )
        {
          QString expression = exprDlg.expressionText().trimmed();
          if ( !expression.isEmpty() )
            text->insertText( QStringLiteral( "[%%1%]" ).arg( expression ) );
        }
      } );

      layout->addWidget( new QLabel( tr( "Title" ) ) );
      layout->addWidget( title );
      QGroupBox *expressionWidgetBox = new QGroupBox( tr( "Text" ) );
      layout->addWidget( expressionWidgetBox );
      expressionWidgetBox->setLayout( new QHBoxLayout );
      expressionWidgetBox->layout()->addWidget( expressionWidget );
      expressionWidgetBox->layout()->addWidget( addFieldButton );
      expressionWidgetBox->layout()->addWidget( editExpressionButton );
      layout->addWidget( text );
      QScrollArea *textPreviewBox = new QgsScrollArea();
      textPreviewBox->setLayout( new QGridLayout );
      textPreviewBox->setMinimumWidth( 200 );
      textPreviewBox->layout()->addWidget( textWrapper->widget() );
      //emit to load preview for the first time
      emit text->textChanged();
      textSplitter->addWidget( textPreviewBox );
      textSplitter->setChildrenCollapsible( false );
      textSplitter->setHandleWidth( 6 );
      textSplitter->setSizes( QList<int>() << 1 << 1 );

      QDialogButtonBox *buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel | QDialogButtonBox::Help );

      connect( buttonBox, &QDialogButtonBox::accepted, &dlg, &QDialog::accept );
      connect( buttonBox, &QDialogButtonBox::rejected, &dlg, &QDialog::reject );
      connect( buttonBox, &QDialogButtonBox::helpRequested, &dlg, [=] {
        QgsHelp::openHelp( QStringLiteral( "working_with_vector/vector_properties.html#other-widgets" ) );
      } );

      mainLayout->addWidget( buttonBox );

      if ( dlg.exec() )
      {
        QgsAttributesFormData::TextElementEditorConfiguration textEdCfg;
        textEdCfg.text = text->text();
        itemData.setTextElementEditorConfiguration( textEdCfg );
        itemData.setShowLabel( showLabelCheckbox->isChecked() );

        mModel->sourceModel()->setData( sourceIndex, itemData, QgsAttributesFormModel::ItemDataRole );
        mModel->sourceModel()->setData( sourceIndex, title->text(), QgsAttributesFormModel::ItemNameRole );
      }
      break;
    }

    case QgsAttributesFormData::SpacerWidget:
    {
      QDialog dlg;
      dlg.setObjectName( "Spacer Form Configuration Widget" );
      QgsGui::enableAutoGeometryRestore( &dlg );
      dlg.setWindowTitle( tr( "Configure Spacer Widget" ) );

      QVBoxLayout *mainLayout = new QVBoxLayout();
      mainLayout->addWidget( new QLabel( tr( "Title" ) ) );
      QLineEdit *title = new QLineEdit( itemName );
      mainLayout->addWidget( title );

      QHBoxLayout *cbLayout = new QHBoxLayout();
      mainLayout->addLayout( cbLayout );
      dlg.setLayout( mainLayout );
      QCheckBox *cb = new QCheckBox { &dlg };
      cb->setChecked( itemData.spacerElementEditorConfiguration().drawLine );
      cbLayout->addWidget( new QLabel( tr( "Draw horizontal line" ), &dlg ) );
      cbLayout->addWidget( cb );

      QDialogButtonBox *buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel | QDialogButtonBox::Help );

      connect( buttonBox, &QDialogButtonBox::accepted, &dlg, &QDialog::accept );
      connect( buttonBox, &QDialogButtonBox::rejected, &dlg, &QDialog::reject );
      connect( buttonBox, &QDialogButtonBox::helpRequested, &dlg, [=] {
        QgsHelp::openHelp( QStringLiteral( "working_with_vector/vector_properties.html#other-widgets" ) );
      } );

      mainLayout->addWidget( buttonBox );

      if ( dlg.exec() )
      {
        QgsAttributesFormData::SpacerElementEditorConfiguration spacerEdCfg;
        spacerEdCfg.drawLine = cb->isChecked();
        itemData.setSpacerElementEditorConfiguration( spacerEdCfg );
        itemData.setShowLabel( false );

        mModel->sourceModel()->setData( sourceIndex, itemData, QgsAttributesFormModel::ItemDataRole );
        mModel->sourceModel()->setData( sourceIndex, title->text(), QgsAttributesFormModel::ItemNameRole );
      }

      break;
    }
  }
}


void QgsAttributesFormProperties::updatedFields()
{
  // Store configuration to insure changes made are kept after refreshing the list
  QMap<QString, QgsAttributesFormData::FieldConfig> fieldConfigs;

  const QModelIndex fieldContainerBefore = mAvailableWidgetsModel->fieldContainer();
  QModelIndex index;

  for ( int i = 0; i < mAvailableWidgetsModel->rowCount( fieldContainerBefore ); i++ )
  {
    index = mAvailableWidgetsModel->index( i, 0, fieldContainerBefore );
    const QString fieldName = index.data( QgsAttributesFormModel::ItemNameRole ).toString();
    const QgsAttributesFormData::FieldConfig config = index.data( QgsAttributesFormModel::ItemFieldConfigRole ).value< QgsAttributesFormData::FieldConfig >();
    fieldConfigs[fieldName] = config;
  }

  initAvailableWidgetsView();

  const QModelIndex fieldContainerAfter = mAvailableWidgetsModel->fieldContainer();

  for ( int i = 0; i < mAvailableWidgetsModel->rowCount( fieldContainerAfter ); i++ )
  {
    index = mAvailableWidgetsModel->index( i, 0, fieldContainerAfter );
    const QString fieldName = index.data( QgsAttributesFormModel::ItemNameRole ).toString();

    if ( fieldConfigs.contains( fieldName ) )
    {
      mAvailableWidgetsModel->setData( index, fieldConfigs[fieldName], QgsAttributesFormModel::ItemFieldConfigRole );
      mAvailableWidgetsModel->setData( index, fieldConfigs[fieldName].mAlias, QgsAttributesFormModel::ItemDisplayRole );
    }
  }
}

void QgsAttributesFormProperties::updateFilteredItems( const QString &filterText )
{
  const int availableWidgetsPreviousSelectionCount = mAvailableWidgetsView->selectionModel()->selectedRows().count();
  const int formLayoutPreviousSelectionCount = mFormLayoutView->selectionModel()->selectedRows().count();

  static_cast< QgsAttributesAvailableWidgetsView *>( mAvailableWidgetsView )->setFilterText( filterText );
  mAvailableWidgetsView->expandAll();

  static_cast< QgsAttributesFormLayoutView *>( mFormLayoutView )->setFilterText( filterText );
  mFormLayoutView->expandAll();

  // If there was no previous selection leave as is, since
  // after a filter change no new selection may be added (only lost).
  if ( !( availableWidgetsPreviousSelectionCount == 0 && formLayoutPreviousSelectionCount == 0 ) )
  {
    const int selectedAvailableWidgetItemCount = mAvailableWidgetsView->selectionModel()->selectedRows().count();
    const int selectedFormLayoutItemCount = mFormLayoutView->selectionModel()->selectedRows().count();

    if ( selectedAvailableWidgetItemCount == 0 && selectedFormLayoutItemCount == 0 )
    {
      // Clear right-hand side panel since all selected items have been filtered out
      clearAttributeTypeFrame();
    }
  }
}

void QgsAttributesFormProperties::onContextMenuRequested( QPoint point )
{
  if ( mAvailableWidgetsView->selectionModel()->selectedRows().count() != 1 )
    return;

  QPoint globalPos = mAvailableWidgetsView->viewport()->mapToGlobal( point );

  const QModelIndex index = mAvailableWidgetsView->indexAt( point );
  const auto itemType = static_cast< QgsAttributesFormData::AttributesFormItemType >( index.data( QgsAttributesFormModel::ItemTypeRole ).toInt() );
  if ( itemType == QgsAttributesFormData::Field )
  {
    const QClipboard *clipboard = QApplication::clipboard();
    const QMimeData *mimeData = clipboard->mimeData();
    if ( !mimeData )
      return;

    const bool pasteEnabled = mimeData->hasFormat( QStringLiteral( "application/x-qgsattributetabledesignerelementclipboard" ) );
    mActionPasteWidgetConfiguration->setEnabled( pasteEnabled );
    mAvailableWidgetsContextMenu->popup( globalPos );
  }
}

void QgsAttributesFormProperties::copyWidgetConfiguration()
{
  if ( mAvailableWidgetsView->selectionModel()->selectedRows().count() != 1 )
    return;

  const QModelIndex index = mAvailableWidgetsView->firstSelectedIndex();
  const auto itemType = static_cast< QgsAttributesFormData::AttributesFormItemType >( index.data( QgsAttributesFormModel::ItemTypeRole ).toInt() );

  if ( itemType != QgsAttributesFormData::Field )
    return;

  const QString fieldName = index.data( QgsAttributesFormModel::ItemNameRole ).toString();
  const int fieldIndex = mLayer->fields().indexOf( fieldName );

  if ( fieldIndex < 0 )
    return;

  const QgsField field = mLayer->fields().field( fieldIndex );

  // We'll copy everything but field aliases or comments
  QDomDocument doc;
  QDomElement documentElement = doc.createElement( QStringLiteral( "FormWidgetClipboard" ) );
  documentElement.setAttribute( QStringLiteral( "name" ), field.name() );

  // Editor widget setup
  QgsEditorWidgetSetup widgetSetup = field.editorWidgetSetup();

  QDomElement editWidgetElement = doc.createElement( QStringLiteral( "editWidget" ) );
  documentElement.appendChild( editWidgetElement );
  editWidgetElement.setAttribute( QStringLiteral( "type" ), widgetSetup.type() );
  QDomElement editWidgetConfigElement = doc.createElement( QStringLiteral( "config" ) );

  editWidgetConfigElement.appendChild( QgsXmlUtils::writeVariant( widgetSetup.config(), doc ) );
  editWidgetElement.appendChild( editWidgetConfigElement );

  // Split policy
  QDomElement splitPolicyElement = doc.createElement( QStringLiteral( "splitPolicy" ) );
  splitPolicyElement.setAttribute( QStringLiteral( "policy" ), qgsEnumValueToKey( field.splitPolicy() ) );
  documentElement.appendChild( splitPolicyElement );

  // Duplicate policy
  QDomElement duplicatePolicyElement = doc.createElement( QStringLiteral( "duplicatePolicy" ) );
  duplicatePolicyElement.setAttribute( QStringLiteral( "policy" ), qgsEnumValueToKey( field.duplicatePolicy() ) );
  documentElement.appendChild( duplicatePolicyElement );

  // Merge policy
  QDomElement mergePolicyElement = doc.createElement( QStringLiteral( "mergePolicy" ) );
  mergePolicyElement.setAttribute( QStringLiteral( "policy" ), qgsEnumValueToKey( field.mergePolicy() ) );
  documentElement.appendChild( mergePolicyElement );

  // Default expressions
  QDomElement defaultElem = doc.createElement( QStringLiteral( "default" ) );
  defaultElem.setAttribute( QStringLiteral( "expression" ), field.defaultValueDefinition().expression() );
  defaultElem.setAttribute( QStringLiteral( "applyOnUpdate" ), field.defaultValueDefinition().applyOnUpdate() ? QStringLiteral( "1" ) : QStringLiteral( "0" ) );
  documentElement.appendChild( defaultElem );

  // Constraints
  QDomElement constraintElem = doc.createElement( QStringLiteral( "constraint" ) );
  constraintElem.setAttribute( QStringLiteral( "constraints" ), field.constraints().constraints() );
  constraintElem.setAttribute( QStringLiteral( "unique_strength" ), field.constraints().constraintStrength( QgsFieldConstraints::ConstraintUnique ) );
  constraintElem.setAttribute( QStringLiteral( "notnull_strength" ), field.constraints().constraintStrength( QgsFieldConstraints::ConstraintNotNull ) );
  constraintElem.setAttribute( QStringLiteral( "exp_strength" ), field.constraints().constraintStrength( QgsFieldConstraints::ConstraintExpression ) );
  documentElement.appendChild( constraintElem );

  // Constraint expressions
  QDomElement constraintExpressionElem = doc.createElement( QStringLiteral( "constraintExpression" ) );
  constraintExpressionElem.setAttribute( QStringLiteral( "exp" ), field.constraints().constraintExpression() );
  constraintExpressionElem.setAttribute( QStringLiteral( "desc" ), field.constraints().constraintDescription() );
  documentElement.appendChild( constraintExpressionElem );

  // Widget general settings
  if ( mAttributeTypeDialog )
  {
    QDomElement widgetGeneralSettingsElem = doc.createElement( QStringLiteral( "widgetGeneralSettings" ) );
    widgetGeneralSettingsElem.setAttribute( QStringLiteral( "editable" ), mAttributeTypeDialog->fieldEditable() );
    widgetGeneralSettingsElem.setAttribute( QStringLiteral( "reuse_last_values" ), mAttributeTypeDialog->labelOnTop() );
    widgetGeneralSettingsElem.setAttribute( QStringLiteral( "label_on_top" ), mAttributeTypeDialog->reuseLastValues() );
    documentElement.appendChild( widgetGeneralSettingsElem );
  }

  // Widget display section
  if ( mAttributeWidgetEdit )
  {
    // Go for the corresponding form layout item and extract its display settings
    if ( mFormLayoutView->selectionModel()->selectedRows().count() != 1 )
      return;

    const QModelIndex indexLayout = mFormLayoutView->firstSelectedIndex();
    const auto layoutData = indexLayout.data( QgsAttributesFormModel::ItemDataRole ).value< QgsAttributesFormData::AttributeFormItemData >();

    QDomElement displayElement = doc.createElement( QStringLiteral( "widgetDisplay" ) );
    displayElement.setAttribute( QStringLiteral( "showLabel" ), layoutData.showLabel() );
    displayElement.setAttribute( QStringLiteral( "horizontalStretch" ), layoutData.horizontalStretch() );
    displayElement.setAttribute( QStringLiteral( "verticalStretch" ), layoutData.verticalStretch() );
    displayElement.appendChild( layoutData.labelStyle().writeXml( doc ) );
    documentElement.appendChild( displayElement );
  }

  doc.appendChild( documentElement );

  QMimeData *mimeData = new QMimeData;
  mimeData->setData( QStringLiteral( "application/x-qgsattributetabledesignerelementclipboard" ), doc.toByteArray() );
  QClipboard *clipboard = QApplication::clipboard();
  clipboard->setMimeData( mimeData );
}

void QgsAttributesFormProperties::pasteWidgetConfiguration()
{
  if ( mAvailableWidgetsView->selectionModel()->selectedRows().count() != 1 )
    return;

  QModelIndex index = mAvailableWidgetsView->firstSelectedIndex();

  const QString fieldName = index.data( QgsAttributesFormModel::ItemNameRole ).toString();
  const int fieldIndex = mLayer->fields().indexOf( fieldName );

  if ( fieldIndex < 0 )
    return;

  // Get base config from target item and ovewrite settings when possible
  auto config = index.data( QgsAttributesFormModel::ItemFieldConfigRole ).value< QgsAttributesFormData::FieldConfig >();

  QDomDocument doc;
  QClipboard *clipboard = QApplication::clipboard();
  const QMimeData *mimeData = clipboard->mimeData();
  if ( !mimeData )
    return;

  if ( doc.setContent( mimeData->data( QStringLiteral( "application/x-qgsattributetabledesignerelementclipboard" ) ) ) )
  {
    QDomElement docElem = doc.documentElement();
    if ( docElem.tagName() != QLatin1String( "FormWidgetClipboard" ) )
      return;

    // When pasting, the target item has already been selected and
    // has triggered attribute type dialog loading. Therefore, we'll
    // only overwrite GUI settings instead of destroying and recreating
    // the whole dialog.

    // Editor widget configuration
    const QDomElement fieldWidgetElement = docElem.firstChildElement( QStringLiteral( "editWidget" ) );
    if ( !fieldWidgetElement.isNull() )
    {
      const QString widgetType = fieldWidgetElement.attribute( QStringLiteral( "type" ) );

      // Only paste if source editor widget type is supported by target field
      const QgsEditorWidgetFactory *factory = QgsGui::editorWidgetRegistry()->factory( widgetType );
      if ( factory->supportsField( mLayer, fieldIndex ) )
      {
        const QDomElement configElement = fieldWidgetElement.firstChildElement( QStringLiteral( "config" ) );
        if ( !configElement.isNull() )
        {
          const QDomElement optionsElem = configElement.childNodes().at( 0 ).toElement();
          QVariantMap optionsMap = QgsXmlUtils::readVariant( optionsElem ).toMap();
          QgsReadWriteContext context;
          // translate widget configuration strings
          if ( widgetType == QStringLiteral( "ValueRelation" ) )
          {
            optionsMap[QStringLiteral( "Value" )] = context.projectTranslator()->translate( QStringLiteral( "project:layers:%1:fields:%2:valuerelationvalue" ).arg( mLayer->id(), fieldName ), optionsMap[QStringLiteral( "Value" )].toString() );
          }
          if ( widgetType == QStringLiteral( "ValueMap" ) )
          {
            if ( optionsMap[QStringLiteral( "map" )].canConvert<QList<QVariant>>() )
            {
              QList<QVariant> translatedValueList;
              const QList<QVariant> valueList = optionsMap[QStringLiteral( "map" )].toList();
              for ( int i = 0, row = 0; i < valueList.count(); i++, row++ )
              {
                QMap<QString, QVariant> translatedValueMap;
                QString translatedKey = context.projectTranslator()->translate( QStringLiteral( "project:layers:%1:fields:%2:valuemapdescriptions" ).arg( mLayer->id(), fieldName ), valueList[i].toMap().constBegin().key() );
                translatedValueMap.insert( translatedKey, valueList[i].toMap().constBegin().value() );
                translatedValueList.append( translatedValueMap );
              }
              optionsMap.insert( QStringLiteral( "map" ), translatedValueList );
            }
          }
          config.mEditorWidgetType = widgetType;
          config.mEditorWidgetConfig = optionsMap;
        }
      }
      else
      {
        mMessageBar->pushMessage( QString(), tr( "Unable to paste widget configuration. The target field (%1) does not support the %2 widget type." ).arg( fieldName, widgetType ), Qgis::MessageLevel::Warning );
      }
    }

    // Split policy
    const QDomElement splitPolicyElement = docElem.firstChildElement( QStringLiteral( "splitPolicy" ) );
    if ( !splitPolicyElement.isNull() )
    {
      const Qgis::FieldDomainSplitPolicy policy = qgsEnumKeyToValue( splitPolicyElement.attribute( QStringLiteral( "policy" ) ), Qgis::FieldDomainSplitPolicy::Duplicate );
      config.mSplitPolicy = policy;
    }

    // Duplicate policy
    const QDomElement duplicatePolicyElement = docElem.firstChildElement( QStringLiteral( "duplicatePolicy" ) );
    if ( !duplicatePolicyElement.isNull() )
    {
      const Qgis::FieldDuplicatePolicy policy = qgsEnumKeyToValue( duplicatePolicyElement.attribute( QStringLiteral( "policy" ) ), Qgis::FieldDuplicatePolicy::Duplicate );
      config.mDuplicatePolicy = policy;
    }

    // Merge policy
    const QDomElement mergePolicyElement = docElem.firstChildElement( QStringLiteral( "mergePolicy" ) );
    if ( !mergePolicyElement.isNull() )
    {
      const Qgis::FieldDomainMergePolicy policy = qgsEnumKeyToValue( mergePolicyElement.attribute( QStringLiteral( "policy" ) ), Qgis::FieldDomainMergePolicy::DefaultValue );
      config.mMergePolicy = policy;
    }

    // Default expressions
    const QDomElement defaultElement = docElem.firstChildElement( QStringLiteral( "default" ) );
    if ( !defaultElement.isNull() )
    {
      mAttributeTypeDialog->setDefaultValueExpression( defaultElement.attribute( QStringLiteral( "expression" ) ) );
      mAttributeTypeDialog->setApplyDefaultValueOnUpdate( defaultElement.attribute( QStringLiteral( "applyOnUpdate" ) ).toInt() );
    }

    // Constraints
    // take target field constraints as a basis
    QgsFieldConstraints fieldConstraints = config.mFieldConstraints;
    const QDomElement constraintElement = docElem.firstChildElement( QStringLiteral( "constraint" ) );
    if ( !constraintElement.isNull() )
    {
      const int intConstraints = constraintElement.attribute( QStringLiteral( "constraints" ), QStringLiteral( "0" ) ).toInt();
      QgsFieldConstraints::Constraints constraints = static_cast< QgsFieldConstraints::Constraints >( intConstraints );

      // always keep provider constraints intact
      if ( fieldConstraints.constraintOrigin( QgsFieldConstraints::ConstraintNotNull ) != QgsFieldConstraints::ConstraintOriginProvider )
      {
        if ( constraints & QgsFieldConstraints::ConstraintNotNull )
          fieldConstraints.setConstraint( QgsFieldConstraints::ConstraintNotNull, QgsFieldConstraints::ConstraintOriginLayer );
        else
          fieldConstraints.removeConstraint( QgsFieldConstraints::ConstraintNotNull );
      }
      if ( fieldConstraints.constraintOrigin( QgsFieldConstraints::ConstraintUnique ) != QgsFieldConstraints::ConstraintOriginProvider )
      {
        if ( constraints & QgsFieldConstraints::ConstraintUnique )
          fieldConstraints.setConstraint( QgsFieldConstraints::ConstraintUnique, QgsFieldConstraints::ConstraintOriginLayer );
        else
          fieldConstraints.removeConstraint( QgsFieldConstraints::ConstraintUnique );
      }
      if ( fieldConstraints.constraintOrigin( QgsFieldConstraints::ConstraintExpression ) != QgsFieldConstraints::ConstraintOriginProvider )
      {
        if ( constraints & QgsFieldConstraints::ConstraintExpression )
          fieldConstraints.setConstraint( QgsFieldConstraints::ConstraintExpression, QgsFieldConstraints::ConstraintOriginLayer );
        else
          fieldConstraints.removeConstraint( QgsFieldConstraints::ConstraintExpression );
      }

      const int uniqueStrength = constraintElement.attribute( QStringLiteral( "unique_strength" ), QStringLiteral( "1" ) ).toInt();
      const int notNullStrength = constraintElement.attribute( QStringLiteral( "notnull_strength" ), QStringLiteral( "1" ) ).toInt();
      const int expStrength = constraintElement.attribute( QStringLiteral( "exp_strength" ), QStringLiteral( "1" ) ).toInt();

      fieldConstraints.setConstraintStrength( QgsFieldConstraints::ConstraintUnique, static_cast< QgsFieldConstraints::ConstraintStrength >( uniqueStrength ) );
      fieldConstraints.setConstraintStrength( QgsFieldConstraints::ConstraintNotNull, static_cast< QgsFieldConstraints::ConstraintStrength >( notNullStrength ) );
      fieldConstraints.setConstraintStrength( QgsFieldConstraints::ConstraintExpression, static_cast< QgsFieldConstraints::ConstraintStrength >( expStrength ) );
    }

    // Constraint expressions
    // always keep provider constraints intact
    if ( fieldConstraints.constraintOrigin( QgsFieldConstraints::ConstraintExpression ) != QgsFieldConstraints::ConstraintOriginProvider )
    {
      const QDomElement constraintExpressionElement = docElem.firstChildElement( QStringLiteral( "constraintExpression" ) );
      if ( !constraintExpressionElement.isNull() )
      {
        QString expression = constraintExpressionElement.attribute( QStringLiteral( "exp" ), QString() );
        QString description = constraintExpressionElement.attribute( QStringLiteral( "desc" ), QString() );
        fieldConstraints.setConstraintExpression( expression, description );
      }
    }
    config.mFieldConstraints = fieldConstraints;

    const QDomElement widgetGeneralSettingsElement = docElem.firstChildElement( QStringLiteral( "widgetGeneralSettings" ) );
    if ( !widgetGeneralSettingsElement.isNull() )
    {
      const int editable = widgetGeneralSettingsElement.attribute( QStringLiteral( "editable" ), QStringLiteral( "0" ) ).toInt();
      const int reuse = widgetGeneralSettingsElement.attribute( QStringLiteral( "reuse_last_values" ), QStringLiteral( "0" ) ).toInt();
      const int labelOnTop = widgetGeneralSettingsElement.attribute( QStringLiteral( "label_on_top" ), QStringLiteral( "0" ) ).toInt();

      config.mEditable = editable;
      config.mReuseLastValues = reuse;
      config.mLabelOnTop = labelOnTop;
    }

    loadAttributeTypeDialogFromConfiguration( config );

    // Widget display section
    if ( mAttributeWidgetEdit )
    {
      const QDomElement displayElement = docElem.firstChildElement( QStringLiteral( "widgetDisplay" ) );
      if ( !displayElement.isNull() )
      {
        const int showLabel = displayElement.attribute( QStringLiteral( "showLabel" ), QStringLiteral( "0" ) ).toInt();
        const int horizontalStretch = displayElement.attribute( QStringLiteral( "horizontalStretch" ), QStringLiteral( "0" ) ).toInt();
        const int verticalStretch = displayElement.attribute( QStringLiteral( "verticalStretch" ), QStringLiteral( "0" ) ).toInt();
        QgsAttributeEditorElement::LabelStyle style;
        style.readXml( displayElement );

        // Update current GUI controls
        mAttributeWidgetEdit->setShowLabel( showLabel );
        mAttributeWidgetEdit->setHorizontalStretch( horizontalStretch );
        mAttributeWidgetEdit->setVerticalStretch( verticalStretch );
        mAttributeWidgetEdit->setLabelStyle( style );
      }
    }
  }
}
