// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F131

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_192_0_F_T_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x7efdaa5bu, 0xf27facbau, 0, 126 }, // 7efdaa5bf27facba = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__16_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0xa11c604du, 0x906f6e5fu, 0, 6 }, // a11c604d906f6e5f = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x553db607u, 0x44e0b047u, 0, 69 }, // 553db60744e0b047 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__16_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xed40fe2du, 0x69a4206du, 0, 29 }, // ed40fe2d69a4206d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xb0a147c0u, 0xfbe59f45u, 23, 109 }, // b0a147c0fbe59f45 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xe31ef55bu, 0x28502c28u, 23, 126 }, // e31ef55b28502c28 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__32_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x8e7aa831u, 0x16047080u, 23, 6 }, // 8e7aa83116047080 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x5d74a35cu, 0x28235a32u, 23, 69 }, // 5d74a35c28235a32 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xce0099bau, 0xd9277dbfu, 63, 109 }, // ce0099bad9277dbf = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x48a99557u, 0xc450de1au, 103, 109 }, // 48a99557c450de1a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__64_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x31c37698u, 0x087018beu, 103, 69 }, // 31c37698087018be = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_192_0_F_T_1___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{ 7, 4, 9, 8, 5, 5, 5},
 { 4, 7, 1, 8, 1, 9, 9},
 { 4, 3, 2, 1, 8, 8, 8},
 { 8, 1, 1, 3, 1, 8, 8},
 { 7, 1, 6, 8, 8, 8, 8},
 { 6,10, 6, 8, 8, 6, 8},
 { 6, 6, 6, 6, 8, 8, 8}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F131 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 192
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 192 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

