use crate::sidebar::SidebarSelection;

use super::{GFeedListItem, GFeedListItemID, GTag, GTagID};
use glib::{Enum, Object, Properties, prelude::*, subclass::prelude::*};
use std::cell::{Cell, RefCell};

#[derive(Debug, Default, Eq, PartialEq, Clone, Copy, Enum)]
#[repr(u32)]
#[enum_type(name = "GSidebarSelectionType")]
pub enum SidebarSelectionType {
    #[default]
    None,
    All,
    Today,
    FeedList,
    TagList,
}

mod imp {
    use super::*;

    #[derive(Default, Properties)]
    #[properties(wrapper_type = super::GSidebarSelection)]
    pub struct GSidebarSelection {
        #[property(get, set, builder(SidebarSelectionType::All))]
        pub selection_type: Cell<SidebarSelectionType>,

        #[property(get, set, name = "feedlist-id")]
        pub feedlist_id: RefCell<GFeedListItemID>,

        #[property(get, set, name = "tag-id")]
        pub tag_id: RefCell<GTagID>,

        #[property(get, set, nullable)]
        pub label: RefCell<Option<String>>,

        #[property(get, set)]
        pub position: Cell<u32>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for GSidebarSelection {
        const NAME: &'static str = "GSidebarSelection";
        type Type = super::GSidebarSelection;
    }

    #[glib::derived_properties]
    impl ObjectImpl for GSidebarSelection {}

    impl GSidebarSelection {}
}

glib::wrapper! {
    pub struct GSidebarSelection(ObjectSubclass<imp::GSidebarSelection>);
}

impl Default for GSidebarSelection {
    fn default() -> Self {
        Object::new()
    }
}

impl From<SidebarSelection> for GSidebarSelection {
    fn from(value: SidebarSelection) -> Self {
        let obj = Self::default();
        let imp = obj.imp();

        match value {
            SidebarSelection::All => imp.selection_type.set(SidebarSelectionType::All),
            SidebarSelection::Today => imp.selection_type.set(SidebarSelectionType::Today),
            SidebarSelection::FeedList(id, label, pos) => {
                imp.selection_type.set(SidebarSelectionType::FeedList);
                imp.feedlist_id.replace(id);
                imp.label.replace(Some(label));
                imp.position.set(pos);
            }
            SidebarSelection::Tag(id, label, pos) => {
                imp.selection_type.set(SidebarSelectionType::TagList);
                imp.tag_id.replace(id);
                imp.label.replace(Some(label));
                imp.position.set(pos);
            }
        }

        obj
    }
}

impl GSidebarSelection {
    pub fn all() -> Self {
        let obj = Self::default();
        obj.imp().selection_type.set(SidebarSelectionType::All);
        obj
    }

    pub fn today() -> Self {
        let obj = Self::default();
        obj.imp().selection_type.set(SidebarSelectionType::Today);
        obj
    }

    pub fn from_feed_list(item: GFeedListItem, pos: u32) -> Self {
        let obj = Self::default();
        let imp = obj.imp();
        imp.selection_type.set(SidebarSelectionType::FeedList);
        imp.feedlist_id.replace(item.item_id());
        imp.label.replace(Some(item.label()));
        imp.position.set(pos);
        obj
    }

    pub fn from_tag_list(tag: GTag, pos: u32) -> Self {
        let obj = Self::default();
        let imp = obj.imp();
        imp.selection_type.set(SidebarSelectionType::TagList);
        imp.tag_id.replace(tag.tag_id());
        imp.label.replace(Some(tag.label()));
        imp.position.set(pos);
        obj
    }

    pub fn eq(&self, other: &GSidebarSelection) -> bool {
        match self.selection_type() {
            SidebarSelectionType::None => matches!(other.selection_type(), SidebarSelectionType::None),
            SidebarSelectionType::Today => matches!(other.selection_type(), SidebarSelectionType::Today),
            SidebarSelectionType::All => matches!(other.selection_type(), SidebarSelectionType::All),
            SidebarSelectionType::FeedList => match other.selection_type() {
                SidebarSelectionType::FeedList => self.feedlist_id() == other.feedlist_id(),
                _ => false,
            },
            SidebarSelectionType::TagList => match other.selection_type() {
                SidebarSelectionType::TagList => self.tag_id() == other.tag_id(),
                _ => false,
            },
        }
    }
}
