mod search_item_row;

use self::search_item_row::SearchItemRow;
use crate::add_dialog::{AddFeedWidget, SelectFeedWidget};
use crate::app::App;
use crate::content_page::ContentPage;
use crate::gobject_models::GFeed;
use crate::i18n::i18n;
use crate::infrastructure::TokioRuntime;
use feedly_api::{ApiError, FeedlyApi, models::SearchResult};
use glib::{Object, Properties, clone, prelude::*, subclass::*};
use gtk4::{
    Button, CompositeTemplate, ListBox, ListBoxRow, Stack, StringList, Widget, prelude::*, subclass::prelude::*,
};
use libadwaita::{Dialog, NavigationView, prelude::*, subclass::prelude::*};
use news_flash::error::NewsFlashError;
use news_flash::feed_api::FeedApiError;
use news_flash::models::{FeedID, Url};
use news_flash::{NewsFlash, ParsedUrl};
use std::cell::{Cell, RefCell};
use std::sync::Arc;
use tokio::sync::Semaphore;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::DiscoverDialog)]
    #[template(file = "data/resources/ui_templates/discover/dialog.blp")]
    pub struct DiscoverDialog {
        #[template_child]
        pub navigation_view: TemplateChild<NavigationView>,
        #[template_child]
        pub add_feed_widget: TemplateChild<AddFeedWidget>,
        #[template_child]
        pub select_feed_widget: TemplateChild<SelectFeedWidget>,
        #[template_child]
        pub language_dropdown_model: TemplateChild<StringList>,
        #[template_child]
        pub search_result_stack: TemplateChild<Stack>,
        #[template_child]
        pub search_result_list: TemplateChild<ListBox>,

        #[property(get, set, name = "search-term")]
        pub search_term: RefCell<String>,

        #[property(get, set, name = "selected-language-index")]
        pub selected_language_index: Cell<u32>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for DiscoverDialog {
        const NAME: &'static str = "DiscoverDialog";
        type ParentType = Dialog;
        type Type = super::DiscoverDialog;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for DiscoverDialog {
        fn constructed(&self) {
            let obj = self.obj();

            obj.connect_search_term_notify(|dialog| {
                dialog.imp().feedly_search();
            });

            obj.connect_selected_language_index_notify(|dialog| {
                dialog.imp().feedly_search();
            });
        }
    }

    impl WidgetImpl for DiscoverDialog {}

    impl AdwDialogImpl for DiscoverDialog {}

    #[gtk4::template_callbacks]
    impl DiscoverDialog {
        #[template_callback]
        fn on_feed_selected(&self, feed: GFeed) {
            self.add_feed_widget.reset();
            self.add_feed_widget.set_feed(feed);
            self.navigation_view.push_by_tag("feed_add_page");
        }

        #[template_callback]
        fn on_feed_added(&self) {
            self.obj().close();
        }

        #[template_callback]
        fn visible_child_name(&self, search_term: String) -> &'static str {
            if search_term.is_empty() { "featured" } else { "search" }
        }

        #[template_callback]
        fn on_card_clicked(&self, button: &Button) {
            let search_term = button.label().unwrap_or_default().as_str().to_lowercase();
            self.obj().set_search_term(format!("#{search_term}"));
        }

        #[template_callback]
        fn on_row_activated(&self, row: &ListBoxRow) {
            if let Some(serach_row) = row.downcast_ref::<SearchItemRow>() {
                self.parse_feed(&serach_row.feed_url());
            }
        }

        fn feedly_search(&self) {
            let locale = self
                .language_dropdown_model
                .string(self.selected_language_index.get())
                .map(|id| id.as_str().to_owned());

            let query = self.search_term.borrow().clone();
            let query_clone = query.clone();
            self.search_result_stack.set_visible_child_name("spinner");
            self.search_result_list.remove_all();

            TokioRuntime::execute_with_callback(
                move || async move {
                    FeedlyApi::search_feedly_cloud(&App::client(), &query, Some(30), locale.as_deref()).await
                },
                clone!(
                    #[weak(rename_to = imp)]
                    self,
                    move |search_result: Result<SearchResult, ApiError>| {
                        if query_clone != *imp.search_term.borrow() {
                            return;
                        }

                        match search_result {
                            Ok(search_result) => {
                                imp.search_result_list.remove_all();

                                let result_count = search_result.results.len();
                                for search_item in search_result.results.into_iter() {
                                    if search_item.title.is_none() {
                                        // dont show items without title
                                        continue;
                                    }
                                    imp.search_result_list.insert(&SearchItemRow::from(search_item), -1);
                                }

                                let child_name = if result_count > 0 { "list" } else { "empty" };
                                imp.search_result_stack.set_visible_child_name(child_name);
                            }
                            Err(e) => {
                                imp.search_result_stack.set_visible_child_name("empty");
                                log::error!("Feedly search query failed: '{e}'");
                            }
                        }
                    }
                ),
            );
        }

        fn parse_feed(&self, url: &str) {
            self.navigation_view.replace_with_tags(&["spinner"]);
            let feed_id = FeedID::new(url);
            let url = Url::parse(url).unwrap();
            let url_clone = url.clone();
            log::info!("url {url}");

            TokioRuntime::execute_with_callback(
                || async move {
                    let news_flash = App::news_flash();
                    let semaphore = news_flash
                        .read()
                        .await
                        .as_ref()
                        .map(NewsFlash::get_semaphore)
                        .unwrap_or(Arc::new(Semaphore::new(1)));
                    news_flash::feed_parser::download_and_parse_feed(
                        &url_clone,
                        &feed_id,
                        None,
                        semaphore,
                        &App::client(),
                    )
                    .await
                },
                clone!(
                    #[weak(rename_to = imp)]
                    self,
                    #[strong]
                    url,
                    #[upgrade_or_panic]
                    move |res| {
                        match res {
                            Ok(ParsedUrl::SingleFeed(feed)) => {
                                imp.add_feed_widget.reset();
                                imp.add_feed_widget.set_feed(GFeed::from(*feed));
                                imp.navigation_view.replace_with_tags(&["discover", "add_feed"]);
                            }
                            Ok(ParsedUrl::MultipleFeeds(feeds)) => {
                                imp.select_feed_widget.fill(feeds);
                                imp.navigation_view.replace_with_tags(&["discover", "select_feed"]);
                            }
                            Err(error) => {
                                log::error!("No feed found for url '{url}': {error}");
                                imp.obj().close();
                                ContentPage::instance().newsflash_error(
                                    &i18n("Failed to parse feed"),
                                    NewsFlashError::API(FeedApiError::ParseFeed(error)),
                                );
                            }
                        }
                    }
                ),
            );
        }
    }
}

glib::wrapper! {
    pub struct DiscoverDialog(ObjectSubclass<imp::DiscoverDialog>)
        @extends Widget, Dialog;
}

impl Default for DiscoverDialog {
    fn default() -> Self {
        Object::builder().property("selected-language-index", 0u32).build()
    }
}
