use super::ArticleView;
use super::builder::ArticleViewResources;
use super::header::ArticleHeader;
use super::scrollable::ArticleScrollable;
use crate::app::{App, WEBKIT_DATA_DIR};
use crate::gobject_models::GArticle;
use crate::main_window::MainWindow;
use crate::self_stack::SelfStack;
use crate::util::{CHANNEL_ERROR, GTK_RESOURCE_FILE_ERROR, GtkUtil};
use futures::channel::oneshot::Sender as OneShotSender;
use gdk4::RGBA;
use gio::Cancellable;
use glib::{ControlFlow, Object, Properties, SignalHandlerId, SourceId, clone, prelude::*, subclass::*};
use gtk4::{
    CompositeTemplate, EventController, EventControllerMotion, EventControllerScroll, PolicyType, PropagationPhase,
    ScrolledWindow, Stack, Widget, prelude::*, subclass::prelude::*,
};
use libadwaita::{Bin, subclass::prelude::*};
use news_flash::models::Url;
use once_cell::sync::Lazy;
use std::cell::{Cell, RefCell};
use std::time::Duration;
use webkit6::{
    CacheModel, ContextMenuAction, ContextMenuItem, LoadEvent, NavigationPolicyDecision, NetworkProxyMode,
    NetworkProxySettings, NetworkSession, PolicyDecisionType, Settings as WebkitSettings, URIRequest,
    UserContentFilterStore, UserContentInjectedFrames, UserScript, UserScriptInjectionTime, UserStyleLevel,
    UserStyleSheet, WebContext, WebView, WebsiteDataTypes, prelude::*,
};

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::Webview)]
    #[template(file = "data/resources/ui_templates/article_view/webview.blp")]
    pub struct Webview {
        #[template_child]
        pub stack: TemplateChild<Stack>,
        #[template_child]
        pub self_stack: TemplateChild<SelfStack>,
        #[template_child]
        pub header: TemplateChild<ArticleHeader>,
        #[template_child]
        pub scrollable: TemplateChild<ArticleScrollable>,
        #[template_child]
        pub scrolled_window: TemplateChild<ScrolledWindow>,

        #[property(get, set, nullable)]
        pub article: RefCell<Option<GArticle>>,

        #[property(get, set)]
        pub view: RefCell<WebView>,

        #[property(get, set)]
        pub web_context: RefCell<WebContext>,

        #[property(get, set)]
        pub network_session: RefCell<NetworkSession>,

        #[property(get, set, nullable)]
        pub hovered_url: RefCell<Option<String>>,

        #[property(get, set, nullable)]
        pub base_uri: RefCell<Option<String>>,

        #[property(get, set, default = 0.0)]
        pub progress: Cell<f64>,

        #[property(get, set, default = 1.0)]
        pub zoom: Cell<f64>,

        #[property(get, set)]
        pub is_fullscreen: Cell<bool>,

        #[property(get, set = Self::set_is_webview_fullscreen)]
        pub is_webview_fullscreen: Cell<bool>,

        #[property(get, set = Self::set_is_enclosure_fullscreen)]
        pub is_enclosure_fullscreen: Cell<bool>,

        #[property(get, set)]
        pub image_dialog_visible: Cell<bool>,

        #[property(get, set, name = "status-margin")]
        pub status_margin: Cell<i32>,

        pub unfreeze_insurance_source: RefCell<Option<SourceId>>,

        pub decide_policy_signal: RefCell<Option<SignalHandlerId>>,
        pub mouse_over_signal: RefCell<Option<SignalHandlerId>>,
        pub ctx_menu_signal: RefCell<Option<SignalHandlerId>>,
        pub load_progress_signal: RefCell<Option<SignalHandlerId>>,
        pub load_changed_signal: RefCell<Option<SignalHandlerId>>,
        pub scroll_fade_cooldown: RefCell<Option<SourceId>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for Webview {
        const NAME: &'static str = "Webview";
        type ParentType = Bin;
        type Type = super::Webview;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for Webview {
        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> = Lazy::new(|| {
                vec![
                    Signal::builder("image-dialog")
                        .param_types([String::static_type()])
                        .build(),
                ]
            });
            SIGNALS.as_ref()
        }

        fn constructed(&self) {
            let network_session = NetworkSession::new(WEBKIT_DATA_DIR.to_str(), WEBKIT_DATA_DIR.to_str());

            let proxies = App::default().settings().advanced().proxy.clone();
            if !proxies.is_empty() {
                let mut proxy_settings = NetworkProxySettings::new(None, &[]);
                for proxy in proxies {
                    match proxy.protocoll() {
                        crate::settings::ProxyProtocoll::All => {
                            proxy_settings.add_proxy_for_scheme("http", &proxy.url());
                            proxy_settings.add_proxy_for_scheme("https", &proxy.url());
                        }
                        crate::settings::ProxyProtocoll::Http => {
                            proxy_settings.add_proxy_for_scheme("http", &proxy.url())
                        }
                        crate::settings::ProxyProtocoll::Https => {
                            proxy_settings.add_proxy_for_scheme("https", &proxy.url())
                        }
                    }
                }
                network_session.set_proxy_settings(NetworkProxyMode::Custom, Some(&proxy_settings));
            }

            let web_context = WebContext::new();
            web_context.set_cache_model(CacheModel::DocumentViewer);

            log::info!(
                "WebKit version: {}.{}.{}",
                webkit6::functions::major_version(),
                webkit6::functions::minor_version(),
                webkit6::functions::micro_version()
            );

            self.web_context.replace(web_context);
            self.network_session.replace(network_session);

            self.new_webview();
            self.load_user_data();

            self.view.borrow().load_html("", None);
            self.view.borrow().set_sensitive(false);
            self.stack.set_visible_child_name("empty");

            self.header
                .bind_property("hovered_url", &*self.obj(), "hovered_url")
                .build();

            let controllers = self.scrolled_window.observe_controllers();
            for pos in 0..100 {
                let Some(controller) = controllers.item(pos).and_downcast::<EventController>() else {
                    break;
                };

                if let Some(motion_controller) = controller.downcast_ref::<EventControllerMotion>() {
                    motion_controller.set_propagation_phase(PropagationPhase::Capture);
                }

                if let Some(scroll_controller) = controller.downcast_ref::<EventControllerScroll>() {
                    scroll_controller.set_propagation_phase(PropagationPhase::Capture);
                }
            }
        }
    }

    impl WidgetImpl for Webview {}

    impl BinImpl for Webview {}

    #[gtk4::template_callbacks]
    impl Webview {
        fn set_is_webview_fullscreen(&self, is_fullscreen: bool) {
            self.is_webview_fullscreen.set(is_fullscreen);
            self.obj()
                .set_is_fullscreen(is_fullscreen || self.is_enclosure_fullscreen.get());
        }

        fn set_is_enclosure_fullscreen(&self, is_fullscreen: bool) {
            self.is_enclosure_fullscreen.set(is_fullscreen);
            self.obj()
                .set_is_fullscreen(is_fullscreen || self.is_webview_fullscreen.get());

            let policy = if is_fullscreen {
                PolicyType::Never
            } else {
                PolicyType::Automatic
            };
            self.scrolled_window.set_vscrollbar_policy(policy);
        }

        fn new_webview(&self) {
            let settings = WebkitSettings::new();
            settings.set_hardware_acceleration_policy(webkit6::HardwareAccelerationPolicy::Always);
            settings.set_enable_html5_database(false);
            settings.set_enable_html5_local_storage(false);
            settings.set_enable_page_cache(false);
            settings.set_enable_smooth_scrolling(true);
            settings.set_enable_javascript(true);
            settings.set_javascript_can_access_clipboard(false);
            settings.set_javascript_can_open_windows_automatically(false);
            settings.set_media_playback_requires_user_gesture(true);
            settings.set_user_agent_with_application_details(Some("Newsflash"), None);
            settings.set_enable_developer_extras(App::default().settings().advanced().inspect_article_view);
            let auto_load = App::default().settings().advanced().article_view_load_images;
            settings.set_auto_load_images(auto_load);

            let obj = self.obj();
            let webview = Object::builder::<WebView>()
                .property("network-session", obj.network_session())
                .property("web-context", obj.web_context())
                .build();
            webview.set_settings(&settings);
            webview.set_vexpand(true);

            self.connect_webview(&webview);
            self.load_content_filter(&webview);

            // disable the webviews own scroll event controller to not interfere with custom scrolling
            let controllers = webview.observe_controllers();
            for pos in 0..100 {
                let Some(controller) = controllers.item(pos).and_downcast::<EventController>() else {
                    break;
                };

                if let Some(scroll_controller) = controller.downcast_ref::<EventControllerScroll>() {
                    scroll_controller.set_propagation_phase(PropagationPhase::None);
                }
            }

            obj.set_view(webview);
        }

        fn connect_webview(&self, webview: &WebView) {
            //----------------------------------
            // open link in external browser
            //----------------------------------
            self.decide_policy_signal
                .replace(Some(webview.connect_decide_policy(clone!(
                    #[weak(rename_to = obj)]
                    self.obj(),
                    #[upgrade_or_panic]
                    move |_closure_webivew, decision, decision_type| {
                        let Some(mut navigation_action) = decision
                            .downcast_ref::<NavigationPolicyDecision>()
                            .and_then(NavigationPolicyDecision::navigation_action)
                        else {
                            return false;
                        };

                        let Some(uri) = navigation_action
                            .request()
                            .as_ref()
                            .and_then(URIRequest::uri)
                            .map(|uri| uri.to_string())
                        else {
                            return false;
                        };

                        log::debug!("navigation action uri: {uri}");

                        if let Some(base_uri) = obj.base_uri() {
                            let suffix = uri.strip_prefix(base_uri.as_str()).unwrap_or_default();
                            if let Some(id) = suffix.strip_prefix('#') {
                                decision.ignore();
                                obj.imp().scrollable.scroll_to_element_by_id(id);
                                return false;
                            }
                        }

                        if Url::parse(&uri).is_err() {
                            return false;
                        }

                        let is_user_gesture = navigation_action.is_user_gesture();

                        let is_blank = navigation_action
                            .frame_name()
                            .map(|name| name == "_blank")
                            .unwrap_or(false);

                        if decision_type == PolicyDecisionType::NewWindowAction && is_blank {
                            decision.ignore();
                            App::open_url_in_default_browser(&uri);
                            true
                        } else if decision_type == PolicyDecisionType::NavigationAction && is_user_gesture {
                            decision.ignore();
                            App::open_url_in_default_browser(&uri);
                            false
                        } else {
                            false
                        }
                    },
                ))));

            //----------------------------------
            // show url overlay
            //----------------------------------
            self.mouse_over_signal
                .replace(Some(webview.connect_mouse_target_changed(clone!(
                    #[weak(rename_to = obj)]
                    self.obj(),
                    #[upgrade_or_panic]
                    move |_closure_webivew, hit_test, _modifiers| obj.set_hovered_url(hit_test.link_uri())
                ))));

            //----------------------------------
            // clean up context menu
            //----------------------------------
            self.ctx_menu_signal.replace(Some(webview.connect_context_menu(
                |_closure_webivew, ctx_menu, hit_test| {
                    let menu_items = ctx_menu.items();

                    for item in menu_items {
                        if item.is_separator() {
                            ctx_menu.remove(&item);
                            continue;
                        }

                        let keep_stock_actions = [
                            ContextMenuAction::CopyLinkToClipboard,
                            ContextMenuAction::Copy,
                            ContextMenuAction::CopyImageToClipboard,
                            ContextMenuAction::CopyImageUrlToClipboard,
                            ContextMenuAction::InspectElement,
                        ];

                        if !keep_stock_actions.contains(&item.stock_action()) {
                            ctx_menu.remove(&item);
                        }
                    }

                    if hit_test.context_is_image()
                        && let Some(save_image_action) = App::default().lookup_action("save-webview-image")
                    {
                        let image_uri = hit_test.image_uri().unwrap().as_str().to_variant();
                        let save_image_item =
                            ContextMenuItem::from_gaction(&save_image_action, "Save Image", Some(&image_uri));
                        ctx_menu.append(&save_image_item);
                    }

                    if hit_test.context_is_link()
                        && let Some(open_uri_action) = MainWindow::instance().lookup_action("open-uri-in-browser")
                    {
                        let uri = hit_test.link_uri().unwrap().as_str().to_variant();
                        let open_uri_item = ContextMenuItem::from_gaction(&open_uri_action, "Open Link", Some(&uri));
                        ctx_menu.insert(&open_uri_item, 0);
                    }

                    if ctx_menu.first().is_none() {
                        return true;
                    }

                    false
                },
            )));

            //----------------------------------
            // display load progress
            //----------------------------------
            self.load_progress_signal
                .replace(Some(webview.connect_estimated_load_progress_notify(clone!(
                    #[weak(rename_to = obj)]
                    self.obj(),
                    #[upgrade_or_panic]
                    move |closure_webivew| {
                        let progress = closure_webivew.estimated_load_progress();
                        obj.set_progress(progress);
                    }
                ))));

            self.load_changed_signal
                .replace(Some(webview.connect_load_changed(clone!(
                    #[weak(rename_to = imp)]
                    self,
                    #[upgrade_or_panic]
                    move |_closure_webview, event| {
                        GtkUtil::remove_source(imp.unfreeze_insurance_source.take());
                        if event == LoadEvent::Committed && imp.self_stack.is_frozen() {
                            imp.self_stack.update(gtk4::StackTransitionType::Crossfade);
                        }
                    }
                ))));

            //----------------------------------
            // crash view
            //----------------------------------
            webview.connect_web_process_terminated(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_closure_webivew, _reason| imp.on_crash()
            ));

            //----------------------------------
            // fullscreen
            //----------------------------------
            webview.connect_enter_fullscreen(clone!(
                #[weak(rename_to = obj)]
                self.obj(),
                #[upgrade_or_panic]
                move |_| {
                    obj.set_is_webview_fullscreen(true);
                    false
                }
            ));
            webview.connect_leave_fullscreen(clone!(
                #[weak(rename_to = obj)]
                self.obj(),
                #[upgrade_or_panic]
                move |_| {
                    obj.set_is_webview_fullscreen(false);
                    false
                }
            ));

            //----------------------------------
            // zoom
            //----------------------------------
            webview
                .bind_property("zoom-level", &*self.obj(), "zoom")
                .sync_create()
                .bidirectional()
                .build();
        }

        fn on_crash(&self) {
            // disconnect webview signals
            GtkUtil::disconnect_signal(self.decide_policy_signal.take(), &*self.view.borrow());
            GtkUtil::disconnect_signal(self.mouse_over_signal.take(), &*self.view.borrow());
            GtkUtil::disconnect_signal(self.ctx_menu_signal.take(), &*self.view.borrow());
            GtkUtil::disconnect_signal(self.load_progress_signal.take(), &*self.view.borrow());
            GtkUtil::disconnect_signal(self.load_changed_signal.take(), &*self.view.borrow());

            self.new_webview();
            self.stack.set_visible_child_name("crash");
        }

        fn load_content_filter(&self, webview: &WebView) {
            let Some(user_content_manager) = webview.user_content_manager() else {
                return;
            };

            let Some(webkit_data_dir) = WEBKIT_DATA_DIR.to_str() else {
                return;
            };

            let store = UserContentFilterStore::new(webkit_data_dir);
            let user_content_manager_clone = user_content_manager.clone();

            let css_filter_data = ArticleViewResources::get("stylesheet_filter.json").expect(GTK_RESOURCE_FILE_ERROR);
            store.save(
                "stylesheetfilter",
                &glib::Bytes::from_owned(css_filter_data.data),
                Cancellable::NONE,
                move |result| {
                    if let Ok(filter) = result {
                        user_content_manager.add_filter(&filter);
                        log::debug!("Stylesheet filter loaded");
                    } else if let Err(error) = result {
                        log::error!("Failed to load Stylesheet filter: {error}");
                    }
                },
            );

            let easylist_adblock_data =
                ArticleViewResources::get("easylist_min_content_blocker.json").expect(GTK_RESOURCE_FILE_ERROR);
            store.save(
                "easylistadblock",
                &glib::Bytes::from_owned(easylist_adblock_data.data),
                Cancellable::NONE,
                move |result| {
                    if let Ok(filter) = result {
                        user_content_manager_clone.add_filter(&filter);
                        log::debug!("Easylist adblock filter loaded");
                    } else if let Err(error) = result {
                        log::error!("Failed to load Easylist adblock filter: {error}");
                    }
                },
            );
        }

        pub fn load_user_data(&self) {
            let Some(user_content_manager) = self.view.borrow().user_content_manager() else {
                return;
            };

            user_content_manager.remove_all_style_sheets();
            user_content_manager.remove_all_scripts();

            let stylesheet = Self::generate_user_style_sheet();
            user_content_manager.add_style_sheet(&stylesheet);

            let image_dialog_js = Self::generate_img_dialog_js();
            user_content_manager.add_script(&image_dialog_js);
            user_content_manager.register_script_message_handler("imageDialog", None);

            let resize_observer_js = Self::generate_resize_observer_js();
            user_content_manager.add_script(&resize_observer_js);
            user_content_manager.register_script_message_handler("contentSize", None);

            let (hightlight_style, hightlight_js) = Self::generate_hightlight_data();
            user_content_manager.add_style_sheet(&hightlight_style);
            user_content_manager.add_script(&hightlight_js);

            user_content_manager.connect_script_message_received(
                Some("imageDialog"),
                clone!(
                    #[weak(rename_to = imp)]
                    self,
                    move |_manager, js_value| {
                        // return if an image dialog is already visible
                        if imp.image_dialog_visible.get() {
                            return;
                        }

                        let data = js_value.to_str();
                        imp.obj().emit_by_name::<()>("image-dialog", &[&data])
                    }
                ),
            );

            user_content_manager.connect_script_message_received(
                Some("contentSize"),
                clone!(
                    #[weak(rename_to = imp)]
                    self,
                    move |_manager, js_value| {
                        // return if an image dialog is already visible
                        if imp.image_dialog_visible.get() {
                            return;
                        }

                        let upper_str = js_value.to_str();
                        let upper = upper_str.parse::<i32>().unwrap();

                        imp.scrollable.update_size(upper);
                    }
                ),
            );
        }

        fn generate_img_dialog_js() -> UserScript {
            let js_data = ArticleViewResources::get("image_dialog.js").expect(GTK_RESOURCE_FILE_ERROR);
            let js_str =
                std::str::from_utf8(js_data.data.as_ref()).expect("Failed to load image_dialog.js css from resources");

            UserScript::new(
                js_str,
                UserContentInjectedFrames::AllFrames,
                UserScriptInjectionTime::Start,
                &[],
                &[],
            )
        }

        fn generate_resize_observer_js() -> UserScript {
            let js_data = ArticleViewResources::get("resize_observer.js").expect(GTK_RESOURCE_FILE_ERROR);
            let js_str =
                std::str::from_utf8(js_data.data.as_ref()).expect("Failed to load image_dialog.js css from resources");

            UserScript::new(
                js_str,
                UserContentInjectedFrames::AllFrames,
                UserScriptInjectionTime::Start,
                &[],
                &[],
            )
        }

        fn generate_hightlight_data() -> (UserStyleSheet, UserScript) {
            let css_data = ArticleViewResources::get("highlight.js/dark.min.css").expect(GTK_RESOURCE_FILE_ERROR);
            let css_str =
                std::str::from_utf8(css_data.data.as_ref()).expect("Failed to load hightlight.js css from resources");

            let style = UserStyleSheet::new(
                css_str,
                UserContentInjectedFrames::AllFrames,
                UserStyleLevel::User,
                &[],
                &[],
            );

            let js_data = ArticleViewResources::get("highlight.js/highlight.min.js").expect(GTK_RESOURCE_FILE_ERROR);
            let js_str =
                std::str::from_utf8(js_data.data.as_ref()).expect("Failed to load hightlight.js code from resources");

            let script = UserScript::new(
                js_str,
                UserContentInjectedFrames::AllFrames,
                UserScriptInjectionTime::Start,
                &[],
                &[],
            );
            (style, script)
        }

        fn generate_user_style_sheet() -> UserStyleSheet {
            let content_width = App::default().settings().article_view().content_width();
            let line_height = App::default().settings().article_view().line_height();

            let css_data = ArticleViewResources::get("style.css").expect(GTK_RESOURCE_FILE_ERROR);
            let css_string: String = std::str::from_utf8(css_data.data.as_ref())
                .expect("Failed to load CSS from resources")
                .into();
            let css_string = css_string.replacen("$CONTENT_WIDTH", &format!("{content_width}px"), 1);
            let css_string = css_string.replacen("$LINE_HEIGHT", &format!("{line_height}em"), 1);

            UserStyleSheet::new(
                &css_string,
                UserContentInjectedFrames::TopFrame,
                UserStyleLevel::User,
                &[],
                &[],
            )
        }

        pub(super) fn freeze_view(&self) {
            self.self_stack.freeze();

            let unfreeze_callback = clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or]
                ControlFlow::Break,
                move || {
                    imp.unfreeze_insurance_source.replace(None);
                    if imp.self_stack.is_frozen() {
                        imp.self_stack.update(gtk4::StackTransitionType::Crossfade);
                    }
                    ControlFlow::Break
                }
            );

            let unfreeze_source_id = glib::timeout_add_local(Duration::from_millis(1500), unfreeze_callback);
            self.unfreeze_insurance_source.replace(Some(unfreeze_source_id));
        }
    }
}

glib::wrapper! {
    pub struct Webview(ObjectSubclass<imp::Webview>)
        @extends Widget, Bin;
}

impl Default for Webview {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl Webview {
    pub fn instance() -> Self {
        ArticleView::instance().imp().view.get()
    }

    pub fn load(&self, html: String, base_url: Option<&str>) {
        let imp = self.imp();

        if !self.is_empty() {
            imp.freeze_view();
        } else {
            imp.stack.set_visible_child_name("article");
        }
        let base_url = base_url.map(str::to_string);
        self.set_base_uri(base_url.clone());
        imp.view.borrow().set_height_request(-1);
        imp.view.borrow().load_html(&html, base_url.as_deref());
        imp.view.borrow().set_sensitive(true);
    }

    pub fn clear(&self) {
        let imp = self.imp();

        imp.view.borrow().set_sensitive(false);
        imp.view.borrow().set_height_request(-1);
        imp.view.borrow().load_html("", None);
        imp.stack.set_visible_child_name("empty");
        self.set_base_uri(None::<String>);
    }

    pub fn is_empty(&self) -> bool {
        self.imp()
            .stack
            .visible_child_name()
            .map(|s| s.as_str() != "article")
            .unwrap_or(false)
    }

    pub fn scroll_diff(&self, diff: f64) {
        let imp = self.imp();
        imp.scrolled_window
            .vadjustment()
            .set_value(imp.scrolled_window.vadjustment().value() + diff);
    }

    pub fn set_scroll_pos(&self, pos: f64) {
        self.imp().scrolled_window.vadjustment().set_value(pos);
    }

    pub fn get_scroll_pos(&self) -> f64 {
        self.imp().scrolled_window.vadjustment().value()
    }

    pub fn get_scroll_upper(&self) -> f64 {
        self.imp().scrolled_window.vadjustment().upper()
    }

    pub fn clear_cache(&self, oneshot_sender: OneShotSender<()>) {
        let imp = self.imp();

        let Some(data_manager) = imp
            .view
            .borrow()
            .network_session()
            .as_ref()
            .and_then(NetworkSession::website_data_manager)
        else {
            return;
        };

        data_manager.clear(
            WebsiteDataTypes::all(),
            glib::TimeSpan::from_seconds(0),
            Cancellable::NONE,
            move |res| {
                if let Err(error) = res {
                    log::error!("Failed to clear webkit cache: {error}");
                }
                oneshot_sender.send(()).expect(CHANNEL_ERROR);
            },
        );
    }

    pub fn update_background_color(&self, color: &RGBA) {
        self.imp().view.borrow().set_background_color(color);
    }
}
