/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */

// This file is a Mako template: http://www.makotemplates.org/





use servo_arc::{Arc, UniqueArc};
use std::{ops, ptr, fmt, mem};

#[cfg(feature = "servo")] use euclid::SideOffsets2D;
#[cfg(feature = "gecko")] use crate::gecko_bindings::structs::{self, NonCustomCSSPropertyId};
#[cfg(feature = "servo")] use crate::logical_geometry::LogicalMargin;
#[cfg(feature = "servo")] use crate::computed_values;
#[cfg(feature = "servo")] use crate::dom::AttributeReferences;
use crate::logical_geometry::WritingMode;
use malloc_size_of::{MallocSizeOf, MallocSizeOfOps};
use crate::computed_value_flags::*;
use cssparser::Parser;
use crate::media_queries::Device;
use crate::parser::ParserContext;
use crate::selector_parser::PseudoElement;
use crate::stylist::Stylist;
use style_traits::{CssStringWriter, CssWriter, KeywordsCollectFn, ParseError, SpecifiedValueInfo, StyleParseErrorKind, ToCss, TypedValue, ToTyped};
use crate::derives::*;
use crate::stylesheets::{CssRuleType, CssRuleTypes, Origin};
use crate::logical_geometry::{LogicalAxis, LogicalCorner, LogicalSide};
use crate::use_counters::UseCounters;
use crate::rule_tree::StrongRuleNode;
use crate::values::{
    computed,
    resolved,
    specified::{font::SystemFont, length::LineHeightBase, color::ColorSchemeFlags},
};
use std::cell::Cell;
use super::{
    PropertyDeclarationId, PropertyId, NonCustomPropertyId,
    NonCustomPropertyIdSet, PropertyFlags, SourcePropertyDeclaration,
    LonghandIdSet, VariableDeclaration, CustomDeclaration,
    WideKeywordDeclaration, NonCustomPropertyIterator, TransitionPropertyIterator,
};
use debug_unreachable::debug_unreachable;

/// Conversion with fewer impls than From/Into
pub trait MaybeBoxed<Out> {
    /// Convert
    fn maybe_boxed(self) -> Out;
}

impl<T> MaybeBoxed<T> for T {
    #[inline]
    fn maybe_boxed(self) -> T { self }
}

impl<T> MaybeBoxed<Box<T>> for T {
    #[inline]
    fn maybe_boxed(self) -> Box<T> { Box::new(self) }
}

/// A module with all the code for longhand properties.
#[allow(missing_docs)]
pub mod longhands {

/// https://drafts.csswg.org/css-align/#propdef-align-items
pub mod align_items {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AlignItems,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_align_items();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ItemPlacement>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_align_items(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AlignItems)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ItemPlacement as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ItemPlacement as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::ItemPlacement::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        specified::ItemPlacement::parse_block(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-inline-3/#alignment-baseline
pub mod alignment_baseline {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AlignmentBaseline,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_alignment_baseline();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::AlignmentBaseline>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_alignment_baseline(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AlignmentBaseline)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AlignmentBaseline as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AlignmentBaseline as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AlignmentBaseline::Baseline }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AlignmentBaseline as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-sizing-4/#aspect-ratio
pub mod aspect_ratio {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AspectRatio,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_aspect_ratio();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::AspectRatio>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_aspect_ratio(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AspectRatio)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AspectRatio as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AspectRatio as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AspectRatio::auto() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AspectRatio as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-transforms/#backface-visibility-property
pub mod backface_visibility {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackfaceVisibility,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_backface_visibility();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::backface_visibility::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_backface_visibility(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackfaceVisibility)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Visible,
        
        Hidden,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Visible
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Visible
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const VISIBLE: u32 = structs::StyleBackfaceVisibility::Visible as u32;
            // We can't match on enum values if we're matching on a u32
            const HIDDEN: u32 = structs::StyleBackfaceVisibility::Hidden as u32;
            match kw {
                VISIBLE => Self::Visible,
                HIDDEN => Self::Hidden,
                _ => panic!("Found unexpected value in style struct for backface-visibility property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-inline-3/#baseline-source
pub mod baseline_source {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BaselineSource,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_baseline_source();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BaselineSource>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_baseline_source(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BaselineSource)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BaselineSource as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BaselineSource as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BaselineSource::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BaselineSource as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-tables/#propdef-border-collapse
pub mod border_collapse {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BorderCollapse,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_border_collapse();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::border_collapse::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_border_collapse(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderCollapse)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Separate,
        
        Collapse,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Separate
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Separate
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const SEPARATE: u32 = structs::StyleBorderCollapse::Separate as u32;
            // We can't match on enum values if we're matching on a u32
            const COLLAPSE: u32 = structs::StyleBorderCollapse::Collapse as u32;
            match kw {
                SEPARATE => Self::Separate,
                COLLAPSE => Self::Collapse,
                _ => panic!("Found unexpected value in style struct for border-collapse property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-backgrounds/#the-border-image-repeat
pub mod border_image_repeat {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BorderImageRepeat,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_border_image_repeat();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BorderImageRepeat>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_border_image_repeat(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderImageRepeat)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderImageRepeat as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderImageRepeat as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderImageRepeat::stretch() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BorderImageRepeat::stretch() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderImageRepeat as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-break/#propdef-box-decoration-break
pub mod box_decoration_break {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BoxDecorationBreak,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_box_decoration_break();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::box_decoration_break::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_box_decoration_break(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BoxDecorationBreak)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Slice,
        
        Clone,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Slice
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Slice
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const SLICE: u32 = structs::StyleBoxDecorationBreak::Slice as u32;
            // We can't match on enum values if we're matching on a u32
            const CLONE: u32 = structs::StyleBoxDecorationBreak::Clone as u32;
            match kw {
                SLICE => Self::Slice,
                CLONE => Self::Clone,
                _ => panic!("Found unexpected value in style struct for box-decoration-break property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ui/#propdef-box-sizing
pub mod box_sizing {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BoxSizing,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_box_sizing();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::box_sizing::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_box_sizing(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BoxSizing)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        ContentBox,
        
        BorderBox,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::ContentBox
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::ContentBox
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const CONTENT_BOX: u32 = structs::StyleBoxSizing::ContentBox as u32;
            // We can't match on enum values if we're matching on a u32
            const BORDER_BOX: u32 = structs::StyleBoxSizing::BorderBox as u32;
            match kw {
                CONTENT_BOX => Self::ContentBox,
                BORDER_BOX => Self::BorderBox,
                _ => panic!("Found unexpected value in style struct for box-sizing property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-break/#propdef-break-inside
pub mod break_inside {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BreakInside,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_break_inside();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BreakWithin>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_break_inside(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BreakInside)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BreakWithin as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BreakWithin as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BreakWithin::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BreakWithin as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-tables/#propdef-caption-side
pub mod caption_side {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::CaptionSide,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_caption_side();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::table::CaptionSide>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_caption_side(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::CaptionSide)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::table::CaptionSide as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::table::CaptionSide as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::table::CaptionSide::Top }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::table::CaptionSide as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css2/#propdef-clear
pub mod clear {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Clear,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_clear();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Clear>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_clear(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Clear)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Clear as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Clear as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Clear::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Clear as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#ColorInterpolationProperty
pub mod color_interpolation {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ColorInterpolation,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_color_interpolation();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::color_interpolation::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_color_interpolation(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ColorInterpolation)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Srgb,
        
        Auto,
        
        Linearrgb,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Srgb
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Srgb
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const SRGB: u32 = structs::StyleColorInterpolation::Srgb as u32;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleColorInterpolation::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const LINEARRGB: u32 = structs::StyleColorInterpolation::Linearrgb as u32;
            match kw {
                SRGB => Self::Srgb,
                AUTO => Self::Auto,
                LINEARRGB => Self::Linearrgb,
                _ => panic!("Found unexpected value in style struct for color-interpolation property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#ColorInterpolationFiltersProperty
pub mod color_interpolation_filters {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ColorInterpolationFilters,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_color_interpolation_filters();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::color_interpolation_filters::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_color_interpolation_filters(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ColorInterpolationFilters)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Linearrgb,
        
        Auto,
        
        Srgb,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Linearrgb
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Linearrgb
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const LINEARRGB: u32 = structs::StyleColorInterpolation::Linearrgb as u32;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleColorInterpolation::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const SRGB: u32 = structs::StyleColorInterpolation::Srgb as u32;
            match kw {
                LINEARRGB => Self::Linearrgb,
                AUTO => Self::Auto,
                SRGB => Self::Srgb,
                _ => panic!("Found unexpected value in style struct for color-interpolation-filters property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-multicol/#propdef-column-count
pub mod column_count {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ColumnCount,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_column_count();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ColumnCount>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_column_count(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ColumnCount)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ColumnCount as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ColumnCount as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ColumnCount::Auto }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::ColumnCount::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ColumnCount as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-multicol/#propdef-column-fill
pub mod column_fill {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ColumnFill,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_column_fill();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::column_fill::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_column_fill(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ColumnFill)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Balance,
        
        Auto,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Balance
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Balance
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const BALANCE: u32 = structs::StyleColumnFill::Balance as u32;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleColumnFill::Auto as u32;
            match kw {
                BALANCE => Self::Balance,
                AUTO => Self::Auto,
                _ => panic!("Found unexpected value in style struct for column-fill property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-multicol/#propdef-column-span
pub mod column_span {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ColumnSpan,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_column_span();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::column_span::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_column_span(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ColumnSpan)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        None,
        
        All,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::None
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::None
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleColumnSpan::None as u32;
            // We can't match on enum values if we're matching on a u32
            const ALL: u32 = structs::StyleColumnSpan::All as u32;
            match kw {
                NONE => Self::None,
                ALL => Self::All,
                _ => panic!("Found unexpected value in style struct for column-span property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-contain/#contain-property
pub mod contain {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Contain,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_contain();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Contain>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_contain(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Contain)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Contain as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Contain as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::Contain::empty() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Contain as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-contain-3/#container-type
pub mod container_type {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ContainerType,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_container_type();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ContainerType>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_container_type(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ContainerType)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ContainerType as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ContainerType as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ContainerType::NORMAL }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ContainerType as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-contain/#content-visibility
pub mod content_visibility {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ContentVisibility,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_content_visibility();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ContentVisibility>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_content_visibility(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ContentVisibility)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ContentVisibility as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ContentVisibility as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ContentVisibility::Visible }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ContentVisibility as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-writing-modes/#propdef-direction
pub mod direction {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Direction,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_direction();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::direction::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_direction(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Direction)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Ltr,
        
        Rtl,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Ltr
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Ltr
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const LTR: u32 = structs::StyleDirection::Ltr as u32;
            // We can't match on enum values if we're matching on a u32
            const RTL: u32 = structs::StyleDirection::Rtl as u32;
            match kw {
                LTR => Self::Ltr,
                RTL => Self::Rtl,
                _ => panic!("Found unexpected value in style struct for direction property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-display/#propdef-display
pub mod display {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Display,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_display();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Display>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_display(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Display)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Display as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Display as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Display::inline() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Display::inline() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Display as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-inline-3/#dominant-baseline-property
pub mod dominant_baseline {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::DominantBaseline,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_dominant_baseline();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::DominantBaseline>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_dominant_baseline(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::DominantBaseline)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::DominantBaseline as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::DominantBaseline as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::DominantBaseline::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::DominantBaseline as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-tables/#propdef-empty-cells
pub mod empty_cells {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::EmptyCells,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_empty_cells();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::empty_cells::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_empty_cells(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::EmptyCells)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Show,
        
        Hide,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Show
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Show
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const SHOW: u32 = structs::StyleEmptyCells::Show as u32;
            // We can't match on enum values if we're matching on a u32
            const HIDE: u32 = structs::StyleEmptyCells::Hide as u32;
            match kw {
                SHOW => Self::Show,
                HIDE => Self::Hide,
                _ => panic!("Found unexpected value in style struct for empty-cells property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ui/#field-sizing
pub mod field_sizing {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FieldSizing,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_field_sizing();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::field_sizing::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_field_sizing(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FieldSizing)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Fixed,
        
        Content,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Fixed
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Fixed
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const FIXED: u32 = structs::StyleFieldSizing::Fixed as u32;
            // We can't match on enum values if we're matching on a u32
            const CONTENT: u32 = structs::StyleFieldSizing::Content as u32;
            match kw {
                FIXED => Self::Fixed,
                CONTENT => Self::Content,
                _ => panic!("Found unexpected value in style struct for field-sizing property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-flexbox/#flex-direction-property
pub mod flex_direction {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FlexDirection,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_flex_direction();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::flex_direction::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_flex_direction(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FlexDirection)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Row,
        
        RowReverse,
        
        Column,
        
        ColumnReverse,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Row
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Row
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const ROW: u32 = structs::StyleFlexDirection::Row as u32;
            // We can't match on enum values if we're matching on a u32
            const ROW_REVERSE: u32 = structs::StyleFlexDirection::RowReverse as u32;
            // We can't match on enum values if we're matching on a u32
            const COLUMN: u32 = structs::StyleFlexDirection::Column as u32;
            // We can't match on enum values if we're matching on a u32
            const COLUMN_REVERSE: u32 = structs::StyleFlexDirection::ColumnReverse as u32;
            match kw {
                ROW => Self::Row,
                ROW_REVERSE => Self::RowReverse,
                COLUMN => Self::Column,
                COLUMN_REVERSE => Self::ColumnReverse,
                _ => panic!("Found unexpected value in style struct for flex-direction property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-flexbox/#flex-wrap-property
pub mod flex_wrap {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FlexWrap,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_flex_wrap();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::flex_wrap::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_flex_wrap(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FlexWrap)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Nowrap,
        
        Wrap,
        
        WrapReverse,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Nowrap
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Nowrap
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NOWRAP: u32 = structs::StyleFlexWrap::Nowrap as u32;
            // We can't match on enum values if we're matching on a u32
            const WRAP: u32 = structs::StyleFlexWrap::Wrap as u32;
            // We can't match on enum values if we're matching on a u32
            const WRAP_REVERSE: u32 = structs::StyleFlexWrap::WrapReverse as u32;
            match kw {
                NOWRAP => Self::Nowrap,
                WRAP => Self::Wrap,
                WRAP_REVERSE => Self::WrapReverse,
                _ => panic!("Found unexpected value in style struct for flex-wrap property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-box/#propdef-float
pub mod float {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Float,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_float();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Float>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_float(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Float)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Float as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Float as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Float::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Float as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-kerning
pub mod font_kerning {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontKerning,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_kerning();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::font_kerning::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_kerning(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontKerning)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        None,
        
        Normal,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::NS_FONT_KERNING_AUTO as u32;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::NS_FONT_KERNING_NONE as u32;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::NS_FONT_KERNING_NORMAL as u32;
            match kw {
                AUTO => Self::Auto,
                NONE => Self::None,
                NORMAL => Self::Normal,
                _ => panic!("Found unexpected value in style struct for font-kerning property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts-3/#propdef-font-language-override
pub mod font_language_override {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontLanguageOverride,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_language_override();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontLanguageOverride>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_language_override(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontLanguageOverride)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontLanguageOverride as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontLanguageOverride as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontLanguageOverride::normal() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontLanguageOverride::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontLanguageOverride as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://www.w3.org/TR/css-fonts-4/#font-optical-sizing-def
pub mod font_optical_sizing {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontOpticalSizing,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_optical_sizing();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::font_optical_sizing::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_optical_sizing(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontOpticalSizing)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        None,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::NS_FONT_OPTICAL_SIZING_AUTO as u32;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::NS_FONT_OPTICAL_SIZING_NONE as u32;
            match kw {
                AUTO => Self::Auto,
                NONE => Self::None,
                _ => panic!("Found unexpected value in style struct for font-optical-sizing property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-size-adjust
pub mod font_size_adjust {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontSizeAdjust,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_size_adjust();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontSizeAdjust>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_size_adjust(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontSizeAdjust)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontSizeAdjust as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontSizeAdjust as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSizeAdjust::None }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSizeAdjust::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontSizeAdjust as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-stretch
pub mod font_stretch {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontStretch,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_stretch();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontStretch>()
            },
        };

        if let Some(sf) = specified_value.get_system() {
            crate::properties::gecko::system_font::resolve_system_font(sf, context);
        }

            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_stretch(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontStretch)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontStretch as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontStretch as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontStretch::hundred() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontStretch::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontStretch as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-style
pub mod font_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontStyle,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_style();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontStyle>()
            },
        };

        if let Some(sf) = specified_value.get_system() {
            crate::properties::gecko::system_font::resolve_system_font(sf, context);
        }

            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontStyle::normal() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontStyle::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts-4/#font-synthesis-style
pub mod font_synthesis_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontSynthesisStyle,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_synthesis_style();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontSynthesisStyle>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_synthesis_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontSynthesisStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontSynthesisStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontSynthesisStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSynthesisStyle::Auto }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSynthesisStyle::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontSynthesisStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-variant-caps
pub mod font_variant_caps {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontVariantCaps,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_variant_caps();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::font_variant_caps::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_variant_caps(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontVariantCaps)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Normal,
        
        SmallCaps,
        
        AllSmallCaps,
        
        PetiteCaps,
        
        AllPetiteCaps,
        
        Unicase,
        
        TitlingCaps,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Normal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Normal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::NS_FONT_VARIANT_CAPS_NORMAL as u32;
            // We can't match on enum values if we're matching on a u32
            const SMALL_CAPS: u32 = structs::NS_FONT_VARIANT_CAPS_SMALL_CAPS as u32;
            // We can't match on enum values if we're matching on a u32
            const ALL_SMALL_CAPS: u32 = structs::NS_FONT_VARIANT_CAPS_ALL_SMALL_CAPS as u32;
            // We can't match on enum values if we're matching on a u32
            const PETITE_CAPS: u32 = structs::NS_FONT_VARIANT_CAPS_PETITE_CAPS as u32;
            // We can't match on enum values if we're matching on a u32
            const ALL_PETITE_CAPS: u32 = structs::NS_FONT_VARIANT_CAPS_ALL_PETITE_CAPS as u32;
            // We can't match on enum values if we're matching on a u32
            const UNICASE: u32 = structs::NS_FONT_VARIANT_CAPS_UNICASE as u32;
            // We can't match on enum values if we're matching on a u32
            const TITLING_CAPS: u32 = structs::NS_FONT_VARIANT_CAPS_TITLING_CAPS as u32;
            match kw {
                NORMAL => Self::Normal,
                SMALL_CAPS => Self::SmallCaps,
                ALL_SMALL_CAPS => Self::AllSmallCaps,
                PETITE_CAPS => Self::PetiteCaps,
                ALL_PETITE_CAPS => Self::AllPetiteCaps,
                UNICASE => Self::Unicase,
                TITLING_CAPS => Self::TitlingCaps,
                _ => panic!("Found unexpected value in style struct for font-variant-caps property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-variant-east-asian
pub mod font_variant_east_asian {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontVariantEastAsian,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_variant_east_asian();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontVariantEastAsian>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_variant_east_asian(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontVariantEastAsian)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontVariantEastAsian as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontVariantEastAsian as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontVariantEastAsian::empty() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontVariantEastAsian::empty() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontVariantEastAsian as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-variant-emoji
pub mod font_variant_emoji {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontVariantEmoji,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_variant_emoji();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::font_variant_emoji::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_variant_emoji(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontVariantEmoji)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Normal,
        
        Text,
        
        Emoji,
        
        Unicode,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Normal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Normal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::StyleFontVariantEmoji::Normal as u32;
            // We can't match on enum values if we're matching on a u32
            const TEXT: u32 = structs::StyleFontVariantEmoji::Text as u32;
            // We can't match on enum values if we're matching on a u32
            const EMOJI: u32 = structs::StyleFontVariantEmoji::Emoji as u32;
            // We can't match on enum values if we're matching on a u32
            const UNICODE: u32 = structs::StyleFontVariantEmoji::Unicode as u32;
            match kw {
                NORMAL => Self::Normal,
                TEXT => Self::Text,
                EMOJI => Self::Emoji,
                UNICODE => Self::Unicode,
                _ => panic!("Found unexpected value in style struct for font-variant-emoji property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-variant-ligatures
pub mod font_variant_ligatures {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontVariantLigatures,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_variant_ligatures();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontVariantLigatures>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_variant_ligatures(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontVariantLigatures)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontVariantLigatures as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontVariantLigatures as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontVariantLigatures::empty() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontVariantLigatures::empty() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontVariantLigatures as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-variant-numeric
pub mod font_variant_numeric {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontVariantNumeric,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_variant_numeric();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontVariantNumeric>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_variant_numeric(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontVariantNumeric)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontVariantNumeric as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontVariantNumeric as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontVariantNumeric::empty() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontVariantNumeric::empty() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontVariantNumeric as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-variant-position
pub mod font_variant_position {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontVariantPosition,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_variant_position();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::font_variant_position::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_variant_position(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontVariantPosition)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Normal,
        
        Sub,
        
        Super,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Normal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Normal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::NS_FONT_VARIANT_POSITION_NORMAL as u32;
            // We can't match on enum values if we're matching on a u32
            const SUB: u32 = structs::NS_FONT_VARIANT_POSITION_SUB as u32;
            // We can't match on enum values if we're matching on a u32
            const SUPER_: u32 = structs::NS_FONT_VARIANT_POSITION_SUPER as u32;
            match kw {
                NORMAL => Self::Normal,
                SUB => Self::Sub,
                SUPER_ => Self::Super,
                _ => panic!("Found unexpected value in style struct for font-variant-position property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts/#propdef-font-weight
pub mod font_weight {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontWeight,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_weight();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontWeight>()
            },
        };

        if let Some(sf) = specified_value.get_system() {
            crate::properties::gecko::system_font::resolve_system_font(sf, context);
        }

            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_weight(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontWeight)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontWeight as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontWeight as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontWeight::normal() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontWeight::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontWeight as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-color-adjust-1/#forced-color-adjust-prop
pub mod forced_color_adjust {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ForcedColorAdjust,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_forced_color_adjust();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ForcedColorAdjust>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_forced_color_adjust(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ForcedColorAdjust)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ForcedColorAdjust as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ForcedColorAdjust as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ForcedColorAdjust::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ForcedColorAdjust as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-grid/#propdef-grid-auto-flow
pub mod grid_auto_flow {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::GridAutoFlow,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_grid_auto_flow();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::GridAutoFlow>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_grid_auto_flow(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::GridAutoFlow)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::GridAutoFlow as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::GridAutoFlow as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::GridAutoFlow::ROW }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::GridAutoFlow as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text/#propdef-hyphens
pub mod hyphens {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Hyphens,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_hyphens();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::hyphens::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_hyphens(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Hyphens)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Manual,
        
        None,
        
        Auto,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Manual
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Manual
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const MANUAL: u32 = structs::StyleHyphens::Manual as u32;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleHyphens::None as u32;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleHyphens::Auto as u32;
            match kw {
                MANUAL => Self::Manual,
                NONE => Self::None,
                AUTO => Self::Auto,
                _ => panic!("Found unexpected value in style struct for hyphens property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-images/#propdef-image-orientation
pub mod image_orientation {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ImageOrientation,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_image_orientation();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::image_orientation::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_image_orientation(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ImageOrientation)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        FromImage,
        
        None,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::FromImage
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::FromImage
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const FROM_IMAGE: u32 = structs::StyleImageOrientation::FromImage as u32;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleImageOrientation::None as u32;
            match kw {
                FROM_IMAGE => Self::FromImage,
                NONE => Self::None,
                _ => panic!("Found unexpected value in style struct for image-orientation property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-images/#propdef-image-rendering
pub mod image_rendering {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ImageRendering,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_image_rendering();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ImageRendering>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_image_rendering(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ImageRendering)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ImageRendering as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ImageRendering as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ImageRendering::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ImageRendering as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ui/#input-method-editor
pub mod ime_mode {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ImeMode,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_ime_mode();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::ime_mode::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_ime_mode(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ImeMode)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        Normal,
        
        Active,
        
        Disabled,
        
        Inactive,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleImeMode::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::StyleImeMode::Normal as u32;
            // We can't match on enum values if we're matching on a u32
            const ACTIVE: u32 = structs::StyleImeMode::Active as u32;
            // We can't match on enum values if we're matching on a u32
            const DISABLED: u32 = structs::StyleImeMode::Disabled as u32;
            // We can't match on enum values if we're matching on a u32
            const INACTIVE: u32 = structs::StyleImeMode::Inactive as u32;
            match kw {
                AUTO => Self::Auto,
                NORMAL => Self::Normal,
                ACTIVE => Self::Active,
                DISABLED => Self::Disabled,
                INACTIVE => Self::Inactive,
                _ => panic!("Found unexpected value in style struct for ime-mode property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-inline/#sizing-drop-initials
pub mod initial_letter {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::InitialLetter,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_initial_letter();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::InitialLetter>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_initial_letter(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::InitialLetter)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::InitialLetter as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::InitialLetter as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::InitialLetter::normal() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::InitialLetter::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::InitialLetter as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.fxtf.org/compositing/#isolation
pub mod isolation {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Isolation,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_isolation();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::isolation::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_isolation(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Isolation)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        Isolate,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleIsolation::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const ISOLATE: u32 = structs::StyleIsolation::Isolate as u32;
            match kw {
                AUTO => Self::Auto,
                ISOLATE => Self::Isolate,
                _ => panic!("Found unexpected value in style struct for isolation property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-align/#propdef-justify-items
pub mod justify_items {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::JustifyItems,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_justify_items();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::JustifyItems>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_justify_items(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::JustifyItems)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::JustifyItems as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::JustifyItems as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::JustifyItems::legacy() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::JustifyItems as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-3/#line-break-property
pub mod line_break {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::LineBreak,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_line_break();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::LineBreak>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_line_break(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::LineBreak)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::LineBreak as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::LineBreak as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::LineBreak::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::LineBreak as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-lists/#propdef-list-style-position
pub mod list_style_position {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ListStylePosition,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_list_style_position();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::list_style_position::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_list_style_position(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ListStylePosition)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Outside,
        
        Inside,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Outside
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Outside
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const OUTSIDE: u32 = structs::StyleListStylePosition::Outside as u32;
            // We can't match on enum values if we're matching on a u32
            const INSIDE: u32 = structs::StyleListStylePosition::Inside as u32;
            match kw {
                OUTSIDE => Self::Outside,
                INSIDE => Self::Inside,
                _ => panic!("Found unexpected value in style struct for list-style-position property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.fxtf.org/css-masking-1/#propdef-mask-type
pub mod mask_type {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MaskType,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_mask_type();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::mask_type::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_mask_type(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MaskType)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Luminance,
        
        Alpha,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Luminance
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Luminance
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const LUMINANCE: u32 = structs::StyleMaskType::Luminance as u32;
            // We can't match on enum values if we're matching on a u32
            const ALPHA: u32 = structs::StyleMaskType::Alpha as u32;
            match kw {
                LUMINANCE => Self::Luminance,
                ALPHA => Self::Alpha,
                _ => panic!("Found unexpected value in style struct for mask-type property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://github.com/w3c/csswg-drafts/issues/4650
pub mod masonry_auto_flow {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MasonryAutoFlow,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_masonry_auto_flow();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::MasonryAutoFlow>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_masonry_auto_flow(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MasonryAutoFlow)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::MasonryAutoFlow as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::MasonryAutoFlow as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::MasonryAutoFlow::initial() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::MasonryAutoFlow as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://mathml-refresh.github.io/mathml-core/#the-math-script-level-property
pub mod math_depth {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MathDepth,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_math_depth();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::MathDepth>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_math_depth(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MathDepth)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::MathDepth as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::MathDepth as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { 0 }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::MathDepth as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://w3c.github.io/mathml-core/#the-math-shift
pub mod math_shift {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MathShift,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_math_shift();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::math_shift::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_math_shift(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MathShift)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Normal,
        
        Compact,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Normal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Normal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::StyleMathShift::Normal as u32;
            // We can't match on enum values if we're matching on a u32
            const COMPACT: u32 = structs::StyleMathShift::Compact as u32;
            match kw {
                NORMAL => Self::Normal,
                COMPACT => Self::Compact,
                _ => panic!("Found unexpected value in style struct for math-shift property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://mathml-refresh.github.io/mathml-core/#the-math-style-property
pub mod math_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MathStyle,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_math_style();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::math_style::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_math_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MathStyle)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Normal,
        
        Compact,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Normal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Normal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::StyleMathStyle::Normal as u32;
            // We can't match on enum values if we're matching on a u32
            const COMPACT: u32 = structs::StyleMathStyle::Compact as u32;
            match kw {
                NORMAL => Self::Normal,
                COMPACT => Self::Compact,
                _ => panic!("Found unexpected value in style struct for math-style property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.fxtf.org/compositing/#propdef-mix-blend-mode
pub mod mix_blend_mode {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MixBlendMode,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_mix_blend_mode();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::mix_blend_mode::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_mix_blend_mode(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MixBlendMode)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Normal,
        
        Multiply,
        
        Screen,
        
        Overlay,
        
        Darken,
        
        Lighten,
        
        ColorDodge,
        
        ColorBurn,
        
        HardLight,
        
        SoftLight,
        
        Difference,
        
        Exclusion,
        
        Hue,
        
        Saturation,
        
        Color,
        
        Luminosity,
        
        PlusLighter,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Normal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Normal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::StyleBlend::Normal as u32;
            // We can't match on enum values if we're matching on a u32
            const MULTIPLY: u32 = structs::StyleBlend::Multiply as u32;
            // We can't match on enum values if we're matching on a u32
            const SCREEN: u32 = structs::StyleBlend::Screen as u32;
            // We can't match on enum values if we're matching on a u32
            const OVERLAY: u32 = structs::StyleBlend::Overlay as u32;
            // We can't match on enum values if we're matching on a u32
            const DARKEN: u32 = structs::StyleBlend::Darken as u32;
            // We can't match on enum values if we're matching on a u32
            const LIGHTEN: u32 = structs::StyleBlend::Lighten as u32;
            // We can't match on enum values if we're matching on a u32
            const COLOR_DODGE: u32 = structs::StyleBlend::ColorDodge as u32;
            // We can't match on enum values if we're matching on a u32
            const COLOR_BURN: u32 = structs::StyleBlend::ColorBurn as u32;
            // We can't match on enum values if we're matching on a u32
            const HARD_LIGHT: u32 = structs::StyleBlend::HardLight as u32;
            // We can't match on enum values if we're matching on a u32
            const SOFT_LIGHT: u32 = structs::StyleBlend::SoftLight as u32;
            // We can't match on enum values if we're matching on a u32
            const DIFFERENCE: u32 = structs::StyleBlend::Difference as u32;
            // We can't match on enum values if we're matching on a u32
            const EXCLUSION: u32 = structs::StyleBlend::Exclusion as u32;
            // We can't match on enum values if we're matching on a u32
            const HUE: u32 = structs::StyleBlend::Hue as u32;
            // We can't match on enum values if we're matching on a u32
            const SATURATION: u32 = structs::StyleBlend::Saturation as u32;
            // We can't match on enum values if we're matching on a u32
            const COLOR: u32 = structs::StyleBlend::Color as u32;
            // We can't match on enum values if we're matching on a u32
            const LUMINOSITY: u32 = structs::StyleBlend::Luminosity as u32;
            // We can't match on enum values if we're matching on a u32
            const PLUS_LIGHTER: u32 = structs::StyleBlend::PlusLighter as u32;
            match kw {
                NORMAL => Self::Normal,
                MULTIPLY => Self::Multiply,
                SCREEN => Self::Screen,
                OVERLAY => Self::Overlay,
                DARKEN => Self::Darken,
                LIGHTEN => Self::Lighten,
                COLOR_DODGE => Self::ColorDodge,
                COLOR_BURN => Self::ColorBurn,
                HARD_LIGHT => Self::HardLight,
                SOFT_LIGHT => Self::SoftLight,
                DIFFERENCE => Self::Difference,
                EXCLUSION => Self::Exclusion,
                HUE => Self::Hue,
                SATURATION => Self::Saturation,
                COLOR => Self::Color,
                LUMINOSITY => Self::Luminosity,
                PLUS_LIGHTER => Self::PlusLighter,
                _ => panic!("Found unexpected value in style struct for mix-blend-mode property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/box-align)
pub mod _moz_box_align {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozBoxAlign,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_box_align();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_box_align::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_box_align(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozBoxAlign)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Stretch,
        
        Start,
        
        Center,
        
        Baseline,
        
        End,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Stretch
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Stretch
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const STRETCH: u32 = structs::StyleBoxAlign::Stretch as u32;
            // We can't match on enum values if we're matching on a u32
            const START: u32 = structs::StyleBoxAlign::Start as u32;
            // We can't match on enum values if we're matching on a u32
            const CENTER: u32 = structs::StyleBoxAlign::Center as u32;
            // We can't match on enum values if we're matching on a u32
            const BASELINE: u32 = structs::StyleBoxAlign::Baseline as u32;
            // We can't match on enum values if we're matching on a u32
            const END: u32 = structs::StyleBoxAlign::End as u32;
            match kw {
                STRETCH => Self::Stretch,
                START => Self::Start,
                CENTER => Self::Center,
                BASELINE => Self::Baseline,
                END => Self::End,
                _ => panic!("Found unexpected value in style struct for -moz-box-align property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// None (internal)
pub mod _moz_box_collapse {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozBoxCollapse,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__moz_box_collapse();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_box_collapse::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_box_collapse(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozBoxCollapse)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Flex,
        
        Legacy,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Flex
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Flex
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const FLEX: u32 = structs::StyleBoxCollapse::Flex as u32;
            // We can't match on enum values if we're matching on a u32
            const LEGACY: u32 = structs::StyleBoxCollapse::Legacy as u32;
            match kw {
                FLEX => Self::Flex,
                LEGACY => Self::Legacy,
                _ => panic!("Found unexpected value in style struct for -moz-box-collapse property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/box-direction)
pub mod _moz_box_direction {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozBoxDirection,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_box_direction();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_box_direction::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_box_direction(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozBoxDirection)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Normal,
        
        Reverse,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Normal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Normal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::StyleBoxDirection::Normal as u32;
            // We can't match on enum values if we're matching on a u32
            const REVERSE: u32 = structs::StyleBoxDirection::Reverse as u32;
            match kw {
                NORMAL => Self::Normal,
                REVERSE => Self::Reverse,
                _ => panic!("Found unexpected value in style struct for -moz-box-direction property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/box-orient)
pub mod _moz_box_orient {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozBoxOrient,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_box_orient();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_box_orient::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_box_orient(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozBoxOrient)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        #[parse(aliases = "inline-axis")]
        Horizontal,
        
        #[parse(aliases = "block-axis")]
        Vertical,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Horizontal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Horizontal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const HORIZONTAL: u32 = structs::StyleBoxOrient::Horizontal as u32;
            // We can't match on enum values if we're matching on a u32
            const VERTICAL: u32 = structs::StyleBoxOrient::Vertical as u32;
            match kw {
                HORIZONTAL => Self::Horizontal,
                VERTICAL => Self::Vertical,
                _ => panic!("Found unexpected value in style struct for -moz-box-orient property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/box-pack)
pub mod _moz_box_pack {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozBoxPack,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_box_pack();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_box_pack::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_box_pack(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozBoxPack)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Start,
        
        Center,
        
        End,
        
        Justify,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Start
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Start
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const START: u32 = structs::StyleBoxPack::Start as u32;
            // We can't match on enum values if we're matching on a u32
            const CENTER: u32 = structs::StyleBoxPack::Center as u32;
            // We can't match on enum values if we're matching on a u32
            const END: u32 = structs::StyleBoxPack::End as u32;
            // We can't match on enum values if we're matching on a u32
            const JUSTIFY: u32 = structs::StyleBoxPack::Justify as u32;
            match kw {
                START => Self::Start,
                CENTER => Self::Center,
                END => Self::End,
                JUSTIFY => Self::Justify,
                _ => panic!("Found unexpected value in style struct for -moz-box-pack property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard
pub mod _moz_control_character_visibility {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozControlCharacterVisibility,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__moz_control_character_visibility();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::text::MozControlCharacterVisibility>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_control_character_visibility(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozControlCharacterVisibility)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::text::MozControlCharacterVisibility as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::text::MozControlCharacterVisibility as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::text::MozControlCharacterVisibility as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/-moz-float-edge)
pub mod _moz_float_edge {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozFloatEdge,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_float_edge();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_float_edge::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_float_edge(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozFloatEdge)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        ContentBox,
        
        MarginBox,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::ContentBox
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::ContentBox
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const CONTENT_BOX: u32 = structs::StyleFloatEdge::ContentBox as u32;
            // We can't match on enum values if we're matching on a u32
            const MARGIN_BOX: u32 = structs::StyleFloatEdge::MarginBox as u32;
            match kw {
                CONTENT_BOX => Self::ContentBox,
                MARGIN_BOX => Self::MarginBox,
                _ => panic!("Found unexpected value in style struct for -moz-float-edge property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://html.spec.whatwg.org/multipage/#inert-subtrees)
pub mod _moz_inert {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozInert,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__moz_inert();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Inert>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_inert(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozInert)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Inert as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Inert as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::Inert::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Inert as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Internal (not web-exposed)
pub mod _moz_math_variant {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozMathVariant,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__moz_math_variant();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_math_variant::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_math_variant(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozMathVariant)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        None,
        
        Normal,
        
        Bold,
        
        Italic,
        
        BoldItalic,
        
        Script,
        
        BoldScript,
        
        Fraktur,
        
        DoubleStruck,
        
        BoldFraktur,
        
        SansSerif,
        
        BoldSansSerif,
        
        SansSerifItalic,
        
        SansSerifBoldItalic,
        
        Monospace,
        
        Initial,
        
        Tailed,
        
        Looped,
        
        Stretched,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::None
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::None
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleMathVariant::None as u32;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::StyleMathVariant::Normal as u32;
            // We can't match on enum values if we're matching on a u32
            const BOLD: u32 = structs::StyleMathVariant::Bold as u32;
            // We can't match on enum values if we're matching on a u32
            const ITALIC: u32 = structs::StyleMathVariant::Italic as u32;
            // We can't match on enum values if we're matching on a u32
            const BOLD_ITALIC: u32 = structs::StyleMathVariant::BoldItalic as u32;
            // We can't match on enum values if we're matching on a u32
            const SCRIPT: u32 = structs::StyleMathVariant::Script as u32;
            // We can't match on enum values if we're matching on a u32
            const BOLD_SCRIPT: u32 = structs::StyleMathVariant::BoldScript as u32;
            // We can't match on enum values if we're matching on a u32
            const FRAKTUR: u32 = structs::StyleMathVariant::Fraktur as u32;
            // We can't match on enum values if we're matching on a u32
            const DOUBLE_STRUCK: u32 = structs::StyleMathVariant::DoubleStruck as u32;
            // We can't match on enum values if we're matching on a u32
            const BOLD_FRAKTUR: u32 = structs::StyleMathVariant::BoldFraktur as u32;
            // We can't match on enum values if we're matching on a u32
            const SANS_SERIF: u32 = structs::StyleMathVariant::SansSerif as u32;
            // We can't match on enum values if we're matching on a u32
            const BOLD_SANS_SERIF: u32 = structs::StyleMathVariant::BoldSansSerif as u32;
            // We can't match on enum values if we're matching on a u32
            const SANS_SERIF_ITALIC: u32 = structs::StyleMathVariant::SansSerifItalic as u32;
            // We can't match on enum values if we're matching on a u32
            const SANS_SERIF_BOLD_ITALIC: u32 = structs::StyleMathVariant::SansSerifBoldItalic as u32;
            // We can't match on enum values if we're matching on a u32
            const MONOSPACE: u32 = structs::StyleMathVariant::Monospace as u32;
            // We can't match on enum values if we're matching on a u32
            const INITIAL: u32 = structs::StyleMathVariant::Initial as u32;
            // We can't match on enum values if we're matching on a u32
            const TAILED: u32 = structs::StyleMathVariant::Tailed as u32;
            // We can't match on enum values if we're matching on a u32
            const LOOPED: u32 = structs::StyleMathVariant::Looped as u32;
            // We can't match on enum values if we're matching on a u32
            const STRETCHED: u32 = structs::StyleMathVariant::Stretched as u32;
            match kw {
                NONE => Self::None,
                NORMAL => Self::Normal,
                BOLD => Self::Bold,
                ITALIC => Self::Italic,
                BOLD_ITALIC => Self::BoldItalic,
                SCRIPT => Self::Script,
                BOLD_SCRIPT => Self::BoldScript,
                FRAKTUR => Self::Fraktur,
                DOUBLE_STRUCK => Self::DoubleStruck,
                BOLD_FRAKTUR => Self::BoldFraktur,
                SANS_SERIF => Self::SansSerif,
                BOLD_SANS_SERIF => Self::BoldSansSerif,
                SANS_SERIF_ITALIC => Self::SansSerifItalic,
                SANS_SERIF_BOLD_ITALIC => Self::SansSerifBoldItalic,
                MONOSPACE => Self::Monospace,
                INITIAL => Self::Initial,
                TAILED => Self::Tailed,
                LOOPED => Self::Looped,
                STRETCHED => Self::Stretched,
                _ => panic!("Found unexpected value in style struct for -moz-math-variant property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (Internal-only)
pub mod _moz_min_font_size_ratio {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozMinFontSizeRatio,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__moz_min_font_size_ratio();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Percentage>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_min_font_size_ratio(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozMinFontSizeRatio)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Percentage as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Percentage as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Percentage::hundred() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Percentage as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/-moz-orient)
pub mod _moz_orient {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozOrient,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_orient();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_orient::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_orient(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozOrient)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Inline,
        
        Block,
        
        Horizontal,
        
        Vertical,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Inline
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Inline
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const INLINE: u32 = structs::StyleOrient::Inline as u32;
            // We can't match on enum values if we're matching on a u32
            const BLOCK: u32 = structs::StyleOrient::Block as u32;
            // We can't match on enum values if we're matching on a u32
            const HORIZONTAL: u32 = structs::StyleOrient::Horizontal as u32;
            // We can't match on enum values if we're matching on a u32
            const VERTICAL: u32 = structs::StyleOrient::Vertical as u32;
            match kw {
                INLINE => Self::Inline,
                BLOCK => Self::Block,
                HORIZONTAL => Self::Horizontal,
                VERTICAL => Self::Vertical,
                _ => panic!("Found unexpected value in style struct for -moz-orient property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/font-smooth)
pub mod _moz_osx_font_smoothing {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozOsxFontSmoothing,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__moz_osx_font_smoothing();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_osx_font_smoothing::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_osx_font_smoothing(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozOsxFontSmoothing)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        #[parse(aliases = "antialiased")]
        Grayscale,
        
        SubpixelAntialiased,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::NS_FONT_SMOOTHING_AUTO as u32;
            // We can't match on enum values if we're matching on a u32
            const GRAYSCALE: u32 = structs::NS_FONT_SMOOTHING_GRAYSCALE as u32;
            // We can't match on enum values if we're matching on a u32
            const SUBPIXEL_ANTIALIASED: u32 = structs::NS_FONT_SMOOTHING_SUBPIXEL_ANTIALIASED as u32;
            match kw {
                AUTO => Self::Auto,
                GRAYSCALE => Self::Grayscale,
                SUBPIXEL_ANTIALIASED => Self::SubpixelAntialiased,
                _ => panic!("Found unexpected value in style struct for -moz-osx-font-smoothing property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-size-adjust/#adjustment-control
pub mod _moz_text_size_adjust {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozTextSizeAdjust,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__moz_text_size_adjust();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_text_size_adjust::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_text_size_adjust(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozTextSizeAdjust)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        None,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleTextSizeAdjust::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleTextSizeAdjust::None as u32;
            match kw {
                AUTO => Self::Auto,
                NONE => Self::None,
                _ => panic!("Found unexpected value in style struct for -moz-text-size-adjust property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Internal
pub mod _moz_theme {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozTheme,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__moz_theme();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::MozTheme>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_theme(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozTheme)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::MozTheme as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::MozTheme as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::MozTheme::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::MozTheme as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Internal (not web-exposed)
pub mod _moz_top_layer {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozTopLayer,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_top_layer();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_top_layer::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_top_layer(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozTopLayer)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        None,
        
        Auto,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::None
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::None
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleTopLayer::None as u32;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleTopLayer::Auto as u32;
            match kw {
                NONE => Self::None,
                AUTO => Self::Auto,
                _ => panic!("Found unexpected value in style struct for -moz-top-layer property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/-moz-user-focus)
pub mod _moz_user_focus {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozUserFocus,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__moz_user_focus();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::UserFocus>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_user_focus(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozUserFocus)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::UserFocus as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::UserFocus as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::UserFocus::Normal }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::UserFocus as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// None (Nonstandard Firefox-only property)
pub mod _moz_window_dragging {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozWindowDragging,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_window_dragging();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_window_dragging::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_window_dragging(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozWindowDragging)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Default,
        
        Drag,
        
        NoDrag,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Default
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Default
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const DEFAULT: u32 = structs::StyleWindowDragging::Default as u32;
            // We can't match on enum values if we're matching on a u32
            const DRAG: u32 = structs::StyleWindowDragging::Drag as u32;
            // We can't match on enum values if we're matching on a u32
            const NO_DRAG: u32 = structs::StyleWindowDragging::NoDrag as u32;
            match kw {
                DEFAULT => Self::Default,
                DRAG => Self::Drag,
                NO_DRAG => Self::NoDrag,
                _ => panic!("Found unexpected value in style struct for -moz-window-dragging property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// None (Nonstandard internal property)
pub mod _moz_window_shadow {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozWindowShadow,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_window_shadow();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_moz_window_shadow::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_window_shadow(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozWindowShadow)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        None,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleWindowShadow::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleWindowShadow::None as u32;
            match kw {
                AUTO => Self::Auto,
                NONE => Self::None,
                _ => panic!("Found unexpected value in style struct for -moz-window-shadow property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-images/#propdef-object-fit
pub mod object_fit {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ObjectFit,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_object_fit();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::object_fit::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_object_fit(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ObjectFit)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Fill,
        
        Contain,
        
        Cover,
        
        None,
        
        ScaleDown,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Fill
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Fill
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const FILL: u32 = structs::StyleObjectFit::Fill as u32;
            // We can't match on enum values if we're matching on a u32
            const CONTAIN: u32 = structs::StyleObjectFit::Contain as u32;
            // We can't match on enum values if we're matching on a u32
            const COVER: u32 = structs::StyleObjectFit::Cover as u32;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleObjectFit::None as u32;
            // We can't match on enum values if we're matching on a u32
            const SCALE_DOWN: u32 = structs::StyleObjectFit::ScaleDown as u32;
            match kw {
                FILL => Self::Fill,
                CONTAIN => Self::Contain,
                COVER => Self::Cover,
                NONE => Self::None,
                SCALE_DOWN => Self::ScaleDown,
                _ => panic!("Found unexpected value in style struct for object-fit property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.fxtf.org/motion-1/#offset-rotate-property
pub mod offset_rotate {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::OffsetRotate,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_offset_rotate();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::OffsetRotate>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_offset_rotate(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OffsetRotate)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::OffsetRotate as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::OffsetRotate as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::OffsetRotate::auto() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::OffsetRotate as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ui/#propdef-outline-style
pub mod outline_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::OutlineStyle,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_outline_style();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::OutlineStyle>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_outline_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OutlineStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::OutlineStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::OutlineStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::OutlineStyle::none() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::OutlineStyle::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::OutlineStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-scroll-anchoring/#exclusion-api
pub mod overflow_anchor {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::OverflowAnchor,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_overflow_anchor();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::OverflowAnchor>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_overflow_anchor(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverflowAnchor)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::OverflowAnchor as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::OverflowAnchor as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverflowAnchor::Auto }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::OverflowAnchor::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::OverflowAnchor as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text/#propdef-overflow-wrap
pub mod overflow_wrap {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::OverflowWrap,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_overflow_wrap();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::OverflowWrap>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_overflow_wrap(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverflowWrap)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::OverflowWrap as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::OverflowWrap as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverflowWrap::Normal }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::OverflowWrap as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-page-3/#page-orientation-prop
pub mod page_orientation {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::PageOrientation,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_page_orientation();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::PageOrientation>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_page_orientation(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::PageOrientation)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::PageOrientation as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::PageOrientation as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::PageOrientation::Upright }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::PageOrientation::Upright }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::PageOrientation as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#PaintOrder
pub mod paint_order {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::PaintOrder,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_paint_order();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::SVGPaintOrder>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_paint_order(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::PaintOrder)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::SVGPaintOrder as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::SVGPaintOrder as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::SVGPaintOrder::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::SVGPaintOrder as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/interact.html#PointerEventsProperty
pub mod pointer_events {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::PointerEvents,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_pointer_events();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::PointerEvents>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_pointer_events(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::PointerEvents)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::PointerEvents as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::PointerEvents as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::PointerEvents::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::PointerEvents as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-position/#position-property
pub mod position {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Position,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_position();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::PositionProperty>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_position(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Position)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::PositionProperty as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::PositionProperty as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::PositionProperty::Static }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::PositionProperty::Static }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::PositionProperty as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-anchor-position-1/#typedef-position-area
pub mod position_area {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::PositionArea,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_position_area();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::PositionArea>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_position_area(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::PositionArea)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::PositionArea as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::PositionArea as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::PositionArea::none() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::PositionArea::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::PositionArea as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-anchor-position-1/#position-try-order-property
pub mod position_try_order {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::PositionTryOrder,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_position_try_order();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::PositionTryOrder>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_position_try_order(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::PositionTryOrder)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::PositionTryOrder as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::PositionTryOrder as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::PositionTryOrder::normal() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::PositionTryOrder::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::PositionTryOrder as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-anchor-position-1/#propdef-position-visibility
pub mod position_visibility {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::PositionVisibility,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_position_visibility();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::PositionVisibility>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_position_visibility(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::PositionVisibility)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::PositionVisibility as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::PositionVisibility as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::PositionVisibility::ANCHORS_VISIBLE }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::PositionVisibility::ANCHORS_VISIBLE }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::PositionVisibility as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-color-adjust/#print-color-adjust
pub mod print_color_adjust {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::PrintColorAdjust,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_print_color_adjust();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::PrintColorAdjust>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_print_color_adjust(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::PrintColorAdjust)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::PrintColorAdjust as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::PrintColorAdjust as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::PrintColorAdjust::Economy }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::PrintColorAdjust as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ui/#propdef-resize
pub mod resize {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Resize,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_resize();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Resize>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_resize(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Resize)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Resize as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Resize as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Resize::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Resize as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ruby/#ruby-align-property
pub mod ruby_align {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::RubyAlign,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_ruby_align();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::ruby_align::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_ruby_align(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::RubyAlign)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        SpaceAround,
        
        Start,
        
        Center,
        
        SpaceBetween,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::SpaceAround
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::SpaceAround
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const SPACE_AROUND: u32 = structs::StyleRubyAlign::SpaceAround as u32;
            // We can't match on enum values if we're matching on a u32
            const START: u32 = structs::StyleRubyAlign::Start as u32;
            // We can't match on enum values if we're matching on a u32
            const CENTER: u32 = structs::StyleRubyAlign::Center as u32;
            // We can't match on enum values if we're matching on a u32
            const SPACE_BETWEEN: u32 = structs::StyleRubyAlign::SpaceBetween as u32;
            match kw {
                SPACE_AROUND => Self::SpaceAround,
                START => Self::Start,
                CENTER => Self::Center,
                SPACE_BETWEEN => Self::SpaceBetween,
                _ => panic!("Found unexpected value in style struct for ruby-align property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ruby/#ruby-position-property
pub mod ruby_position {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::RubyPosition,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_ruby_position();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::RubyPosition>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_ruby_position(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::RubyPosition)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::RubyPosition as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::RubyPosition as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::RubyPosition::AlternateOver }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::RubyPosition as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/cssom-view/#propdef-scroll-behavior
pub mod scroll_behavior {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ScrollBehavior,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_scroll_behavior();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::scroll_behavior::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_scroll_behavior(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ScrollBehavior)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        Smooth,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleScrollBehavior::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const SMOOTH: u32 = structs::StyleScrollBehavior::Smooth as u32;
            match kw {
                AUTO => Self::Auto,
                SMOOTH => Self::Smooth,
                _ => panic!("Found unexpected value in style struct for scroll-behavior property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-scroll-snap-1/#scroll-snap-align
pub mod scroll_snap_align {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ScrollSnapAlign,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_scroll_snap_align();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ScrollSnapAlign>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_scroll_snap_align(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ScrollSnapAlign)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ScrollSnapAlign as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ScrollSnapAlign as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ScrollSnapAlign::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ScrollSnapAlign as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-scroll-snap-1/#scroll-snap-stop
pub mod scroll_snap_stop {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ScrollSnapStop,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_scroll_snap_stop();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ScrollSnapStop>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_scroll_snap_stop(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ScrollSnapStop)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ScrollSnapStop as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ScrollSnapStop as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ScrollSnapStop::Normal }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ScrollSnapStop as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-scroll-snap-1/#scroll-snap-type
pub mod scroll_snap_type {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ScrollSnapType,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_scroll_snap_type();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ScrollSnapType>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_scroll_snap_type(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ScrollSnapType)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ScrollSnapType as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ScrollSnapType as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ScrollSnapType::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ScrollSnapType as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-overflow-3/#scrollbar-gutter-property
pub mod scrollbar_gutter {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ScrollbarGutter,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_scrollbar_gutter();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ScrollbarGutter>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_scrollbar_gutter(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ScrollbarGutter)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ScrollbarGutter as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ScrollbarGutter as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ScrollbarGutter::AUTO }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ScrollbarGutter as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-scrollbars-1/#scrollbar-width
pub mod scrollbar_width {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ScrollbarWidth,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_scrollbar_width();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::scrollbar_width::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_scrollbar_width(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ScrollbarWidth)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        Thin,
        
        None,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleScrollbarWidth::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const THIN: u32 = structs::StyleScrollbarWidth::Thin as u32;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleScrollbarWidth::None as u32;
            match kw {
                AUTO => Self::Auto,
                THIN => Self::Thin,
                NONE => Self::None,
                _ => panic!("Found unexpected value in style struct for scrollbar-width property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#ShapeRenderingProperty
pub mod shape_rendering {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ShapeRendering,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_shape_rendering();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::shape_rendering::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_shape_rendering(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ShapeRendering)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        Optimizespeed,
        
        Crispedges,
        
        Geometricprecision,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleShapeRendering::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const OPTIMIZESPEED: u32 = structs::StyleShapeRendering::Optimizespeed as u32;
            // We can't match on enum values if we're matching on a u32
            const CRISPEDGES: u32 = structs::StyleShapeRendering::Crispedges as u32;
            // We can't match on enum values if we're matching on a u32
            const GEOMETRICPRECISION: u32 = structs::StyleShapeRendering::Geometricprecision as u32;
            match kw {
                AUTO => Self::Auto,
                OPTIMIZESPEED => Self::Optimizespeed,
                CRISPEDGES => Self::Crispedges,
                GEOMETRICPRECISION => Self::Geometricprecision,
                _ => panic!("Found unexpected value in style struct for shape-rendering property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#StrokeLinecapProperty
pub mod stroke_linecap {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::StrokeLinecap,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_stroke_linecap();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::stroke_linecap::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_stroke_linecap(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::StrokeLinecap)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Butt,
        
        Round,
        
        Square,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Butt
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Butt
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const BUTT: u32 = structs::StyleStrokeLinecap::Butt as u32;
            // We can't match on enum values if we're matching on a u32
            const ROUND: u32 = structs::StyleStrokeLinecap::Round as u32;
            // We can't match on enum values if we're matching on a u32
            const SQUARE: u32 = structs::StyleStrokeLinecap::Square as u32;
            match kw {
                BUTT => Self::Butt,
                ROUND => Self::Round,
                SQUARE => Self::Square,
                _ => panic!("Found unexpected value in style struct for stroke-linecap property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#StrokeLinejoinProperty
pub mod stroke_linejoin {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::StrokeLinejoin,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_stroke_linejoin();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::stroke_linejoin::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_stroke_linejoin(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::StrokeLinejoin)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Miter,
        
        Round,
        
        Bevel,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Miter
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Miter
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const MITER: u32 = structs::StyleStrokeLinejoin::Miter as u32;
            // We can't match on enum values if we're matching on a u32
            const ROUND: u32 = structs::StyleStrokeLinejoin::Round as u32;
            // We can't match on enum values if we're matching on a u32
            const BEVEL: u32 = structs::StyleStrokeLinejoin::Bevel as u32;
            match kw {
                MITER => Self::Miter,
                ROUND => Self::Round,
                BEVEL => Self::Bevel,
                _ => panic!("Found unexpected value in style struct for stroke-linejoin property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-tables/#propdef-table-layout
pub mod table_layout {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TableLayout,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_table_layout();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::table_layout::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_table_layout(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TableLayout)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        Fixed,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleTableLayout::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const FIXED: u32 = structs::StyleTableLayout::Fixed as u32;
            match kw {
                AUTO => Self::Auto,
                FIXED => Self::Fixed,
                _ => panic!("Found unexpected value in style struct for table-layout property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text/#propdef-text-align
pub mod text_align {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextAlign,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_align();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextAlign>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_align(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextAlign)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextAlign as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextAlign as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextAlign::Start }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextAlign as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text/#propdef-text-align-last
pub mod text_align_last {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextAlignLast,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_align_last();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextAlignLast>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_align_last(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextAlignLast)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextAlignLast as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextAlignLast as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::text::TextAlignLast::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextAlignLast as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/text.html#TextAnchorProperty
pub mod text_anchor {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextAnchor,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_anchor();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::text_anchor::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_anchor(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextAnchor)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Start,
        
        Middle,
        
        End,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Start
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Start
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const START: u32 = structs::StyleTextAnchor::Start as u32;
            // We can't match on enum values if we're matching on a u32
            const MIDDLE: u32 = structs::StyleTextAnchor::Middle as u32;
            // We can't match on enum values if we're matching on a u32
            const END: u32 = structs::StyleTextAnchor::End as u32;
            match kw {
                START => Self::Start,
                MIDDLE => Self::Middle,
                END => Self::End,
                _ => panic!("Found unexpected value in style struct for text-anchor property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-4/#text-autospace-property
pub mod text_autospace {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextAutospace,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_autospace();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextAutospace>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_autospace(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextAutospace)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextAutospace as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextAutospace as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::text::TextAutospace::NO_AUTOSPACE }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextAutospace as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-inline/#text-box-edge
pub mod text_box_edge {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextBoxEdge,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_box_edge();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextBoxEdge>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_box_edge(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextBoxEdge)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextBoxEdge as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextBoxEdge as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextBoxEdge::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextBoxEdge as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-inline/#text-box-trim
pub mod text_box_trim {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextBoxTrim,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_text_box_trim();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextBoxTrim>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_box_trim(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextBoxTrim)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextBoxTrim as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextBoxTrim as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextBoxTrim::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextBoxTrim as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-writing-modes-3/#text-combine-upright
pub mod text_combine_upright {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextCombineUpright,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_combine_upright();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::text_combine_upright::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_combine_upright(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextCombineUpright)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        None,
        
        All,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::None
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::None
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleTextCombineUpright::None as u32;
            // We can't match on enum values if we're matching on a u32
            const ALL: u32 = structs::StyleTextCombineUpright::All as u32;
            match kw {
                NONE => Self::None,
                ALL => Self::All,
                _ => panic!("Found unexpected value in style struct for text-combine-upright property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-decor/#propdef-text-decoration-line
pub mod text_decoration_line {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextDecorationLine,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_text_decoration_line();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextDecorationLine>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_decoration_line(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextDecorationLine)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextDecorationLine as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextDecorationLine as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::TextDecorationLine::none() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::TextDecorationLine::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextDecorationLine as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-decor-4/#text-decoration-skip-ink-property
pub mod text_decoration_skip_ink {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextDecorationSkipInk,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_decoration_skip_ink();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextDecorationSkipInk>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_decoration_skip_ink(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextDecorationSkipInk)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextDecorationSkipInk as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextDecorationSkipInk as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextDecorationSkipInk::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextDecorationSkipInk as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-decor/#propdef-text-decoration-style
pub mod text_decoration_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextDecorationStyle,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_text_decoration_style();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::text_decoration_style::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_decoration_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextDecorationStyle)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Solid,
        
        Double,
        
        Dotted,
        
        Dashed,
        
        Wavy,
        
        MozNone,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Solid
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Solid
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const SOLID: u32 = structs::StyleTextDecorationStyle::Solid as u32;
            // We can't match on enum values if we're matching on a u32
            const DOUBLE: u32 = structs::StyleTextDecorationStyle::Double as u32;
            // We can't match on enum values if we're matching on a u32
            const DOTTED: u32 = structs::StyleTextDecorationStyle::Dotted as u32;
            // We can't match on enum values if we're matching on a u32
            const DASHED: u32 = structs::StyleTextDecorationStyle::Dashed as u32;
            // We can't match on enum values if we're matching on a u32
            const WAVY: u32 = structs::StyleTextDecorationStyle::Wavy as u32;
            // We can't match on enum values if we're matching on a u32
            const _MOZ_NONE: u32 = structs::StyleTextDecorationStyle::None as u32;
            match kw {
                SOLID => Self::Solid,
                DOUBLE => Self::Double,
                DOTTED => Self::Dotted,
                DASHED => Self::Dashed,
                WAVY => Self::Wavy,
                _MOZ_NONE => Self::MozNone,
                _ => panic!("Found unexpected value in style struct for text-decoration-style property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-decor/#propdef-text-emphasis-position
pub mod text_emphasis_position {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextEmphasisPosition,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_emphasis_position();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextEmphasisPosition>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_emphasis_position(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextEmphasisPosition)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextEmphasisPosition as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextEmphasisPosition as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextEmphasisPosition::AUTO }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::TextEmphasisPosition::AUTO }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextEmphasisPosition as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text/#propdef-text-justify
pub mod text_justify {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextJustify,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_justify();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextJustify>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_justify(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextJustify)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextJustify as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextJustify as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextJustify::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextJustify as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-writing-modes/#propdef-text-orientation
pub mod text_orientation {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextOrientation,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_orientation();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::text_orientation::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_orientation(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextOrientation)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Mixed,
        
        Upright,
        
        #[parse(aliases = "sideways-right")]
        Sideways,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Mixed
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Mixed
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const MIXED: u32 = structs::StyleTextOrientation::Mixed as u32;
            // We can't match on enum values if we're matching on a u32
            const UPRIGHT: u32 = structs::StyleTextOrientation::Upright as u32;
            // We can't match on enum values if we're matching on a u32
            const SIDEWAYS: u32 = structs::StyleTextOrientation::Sideways as u32;
            match kw {
                MIXED => Self::Mixed,
                UPRIGHT => Self::Upright,
                SIDEWAYS => Self::Sideways,
                _ => panic!("Found unexpected value in style struct for text-orientation property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#TextRenderingProperty
pub mod text_rendering {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextRendering,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_rendering();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::text_rendering::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_rendering(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextRendering)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        Optimizespeed,
        
        Optimizelegibility,
        
        Geometricprecision,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleTextRendering::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const OPTIMIZESPEED: u32 = structs::StyleTextRendering::Optimizespeed as u32;
            // We can't match on enum values if we're matching on a u32
            const OPTIMIZELEGIBILITY: u32 = structs::StyleTextRendering::Optimizelegibility as u32;
            // We can't match on enum values if we're matching on a u32
            const GEOMETRICPRECISION: u32 = structs::StyleTextRendering::Geometricprecision as u32;
            match kw {
                AUTO => Self::Auto,
                OPTIMIZESPEED => Self::Optimizespeed,
                OPTIMIZELEGIBILITY => Self::Optimizelegibility,
                GEOMETRICPRECISION => Self::Geometricprecision,
                _ => panic!("Found unexpected value in style struct for text-rendering property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text/#propdef-text-transform
pub mod text_transform {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextTransform,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_transform();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextTransform>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_transform(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextTransform)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextTransform as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextTransform as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextTransform::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextTransform as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-decor-3/#text-underline-position-property
pub mod text_underline_position {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextUnderlinePosition,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_underline_position();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TextUnderlinePosition>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_underline_position(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextUnderlinePosition)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TextUnderlinePosition as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TextUnderlinePosition as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextUnderlinePosition::AUTO }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TextUnderlinePosition as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-4/#propdef-text-wrap-mode
pub mod text_wrap_mode {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextWrapMode,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_wrap_mode();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::text_wrap_mode::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_wrap_mode(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextWrapMode)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Wrap,
        
        Nowrap,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Wrap
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Wrap
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const WRAP: u32 = structs::StyleTextWrapMode::Wrap as u32;
            // We can't match on enum values if we're matching on a u32
            const NOWRAP: u32 = structs::StyleTextWrapMode::Nowrap as u32;
            match kw {
                WRAP => Self::Wrap,
                NOWRAP => Self::Nowrap,
                _ => panic!("Found unexpected value in style struct for text-wrap-mode property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-4/#text-wrap-style
pub mod text_wrap_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TextWrapStyle,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_text_wrap_style();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::text_wrap_style::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_text_wrap_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TextWrapStyle)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Auto,
        
        Stable,
        
        Balance,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Auto
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Auto
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const AUTO: u32 = structs::StyleTextWrapStyle::Auto as u32;
            // We can't match on enum values if we're matching on a u32
            const STABLE: u32 = structs::StyleTextWrapStyle::Stable as u32;
            // We can't match on enum values if we're matching on a u32
            const BALANCE: u32 = structs::StyleTextWrapStyle::Balance as u32;
            match kw {
                AUTO => Self::Auto,
                STABLE => Self::Stable,
                BALANCE => Self::Balance,
                _ => panic!("Found unexpected value in style struct for text-wrap-style property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://compat.spec.whatwg.org/#touch-action
pub mod touch_action {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TouchAction,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_touch_action();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TouchAction>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_touch_action(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TouchAction)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TouchAction as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TouchAction as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TouchAction::auto() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TouchAction as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-transforms/#transform-box
pub mod transform_box {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TransformBox,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_transform_box();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TransformBox>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_transform_box(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TransformBox)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TransformBox as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TransformBox as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TransformBox::ViewBox }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TransformBox as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-transforms-2/#transform-style-property
pub mod transform_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::TransformStyle,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_transform_style();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::TransformStyle>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_transform_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::TransformStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TransformStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TransformStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TransformStyle::Flat }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TransformStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-writing-modes/#propdef-unicode-bidi
pub mod unicode_bidi {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::UnicodeBidi,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_unicode_bidi();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::unicode_bidi::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_unicode_bidi(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::UnicodeBidi)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Normal,
        
        Embed,
        
        Isolate,
        
        BidiOverride,
        
        IsolateOverride,
        
        Plaintext,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Normal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Normal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::StyleUnicodeBidi::Normal as u32;
            // We can't match on enum values if we're matching on a u32
            const EMBED: u32 = structs::StyleUnicodeBidi::Embed as u32;
            // We can't match on enum values if we're matching on a u32
            const ISOLATE: u32 = structs::StyleUnicodeBidi::Isolate as u32;
            // We can't match on enum values if we're matching on a u32
            const BIDI_OVERRIDE: u32 = structs::StyleUnicodeBidi::BidiOverride as u32;
            // We can't match on enum values if we're matching on a u32
            const ISOLATE_OVERRIDE: u32 = structs::StyleUnicodeBidi::IsolateOverride as u32;
            // We can't match on enum values if we're matching on a u32
            const PLAINTEXT: u32 = structs::StyleUnicodeBidi::Plaintext as u32;
            match kw {
                NORMAL => Self::Normal,
                EMBED => Self::Embed,
                ISOLATE => Self::Isolate,
                BIDI_OVERRIDE => Self::BidiOverride,
                ISOLATE_OVERRIDE => Self::IsolateOverride,
                PLAINTEXT => Self::Plaintext,
                _ => panic!("Found unexpected value in style struct for unicode-bidi property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ui-4/#propdef-user-select
pub mod user_select {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::UserSelect,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_user_select();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::UserSelect>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_user_select(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::UserSelect)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::UserSelect as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::UserSelect as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::UserSelect::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::UserSelect as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/coords.html#VectorEffects
pub mod vector_effect {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::VectorEffect,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_vector_effect();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::VectorEffect>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_vector_effect(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::VectorEffect)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::VectorEffect as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::VectorEffect as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::VectorEffect::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::VectorEffect as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-box/#propdef-visibility
pub mod visibility {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Visibility,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_visibility();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::visibility::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_visibility(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Visibility)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Visible,
        
        Hidden,
        
        Collapse,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Visible
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Visible
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const VISIBLE: u32 = structs::StyleVisibility::Visible as u32;
            // We can't match on enum values if we're matching on a u32
            const HIDDEN: u32 = structs::StyleVisibility::Hidden as u32;
            // We can't match on enum values if we're matching on a u32
            const COLLAPSE: u32 = structs::StyleVisibility::Collapse as u32;
            match kw {
                VISIBLE => Self::Visible,
                HIDDEN => Self::Hidden,
                COLLAPSE => Self::Collapse,
                _ => panic!("Found unexpected value in style struct for visibility property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-overflow-3/#line-clamp
pub mod _webkit_line_clamp {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::WebkitLineClamp,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__webkit_line_clamp();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::LineClamp>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__webkit_line_clamp(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::WebkitLineClamp)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::LineClamp as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::LineClamp as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::LineClamp::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::LineClamp as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text/#MISSING
pub mod _webkit_text_security {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::WebkitTextSecurity,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__webkit_text_security();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::_webkit_text_security::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__webkit_text_security(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::WebkitTextSecurity)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        None,
        
        Circle,
        
        Disc,
        
        Square,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::None
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::None
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NONE: u32 = structs::StyleTextSecurity::None as u32;
            // We can't match on enum values if we're matching on a u32
            const CIRCLE: u32 = structs::StyleTextSecurity::Circle as u32;
            // We can't match on enum values if we're matching on a u32
            const DISC: u32 = structs::StyleTextSecurity::Disc as u32;
            // We can't match on enum values if we're matching on a u32
            const SQUARE: u32 = structs::StyleTextSecurity::Square as u32;
            match kw {
                NONE => Self::None,
                CIRCLE => Self::Circle,
                DISC => Self::Disc,
                SQUARE => Self::Square,
                _ => panic!("Found unexpected value in style struct for -webkit-text-security property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text-4/#propdef-white-space-collapse
pub mod white_space_collapse {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::WhiteSpaceCollapse,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_white_space_collapse();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::white_space_collapse::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_white_space_collapse(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::WhiteSpaceCollapse)
    }

    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Collapse,
        
        Preserve,
        
        PreserveBreaks,
        
        BreakSpaces,
        
        #[parse(aliases = "-moz-pre-space")]
        PreserveSpaces,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Collapse
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Collapse
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const COLLAPSE: u32 = structs::StyleWhiteSpaceCollapse::Collapse as u32;
            // We can't match on enum values if we're matching on a u32
            const PRESERVE: u32 = structs::StyleWhiteSpaceCollapse::Preserve as u32;
            // We can't match on enum values if we're matching on a u32
            const PRESERVE_BREAKS: u32 = structs::StyleWhiteSpaceCollapse::PreserveBreaks as u32;
            // We can't match on enum values if we're matching on a u32
            const BREAK_SPACES: u32 = structs::StyleWhiteSpaceCollapse::BreakSpaces as u32;
            // We can't match on enum values if we're matching on a u32
            const PRESERVE_SPACES: u32 = structs::StyleWhiteSpaceCollapse::PreserveSpaces as u32;
            match kw {
                COLLAPSE => Self::Collapse,
                PRESERVE => Self::Preserve,
                PRESERVE_BREAKS => Self::PreserveBreaks,
                BREAK_SPACES => Self::BreakSpaces,
                PRESERVE_SPACES => Self::PreserveSpaces,
                _ => panic!("Found unexpected value in style struct for white-space-collapse property"),
            }
        }
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-text/#propdef-word-break
pub mod word_break {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::WordBreak,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_word_break();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::WordBreak>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_word_break(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::WordBreak)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::WordBreak as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::WordBreak as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::WordBreak::Normal }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::WordBreak as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-writing-modes/#propdef-writing-mode
pub mod writing_mode {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::WritingMode,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_writing_mode();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::WritingModeProperty>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_writing_mode(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::WritingMode)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::WritingModeProperty as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::WritingModeProperty as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::WritingModeProperty::HorizontalTb }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::WritingModeProperty as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Internal (not web-exposed)
pub mod _x_text_scale {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::XTextScale,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset__x_text_scale();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::XTextScale>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__x_text_scale(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::XTextScale)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::XTextScale as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::XTextScale as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::XTextScale::All }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::XTextScale as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://www.w3.org/TR/CSS2/visuren.html#z-index
pub mod z_index {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ZIndex,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_z_index();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ZIndex>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_z_index(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ZIndex)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ZIndex as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ZIndex as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::ZIndex::auto() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ZIndex as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Non-standard (https://github.com/atanassov/css-zoom/ is the closest)
pub mod zoom {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Zoom,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_zoom();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Zoom>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_zoom(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Zoom)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Zoom as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Zoom as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::box_::Zoom::ONE }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Zoom as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ui-4/#propdef-appearance
pub mod appearance {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Appearance,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_appearance();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Appearance>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_appearance(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Appearance)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Appearance as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Appearance as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Appearance::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Appearance as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Internal (not web-exposed)
pub mod _moz_default_appearance {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozDefaultAppearance,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_default_appearance();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Appearance>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_default_appearance(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozDefaultAppearance)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Appearance as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Appearance as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Appearance::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Appearance as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// None (Nonstandard Firefox-only property)
pub mod _moz_force_broken_image_icon {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozForceBrokenImageIcon,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_force_broken_image_icon();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BoolInteger>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_force_broken_image_icon(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozForceBrokenImageIcon)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BoolInteger as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BoolInteger as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BoolInteger::zero() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BoolInteger as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// None (Nonstandard internal property)
pub mod _moz_subtree_hidden_only_visually {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozSubtreeHiddenOnlyVisually,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_subtree_hidden_only_visually();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BoolInteger>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_subtree_hidden_only_visually(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozSubtreeHiddenOnlyVisually)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BoolInteger as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BoolInteger as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BoolInteger::zero() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BoolInteger as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-break/#propdef-break-after
pub mod break_after {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BreakAfter,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_break_after();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BreakBetween>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_break_after(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BreakAfter)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BreakBetween as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BreakBetween as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BreakBetween::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BreakBetween as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-break/#propdef-break-before
pub mod break_before {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BreakBefore,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_break_before();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BreakBetween>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_break_before(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BreakBefore)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BreakBetween as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BreakBetween as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BreakBetween::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BreakBetween as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-align/#propdef-align-content
pub mod align_content {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AlignContent,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_align_content();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ContentDistribution>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_align_content(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AlignContent)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ContentDistribution as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ContentDistribution as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::ContentDistribution::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        specified::ContentDistribution::parse_block(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-align/#propdef-justify-content
pub mod justify_content {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::JustifyContent,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_justify_content();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ContentDistribution>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_justify_content(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::JustifyContent)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ContentDistribution as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ContentDistribution as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::ContentDistribution::normal() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        specified::ContentDistribution::parse_inline(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/masking.html#ClipRuleProperty
pub mod clip_rule {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ClipRule,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_clip_rule();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FillRule>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_clip_rule(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ClipRule)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FillRule as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FillRule as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FillRule as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#FillRuleProperty
pub mod fill_rule {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FillRule,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_fill_rule();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FillRule>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_fill_rule(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FillRule)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FillRule as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FillRule as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FillRule as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#FillOpacity
pub mod fill_opacity {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FillOpacity,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_fill_opacity();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::SVGOpacity>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_fill_opacity(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FillOpacity)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::SVGOpacity as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::SVGOpacity as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::SVGOpacity as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#StrokeOpacity
pub mod stroke_opacity {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::StrokeOpacity,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_stroke_opacity();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::SVGOpacity>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_stroke_opacity(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::StrokeOpacity)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::SVGOpacity as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::SVGOpacity as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::SVGOpacity as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-align/#align-self-property
pub mod align_self {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AlignSelf,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_align_self();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::SelfAlignment>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_align_self(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AlignSelf)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::SelfAlignment as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::SelfAlignment as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::SelfAlignment::auto() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        specified::SelfAlignment::parse_block(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-align/#justify-self-property
pub mod justify_self {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::JustifySelf,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_justify_self();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::SelfAlignment>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_justify_self(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::JustifySelf)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::SelfAlignment as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::SelfAlignment as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::SelfAlignment::auto() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        specified::SelfAlignment::parse_inline(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts-4/#font-synthesis-position
pub mod font_synthesis_position {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontSynthesisPosition,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_synthesis_position();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontSynthesis>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_synthesis_position(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontSynthesisPosition)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontSynthesis as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontSynthesis as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSynthesis::Auto }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSynthesis::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontSynthesis as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts-4/#font-synthesis-small-caps
pub mod font_synthesis_small_caps {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontSynthesisSmallCaps,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_synthesis_small_caps();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontSynthesis>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_synthesis_small_caps(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontSynthesisSmallCaps)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontSynthesis as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontSynthesis as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSynthesis::Auto }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSynthesis::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontSynthesis as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-fonts-4/#font-synthesis-weight
pub mod font_synthesis_weight {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FontSynthesisWeight,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_font_synthesis_weight();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::FontSynthesis>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_font_synthesis_weight(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FontSynthesisWeight)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::FontSynthesis as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::FontSynthesis as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSynthesis::Auto }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSynthesis::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::FontSynthesis as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/-moz-box-ordinal-group)
pub mod _moz_box_ordinal_group {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozBoxOrdinalGroup,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_box_ordinal_group();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Integer>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_box_ordinal_group(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozBoxOrdinalGroup)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Integer as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Integer as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { 1 }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        specified::Integer::parse_non_negative(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-flexbox/#order-property
pub mod order {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Order,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_order();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Integer>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_order(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Order)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Integer as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Integer as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { 0 }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Integer as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Internal-only (for `<col span>` pres attr)
pub mod _x_span {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::XSpan,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__x_span();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Integer>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__x_span(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::XSpan)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Integer as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Integer as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { 1 }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Integer as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-flexbox/#flex-grow-property
pub mod flex_grow {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FlexGrow,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_flex_grow();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::NonNegativeNumber>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_flex_grow(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FlexGrow)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::NonNegativeNumber as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::NonNegativeNumber as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { From::from(0.0) }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::NonNegativeNumber as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-flexbox/#flex-shrink-property
pub mod flex_shrink {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FlexShrink,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_flex_shrink();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::NonNegativeNumber>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_flex_shrink(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FlexShrink)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::NonNegativeNumber as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::NonNegativeNumber as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { From::from(1.0) }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::NonNegativeNumber as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/box-flex)
pub mod _moz_box_flex {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozBoxFlex,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_box_flex();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::NonNegativeNumber>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_box_flex(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozBoxFlex)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::NonNegativeNumber as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::NonNegativeNumber as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { From::from(0.) }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::NonNegativeNumber as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/painting.html#StrokeMiterlimitProperty
pub mod stroke_miterlimit {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::StrokeMiterlimit,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_stroke_miterlimit();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::NonNegativeNumber>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_stroke_miterlimit(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::StrokeMiterlimit)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::NonNegativeNumber as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::NonNegativeNumber as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { From::from(4.0) }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::NonNegativeNumber as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-overflow-3/#propdef-overflow-block
pub mod overflow_block {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        declaration.debug_crash("Should physicalize before entering here");
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverflowBlock)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Overflow as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Overflow as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Overflow::Visible }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Overflow as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-overflow-3/#propdef-overflow-inline
pub mod overflow_inline {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        declaration.debug_crash("Should physicalize before entering here");
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverflowInline)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Overflow as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Overflow as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Overflow::Visible }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Overflow as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-overflow-3/#propdef-overflow-x
pub mod overflow_x {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap().logical_group(),
            LonghandId::OverflowX.logical_group(),
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_overflow_x();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Overflow>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_overflow_x(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverflowX)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Overflow as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Overflow as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Overflow::Visible }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Overflow as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-overflow-3/#propdef-overflow-y
pub mod overflow_y {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap().logical_group(),
            LonghandId::OverflowY.logical_group(),
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_overflow_y();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Overflow>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_overflow_y(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverflowY)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Overflow as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Overflow as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Overflow::Visible }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Overflow as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://wicg.github.io/overscroll-behavior/#overscroll-behavior-properties
pub mod overscroll_behavior_block {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        declaration.debug_crash("Should physicalize before entering here");
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverscrollBehaviorBlock)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::OverscrollBehavior as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::OverscrollBehavior as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverscrollBehavior::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::OverscrollBehavior as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://wicg.github.io/overscroll-behavior/#overscroll-behavior-properties
pub mod overscroll_behavior_inline {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        declaration.debug_crash("Should physicalize before entering here");
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverscrollBehaviorInline)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::OverscrollBehavior as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::OverscrollBehavior as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverscrollBehavior::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::OverscrollBehavior as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://wicg.github.io/overscroll-behavior/#overscroll-behavior-properties
pub mod overscroll_behavior_x {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap().logical_group(),
            LonghandId::OverscrollBehaviorX.logical_group(),
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_overscroll_behavior_x();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::OverscrollBehavior>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_overscroll_behavior_x(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverscrollBehaviorX)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::OverscrollBehavior as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::OverscrollBehavior as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverscrollBehavior::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::OverscrollBehavior as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://wicg.github.io/overscroll-behavior/#overscroll-behavior-properties
pub mod overscroll_behavior_y {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap().logical_group(),
            LonghandId::OverscrollBehaviorY.logical_group(),
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_overscroll_behavior_y();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::OverscrollBehavior>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_overscroll_behavior_y(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::OverscrollBehaviorY)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::OverscrollBehavior as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::OverscrollBehavior as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverscrollBehavior::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::OverscrollBehavior as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.fxtf.org/filter-effects-1/#FloodOpacityProperty
pub mod flood_opacity {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::FloodOpacity,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_flood_opacity();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Opacity>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_flood_opacity(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::FloodOpacity)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Opacity as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Opacity as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { 1.0 }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Opacity as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// None (Nonstandard internal property)
pub mod _moz_window_opacity {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::MozWindowOpacity,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit__moz_window_opacity();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Opacity>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set__moz_window_opacity(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::MozWindowOpacity)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Opacity as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Opacity as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { 1.0 }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Opacity as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-color/#transparency
pub mod opacity {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::Opacity,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_opacity();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Opacity>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_opacity(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::Opacity)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Opacity as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Opacity as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { 1.0 }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Opacity as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-shapes/#shape-image-threshold-property
pub mod shape_image_threshold {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ShapeImageThreshold,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_shape_image_threshold();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Opacity>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_shape_image_threshold(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ShapeImageThreshold)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Opacity as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Opacity as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { 0.0 }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Opacity as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://svgwg.org/svg2-draft/pservers.html#StopOpacityProperty
pub mod stop_opacity {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::StopOpacity,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_stop_opacity();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::Opacity>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_stop_opacity(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::StopOpacity)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Opacity as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Opacity as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { 1.0 }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Opacity as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-logical-props/#propdef-border-block-end-style
pub mod border_block_end_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        declaration.debug_crash("Should physicalize before entering here");
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderBlockEndStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-logical-props/#propdef-border-block-start-style
pub mod border_block_start_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        declaration.debug_crash("Should physicalize before entering here");
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderBlockStartStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-backgrounds/#border-bottom-style
pub mod border_bottom_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap().logical_group(),
            LonghandId::BorderBottomStyle.logical_group(),
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_border_bottom_style();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_border_bottom_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderBottomStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-logical-props/#propdef-border-inline-end-style
pub mod border_inline_end_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        declaration.debug_crash("Should physicalize before entering here");
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderInlineEndStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-logical-props/#propdef-border-inline-start-style
pub mod border_inline_start_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        declaration.debug_crash("Should physicalize before entering here");
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderInlineStartStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-backgrounds/#border-left-style
pub mod border_left_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap().logical_group(),
            LonghandId::BorderLeftStyle.logical_group(),
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_border_left_style();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_border_left_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderLeftStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-backgrounds/#border-right-style
pub mod border_right_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap().logical_group(),
            LonghandId::BorderRightStyle.logical_group(),
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_border_right_style();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_border_right_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderRightStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-backgrounds/#border-top-style
pub mod border_top_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap().logical_group(),
            LonghandId::BorderTopStyle.logical_group(),
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_border_top_style();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_border_top_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BorderTopStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-multicol/#propdef-column-rule-style
pub mod column_rule_style {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::ColumnRuleStyle,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_column_rule_style();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_column_rule_style(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::ColumnRuleStyle)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BorderStyle as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BorderStyle as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderStyle::None }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BorderStyle::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-ui-4/#widget-accent
pub mod accent_color {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = false;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AccentColor,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Initial => {
                            context.builder.reset_accent_color();
                    },
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Inherit => {
                            declaration.debug_crash("Unexpected inherit or unset for non-zoom-dependent inherited property");
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::ColorOrAuto>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_accent_color(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AccentColor)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::ColorOrAuto as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::ColorOrAuto as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { generics::color::ColorOrAuto::Auto }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::ColorOrAuto as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-anchor-position-1/#propdef-anchor-name
pub mod anchor_name {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnchorName,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_anchor_name();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::AnchorName>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_anchor_name(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnchorName)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnchorName as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnchorName as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnchorName::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnchorName as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-anchor-position-1/#propdef-scope
pub mod anchor_scope {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnchorScope,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_anchor_scope();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::AnchorScope>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_anchor_scope(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnchorScope)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnchorScope as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnchorScope as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnchorScope::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnchorScope as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-animations-2/#animation-composition
pub mod animation_composition {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationComposition,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_composition();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_composition::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_composition(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationComposition)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnimationComposition as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnimationComposition as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnimationComposition::Replace }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { computed::AnimationComposition::Replace }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnimationComposition as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-composition.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-composition.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-animations/#propdef-animation-delay
pub mod animation_delay {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationDelay,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_delay();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_delay::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_delay(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationDelay)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Time as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Time as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Time::zero() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Time::zero() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Time as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-delay.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-delay.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-animations/#propdef-animation-direction
pub mod animation_direction {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationDirection,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_direction();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_direction::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_direction(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationDirection)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnimationDirection as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnimationDirection as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnimationDirection::Normal }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::AnimationDirection::Normal }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnimationDirection as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-direction.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-direction.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-animations-2/#animation-duration
pub mod animation_duration {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationDuration,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_duration();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_duration::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_duration(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationDuration)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnimationDuration as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnimationDuration as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnimationDuration::auto() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::AnimationDuration::auto() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnimationDuration as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-duration.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-duration.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-animations/#propdef-animation-fill-mode
pub mod animation_fill_mode {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationFillMode,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_fill_mode();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_fill_mode::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_fill_mode(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationFillMode)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnimationFillMode as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnimationFillMode as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnimationFillMode::None }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { computed::AnimationFillMode::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnimationFillMode as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-fill-mode.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-fill-mode.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-animations/#propdef-animation-iteration-count
pub mod animation_iteration_count {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationIterationCount,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_iteration_count();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_iteration_count::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_iteration_count(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationIterationCount)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnimationIterationCount as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnimationIterationCount as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnimationIterationCount::one() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::AnimationIterationCount::one() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnimationIterationCount as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-iteration-count.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-iteration-count.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-animations/#propdef-animation-name
pub mod animation_name {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationName,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_name();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_name::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_name(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationName)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnimationName as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnimationName as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnimationName::none() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::AnimationName::none() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnimationName as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-name.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-name.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-animations/#propdef-animation-play-state
pub mod animation_play_state {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationPlayState,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_play_state();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_play_state::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_play_state(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationPlayState)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnimationPlayState as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnimationPlayState as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnimationPlayState::Running }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { computed::AnimationPlayState::Running }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnimationPlayState as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-play-state.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-play-state.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-animations-2/#propdef-animation-timeline
pub mod animation_timeline {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationTimeline,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_timeline();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_timeline::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_timeline(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationTimeline)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::AnimationTimeline as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::AnimationTimeline as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnimationTimeline::auto() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::AnimationTimeline::auto() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::AnimationTimeline as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-timeline.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-timeline.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-transitions/#propdef-animation-timing-function
pub mod animation_timing_function {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::AnimationTimingFunction,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_animation_timing_function();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::animation_timing_function::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_ui();
            {
                let iter = specified_value.compute_iter(context);
                s.set_animation_timing_function(iter);
            }
            context.builder.put_ui(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::AnimationTimingFunction)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::TimingFunction as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::TimingFunction as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::TimingFunction::ease() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::TimingFunction::ease() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::TimingFunction as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for animation-timing-function.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of animation-timing-function.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.fxtf.org/filter-effects-2/#propdef-backdrop-filter
pub mod backdrop_filter {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackdropFilter,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_backdrop_filter();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::backdrop_filter::SpecifiedValue>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_backdrop_filter(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackdropFilter)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Filter as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Filter as T;
    }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Filter as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for backdrop-filter.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                crate::OwnedSlice<T>;

        pub type UnderlyingOwnedList<T> =
                crate::OwnedSlice<T>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        pub struct OwnedList<T>(
            #[css(if_empty = "none", iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;

        impl From<ComputedList> for UnderlyingList<single_value::T> {
            #[inline]
            fn from(l: ComputedList) -> Self {
                l.0
            }
        }
        impl From<UnderlyingList<single_value::T>> for ComputedList {
            #[inline]
            fn from(l: UnderlyingList<single_value::T>) -> Self {
                List(l)
            }
        }

        use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
        use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

        // FIXME(emilio): For some reason rust thinks that this alias is
        // unused, even though it's clearly used below?
        #[allow(unused)]
        type AnimatedList = <OwnedList<single_value::T> as ToAnimatedValue>::AnimatedValue;

        impl ToAnimatedZero for AnimatedList {
            fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
        }

        impl Animate for AnimatedList {
            fn animate(
                &self,
                other: &Self,
                procedure: Procedure,
            ) -> Result<Self, ()> {
                Ok(OwnedList(
                    lists::with_zero::animate(&self.0, &other.0, procedure)?
                ))
            }
        }
        impl ComputeSquaredDistance for AnimatedList {
            fn compute_squared_distance(
                &self,
                other: &Self,
            ) -> Result<SquaredDistance, ()> {
                lists::with_zero::squared_distance(&self.0, &other.0)
            }
        }

        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of backdrop-filter.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    pub struct SpecifiedValue(
        #[css(if_empty = "none", iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            computed_value::List(Default::default())
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;

        if input.try_parse(|input| input.expect_ident_matching("none")).is_ok() {
            return Ok(SpecifiedValue(Default::default()))
        }

        let v = style_traits::Space::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;


    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-backgrounds/#the-background-attachment
pub mod background_attachment {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackgroundAttachment,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_background_attachment();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::background_attachment::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_background();
            {
                let iter = specified_value.compute_iter(context);
                s.set_background_attachment(iter);
            }
            context.builder.put_background(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackgroundAttachment)
    }

    pub mod single_value {
        use super::*;
    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Scroll,
        
        Fixed,
        
        Local,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Scroll
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Scroll
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const SCROLL: u32 = structs::StyleImageLayerAttachment::Scroll as u32;
            // We can't match on enum values if we're matching on a u32
            const FIXED: u32 = structs::StyleImageLayerAttachment::Fixed as u32;
            // We can't match on enum values if we're matching on a u32
            const LOCAL: u32 = structs::StyleImageLayerAttachment::Local as u32;
            match kw {
                SCROLL => Self::Scroll,
                FIXED => Self::Fixed,
                LOCAL => Self::Local,
                _ => panic!("Found unexpected value in style struct for background-attachment property"),
            }
        }
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for background-attachment.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of background-attachment.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.fxtf.org/compositing/#background-blend-mode
pub mod background_blend_mode {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackgroundBlendMode,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_background_blend_mode();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::background_blend_mode::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_background();
            {
                let iter = specified_value.compute_iter(context);
                s.set_background_blend_mode(iter);
            }
            context.builder.put_background(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackgroundBlendMode)
    }

    pub mod single_value {
        use super::*;
    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        Normal,
        
        Multiply,
        
        Screen,
        
        Overlay,
        
        Darken,
        
        Lighten,
        
        ColorDodge,
        
        ColorBurn,
        
        HardLight,
        
        SoftLight,
        
        Difference,
        
        Exclusion,
        
        Hue,
        
        Saturation,
        
        Color,
        
        Luminosity,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::Normal
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::Normal
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const NORMAL: u32 = structs::StyleBlend::Normal as u32;
            // We can't match on enum values if we're matching on a u32
            const MULTIPLY: u32 = structs::StyleBlend::Multiply as u32;
            // We can't match on enum values if we're matching on a u32
            const SCREEN: u32 = structs::StyleBlend::Screen as u32;
            // We can't match on enum values if we're matching on a u32
            const OVERLAY: u32 = structs::StyleBlend::Overlay as u32;
            // We can't match on enum values if we're matching on a u32
            const DARKEN: u32 = structs::StyleBlend::Darken as u32;
            // We can't match on enum values if we're matching on a u32
            const LIGHTEN: u32 = structs::StyleBlend::Lighten as u32;
            // We can't match on enum values if we're matching on a u32
            const COLOR_DODGE: u32 = structs::StyleBlend::ColorDodge as u32;
            // We can't match on enum values if we're matching on a u32
            const COLOR_BURN: u32 = structs::StyleBlend::ColorBurn as u32;
            // We can't match on enum values if we're matching on a u32
            const HARD_LIGHT: u32 = structs::StyleBlend::HardLight as u32;
            // We can't match on enum values if we're matching on a u32
            const SOFT_LIGHT: u32 = structs::StyleBlend::SoftLight as u32;
            // We can't match on enum values if we're matching on a u32
            const DIFFERENCE: u32 = structs::StyleBlend::Difference as u32;
            // We can't match on enum values if we're matching on a u32
            const EXCLUSION: u32 = structs::StyleBlend::Exclusion as u32;
            // We can't match on enum values if we're matching on a u32
            const HUE: u32 = structs::StyleBlend::Hue as u32;
            // We can't match on enum values if we're matching on a u32
            const SATURATION: u32 = structs::StyleBlend::Saturation as u32;
            // We can't match on enum values if we're matching on a u32
            const COLOR: u32 = structs::StyleBlend::Color as u32;
            // We can't match on enum values if we're matching on a u32
            const LUMINOSITY: u32 = structs::StyleBlend::Luminosity as u32;
            match kw {
                NORMAL => Self::Normal,
                MULTIPLY => Self::Multiply,
                SCREEN => Self::Screen,
                OVERLAY => Self::Overlay,
                DARKEN => Self::Darken,
                LIGHTEN => Self::Lighten,
                COLOR_DODGE => Self::ColorDodge,
                COLOR_BURN => Self::ColorBurn,
                HARD_LIGHT => Self::HardLight,
                SOFT_LIGHT => Self::SoftLight,
                DIFFERENCE => Self::Difference,
                EXCLUSION => Self::Exclusion,
                HUE => Self::Hue,
                SATURATION => Self::Saturation,
                COLOR => Self::Color,
                LUMINOSITY => Self::Luminosity,
                _ => panic!("Found unexpected value in style struct for background-blend-mode property"),
            }
        }
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for background-blend-mode.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of background-blend-mode.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-backgrounds/#the-background-clip
pub mod background_clip {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackgroundClip,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_background_clip();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::background_clip::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_background();
            {
                let iter = specified_value.compute_iter(context);
                s.set_background_clip(iter);
            }
            context.builder.put_background(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackgroundClip)
    }

    pub mod single_value {
        use super::*;
    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        BorderBox,
        
        PaddingBox,
        
        ContentBox,
        
        Text,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::BorderBox
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::BorderBox
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const BORDER_BOX: u32 = structs::StyleGeometryBox::BorderBox as u32;
            // We can't match on enum values if we're matching on a u32
            const PADDING_BOX: u32 = structs::StyleGeometryBox::PaddingBox as u32;
            // We can't match on enum values if we're matching on a u32
            const CONTENT_BOX: u32 = structs::StyleGeometryBox::ContentBox as u32;
            // We can't match on enum values if we're matching on a u32
            const TEXT: u32 = structs::StyleGeometryBox::Text as u32;
            match kw {
                BORDER_BOX => Self::BorderBox,
                PADDING_BOX => Self::PaddingBox,
                CONTENT_BOX => Self::ContentBox,
                TEXT => Self::Text,
                _ => panic!("Found unexpected value in style struct for background-clip property"),
            }
        }
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for background-clip.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of background-clip.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-backgrounds/#the-background-image
pub mod background_image {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackgroundImage,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_background_image();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::background_image::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_background();
            {
                let iter = specified_value.compute_iter(context);
                s.set_background_image(iter);
            }
            context.builder.put_background(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackgroundImage)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::Image as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::Image as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::Image::None }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Image::None }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::Image as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for background-image.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of background-image.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-backgrounds/#the-background-origin
pub mod background_origin {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackgroundOrigin,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_background_origin();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::background_origin::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_background();
            {
                let iter = specified_value.compute_iter(context);
                s.set_background_origin(iter);
            }
            context.builder.put_background(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackgroundOrigin)
    }

    pub mod single_value {
        use super::*;
    pub use self::computed_value::T as SpecifiedValue;
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::derives::*;
        #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
        #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem, ToTyped)]
        pub enum T {
        
        PaddingBox,
        
        BorderBox,
        
        ContentBox,
        }
    }
    #[inline]
    pub fn get_initial_value() -> computed_value::T {
        computed_value::T::PaddingBox
    }
    #[inline]
    pub fn get_initial_specified_value() -> SpecifiedValue {
        SpecifiedValue::PaddingBox
    }
    #[inline]
    pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                         -> Result<SpecifiedValue, ParseError<'i>> {
        SpecifiedValue::parse(input)
    }

    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
            // We can't match on enum values if we're matching on a u32
            const PADDING_BOX: u32 = structs::StyleGeometryBox::PaddingBox as u32;
            // We can't match on enum values if we're matching on a u32
            const BORDER_BOX: u32 = structs::StyleGeometryBox::BorderBox as u32;
            // We can't match on enum values if we're matching on a u32
            const CONTENT_BOX: u32 = structs::StyleGeometryBox::ContentBox as u32;
            match kw {
                PADDING_BOX => Self::PaddingBox,
                BORDER_BOX => Self::BorderBox,
                CONTENT_BOX => Self::ContentBox,
                _ => panic!("Found unexpected value in style struct for background-origin property"),
            }
        }
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for background-origin.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of background-origin.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-backgrounds-4/#propdef-background-position-x
pub mod background_position_x {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackgroundPositionX,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_background_position_x();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::background_position_x::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_background();
            {
                let iter = specified_value.compute_iter(context);
                s.set_background_position_x(iter);
            }
            context.builder.put_background(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackgroundPositionX)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::position::HorizontalPosition as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::position::HorizontalPosition as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::LengthPercentage::zero_percent() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { SpecifiedValue::initial_specified_value() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::position::HorizontalPosition as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for background-position-x.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;


        use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
        use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

        // FIXME(emilio): For some reason rust thinks that this alias is
        // unused, even though it's clearly used below?
        #[allow(unused)]
        type AnimatedList = <OwnedList<single_value::T> as ToAnimatedValue>::AnimatedValue;

        impl ToAnimatedZero for AnimatedList {
            fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
        }

        impl Animate for AnimatedList {
            fn animate(
                &self,
                other: &Self,
                procedure: Procedure,
            ) -> Result<Self, ()> {
                Ok(OwnedList(
                    lists::repeatable_list::animate(&self.0, &other.0, procedure)?
                ))
            }
        }
        impl ComputeSquaredDistance for AnimatedList {
            fn compute_squared_distance(
                &self,
                other: &Self,
            ) -> Result<SquaredDistance, ()> {
                lists::repeatable_list::squared_distance(&self.0, &other.0)
            }
        }

        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of background-position-x.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-backgrounds-4/#propdef-background-position-y
pub mod background_position_y {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackgroundPositionY,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_background_position_y();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::background_position_y::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_background();
            {
                let iter = specified_value.compute_iter(context);
                s.set_background_position_y(iter);
            }
            context.builder.put_background(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackgroundPositionY)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::position::VerticalPosition as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::position::VerticalPosition as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::LengthPercentage::zero_percent() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { SpecifiedValue::initial_specified_value() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::position::VerticalPosition as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for background-position-y.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;


        use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
        use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

        // FIXME(emilio): For some reason rust thinks that this alias is
        // unused, even though it's clearly used below?
        #[allow(unused)]
        type AnimatedList = <OwnedList<single_value::T> as ToAnimatedValue>::AnimatedValue;

        impl ToAnimatedZero for AnimatedList {
            fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
        }

        impl Animate for AnimatedList {
            fn animate(
                &self,
                other: &Self,
                procedure: Procedure,
            ) -> Result<Self, ()> {
                Ok(OwnedList(
                    lists::repeatable_list::animate(&self.0, &other.0, procedure)?
                ))
            }
        }
        impl ComputeSquaredDistance for AnimatedList {
            fn compute_squared_distance(
                &self,
                other: &Self,
            ) -> Result<SquaredDistance, ()> {
                lists::repeatable_list::squared_distance(&self.0, &other.0)
            }
        }

        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of background-position-y.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-backgrounds/#the-background-repeat
pub mod background_repeat {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackgroundRepeat,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_background_repeat();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::background_repeat::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_background();
            {
                let iter = specified_value.compute_iter(context);
                s.set_background_repeat(iter);
            }
            context.builder.put_background(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackgroundRepeat)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BackgroundRepeat as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BackgroundRepeat as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BackgroundRepeat::repeat() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BackgroundRepeat::repeat() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BackgroundRepeat as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for background-repeat.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;



        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of background-repeat.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-backgrounds/#the-background-size
pub mod background_size {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BackgroundSize,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_background_size();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<longhands::background_size::SpecifiedValue>()
            },
        };


            // In the case of a vector property we want to pass down an
            // iterator so that this can be computed without allocation.
            //
            // However, computing requires a context, but the style struct
            // being mutated is on the context. We temporarily remove it,
            // mutate it, and then put it back. Vector longhands cannot
            // touch their own style struct whilst computing, else this will
            // panic.
            let mut s =
                context.builder.take_background();
            {
                let iter = specified_value.compute_iter(context);
                s.set_background_size(iter);
            }
            context.builder.put_background(s);
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BackgroundSize)
    }

    pub mod single_value {
        use super::*;
    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BackgroundSize as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BackgroundSize as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BackgroundSize::auto() }
    #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BackgroundSize::auto() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BackgroundSize as crate::parser::Parse>::parse(context, input)
    }
    } // single_value

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
    
    #[allow(unused_imports)]
    use smallvec::SmallVec;

    /// The definition of the computed value for background-size.
    pub mod computed_value {
        #[allow(unused_imports)]
        use crate::values::animated::ToAnimatedValue;
        #[allow(unused_imports)]
        use crate::values::resolved::ToResolvedValue;
        #[allow(unused_imports)]
        use crate::derives::*;
        pub use super::single_value::computed_value as single_value;
        pub use self::single_value::T as SingleComputedValue;
        use smallvec::SmallVec;
        use crate::values::computed::ComputedVecIter;

        

        // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
        // something for transition-name, which is the only remaining user
        // of NotInitial.
        pub type UnderlyingList<T> =
                SmallVec<[T; 1]>;

        pub type UnderlyingOwnedList<T> =
                SmallVec<[T; 1]>;


        /// The generic type defining the animated and resolved values for
        /// this property.
        ///
        /// Making this type generic allows the compiler to figure out the
        /// animated value for us, instead of having to implement it
        /// manually for every type we care about.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss, ToTyped)]
        #[css(comma)]
        pub struct OwnedList<T>(
            #[css(iterable)]
            pub UnderlyingOwnedList<T>,
        );

        /// The computed value for this property.
        pub type ComputedList = OwnedList<single_value::T>;
        pub use self::OwnedList as List;


        use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
        use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

        // FIXME(emilio): For some reason rust thinks that this alias is
        // unused, even though it's clearly used below?
        #[allow(unused)]
        type AnimatedList = <OwnedList<single_value::T> as ToAnimatedValue>::AnimatedValue;

        impl ToAnimatedZero for AnimatedList {
            fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
        }

        impl Animate for AnimatedList {
            fn animate(
                &self,
                other: &Self,
                procedure: Procedure,
            ) -> Result<Self, ()> {
                Ok(OwnedList(
                    lists::repeatable_list::animate(&self.0, &other.0, procedure)?
                ))
            }
        }
        impl ComputeSquaredDistance for AnimatedList {
            fn compute_squared_distance(
                &self,
                other: &Self,
            ) -> Result<SquaredDistance, ()> {
                lists::repeatable_list::squared_distance(&self.0, &other.0)
            }
        }

        /// The computed value, effectively a list of single values.
        pub use self::ComputedList as T;

        pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
    }

    /// The specified value of background-size.
    #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem, ToTyped)]
    #[css(comma)]
    pub struct SpecifiedValue(
        #[css(iterable)]
        pub crate::OwnedSlice<single_value::SpecifiedValue>,
    );

    pub fn get_initial_value() -> computed_value::T {
            let mut v = SmallVec::new();
            v.push(single_value::get_initial_value());
            computed_value::List(v)
    }

    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        use style_traits::Separator;


        let v = style_traits::Comma::parse(input, |parser| {
            single_value::parse(context, parser)
        })?;
        Ok(SpecifiedValue(v.into()))
    }

    pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

    impl SpecifiedValue {
        fn compute_iter<'a, 'cx, 'cx_a>(
            &'a self,
            context: &'cx computed::Context<'cx_a>,
        ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
            computed_value::Iter::new(context, &self.0)
        }
    }

    impl ToComputedValue for SpecifiedValue {
        type ComputedValue = computed_value::T;

        #[inline]
        fn to_computed_value(&self, context: &computed::Context) -> computed_value::T {
            use std::iter::FromIterator;
            computed_value::List(computed_value::UnderlyingList::from_iter(
                self.0.iter().map(|i| i.to_computed_value(context))
            ))
        }

        #[inline]
        fn from_computed_value(computed: &computed_value::T) -> Self {
            let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
            SpecifiedValue(iter.collect())
        }
    }
}

/// https://drafts.csswg.org/css-inline-3/#baseline-shift
pub mod baseline_shift {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BaselineShift,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            context.builder.inherit_baseline_shift();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<crate::values::specified::BaselineShift>()
            },
        };


            let computed = specified_value.to_computed_value(context);
            context.builder.set_baseline_shift(computed)
    }

    pub fn parse_declared<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<PropertyDeclaration, ParseError<'i>> {
        parse(context, input)
            .map(PropertyDeclaration::BaselineShift)
    }

    #[allow(unused_imports)]
    use app_units::Au;
    #[allow(unused_imports)]
    use crate::values::specified::AllowQuirks;
    #[allow(unused_imports)]
    use crate::Zero;
    #[allow(unused_imports)]
    use smallvec::SmallVec;
    pub use crate::values::specified::BaselineShift as SpecifiedValue;
    pub mod computed_value {
        pub use crate::values::computed::BaselineShift as T;
    }
    #[inline] pub fn get_initial_value() -> computed_value::T { computed::BaselineShift::zero() }
    #[allow(unused_variables)]
    #[inline]
    pub fn parse<'i, 't>(
        context: &ParserContext,
        input: &mut Parser<'i, 't>,
    ) -> Result<SpecifiedValue, ParseError<'i>> {
        <specified::BaselineShift as crate::parser::Parse>::parse(context, input)
    }

    // The setup here is roughly:
    //
    //  * UnderlyingList is the list that is stored in the computed value. This may
    //    be a shared ArcSlice if the property is inherited.
    //  * UnderlyingOwnedList is the list that is used for animation.
    //  * Specified values always use OwnedSlice, since it's more compact.
    //  * computed_value::List is just a convenient alias that you can use for the
    //    computed value list, since this is in the computed_value module.
    //
    // If vector.simple_bindings is true, then we don't use the complex iterator
    // machinery and set_foo_from, and just compute the value like any other
    // longhand.
}

/// https://drafts.csswg.org/css-backgrounds/#border-image-outset
pub mod border_image_outset {
    #[allow(unused_imports)]
    use crate::derives::*;
    #[allow(unused_imports)]
    use crate::values::{computed, generics, specified};
    #[allow(unused_imports)]
    use crate::values::computed::ToComputedValue;
    #[allow(unused_imports)]
    use servo_arc::Arc;
    use cssparser::Parser;
    #[allow(unused_imports)]
    use crate::parser::{Parse, ParserContext};
    use style_traits::ParseError;
    #[allow(unused_imports)]
    use crate::properties::{longhands, LonghandId, CSSWideKeyword, PropertyDeclaration};

    #[allow(unused_variables)]
    pub unsafe fn cascade_property(
        declaration: &PropertyDeclaration,
        context: &mut computed::Context,
    ) {
        context.for_non_inherited_property = true;
        debug_assert_eq!(
            declaration.id().as_longhand().unwrap(),
            LonghandId::BorderImageOutset,
        );
        let specified_value = match *declaration {
            PropertyDeclaration::CSSWideKeyword(ref wk) => {
                match wk.keyword {
                    CSSWideKeyword::Unset |
                    CSSWideKeyword::Initial => {
                            declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                    },
                    CSSWideKeyword::Inherit => {
                            context.rule_cache_conditions.borrow_mut().set_uncacheable();
                            if !context.builder.effective_zoom_for_inheritance.is_one() {
                                let old_zoom = context.builder.effective_zoom;
                                context.builder.effective_zoom = context.builder.effective_zoom_for_inheritance;
                                let computed = context.builder.inherited_style.clone_border_image_outset();
                                let specified = computed::ToComputedValue::from_computed_value(&computed);
                                let specified = Box::new(specified);
                                let decl = PropertyDeclaration::BorderImageOutset(specified);
                                cascade_property(&decl, context);
                                context.builder.effective_zoom = old_zoom;
                                return;
                            }
                            context.builder.inherit_border_image_outset();
                    }
                    CSSWideKeyword::RevertLayer |
                    CSSWideKeyword::Revert => {
                        declaration.debug_crash("Found revert/revert-layer not dealt with");
                    },
                }
                return;
            },
            #[cfg(debug_assertions)]
            PropertyDeclaration::WithVariables(..) => {
                declaration.debug_crash("Found variables not substituted");
                return;
            },
            _ => unsafe {
                declaration.unchecked_value_as::<Box<crate::values::specified::NonNegativeLengthOrNumberRect>>()
            },
        };


            let computed = (**specified_value).to_computed_value(context