/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { debugLogger } from '@google/gemini-cli-core';
import clipboardy from 'clipboardy';
/**
 * Checks if a query string potentially represents an '@' command.
 * It triggers if the query starts with '@' or contains '@' preceded by whitespace
 * and followed by a non-whitespace character.
 *
 * @param query The input query string.
 * @returns True if the query looks like an '@' command, false otherwise.
 */
export const isAtCommand = (query) => 
// Check if starts with @ OR has a space, then @
query.startsWith('@') || /\s@/.test(query);
/**
 * Checks if a query string potentially represents an '/' command.
 * It triggers if the query starts with '/' but excludes code comments like '//' and '/*'.
 *
 * @param query The input query string.
 * @returns True if the query looks like an '/' command, false otherwise.
 */
export const isSlashCommand = (query) => {
    if (!query.startsWith('/')) {
        return false;
    }
    // Exclude line comments that start with '//'
    if (query.startsWith('//')) {
        return false;
    }
    // Exclude block comments that start with '/*'
    if (query.startsWith('/*')) {
        return false;
    }
    return true;
};
// Copies a string snippet to the clipboard
export const copyToClipboard = async (text) => {
    await clipboardy.write(text);
};
export const getUrlOpenCommand = () => {
    // --- Determine the OS-specific command to open URLs ---
    let openCmd;
    switch (process.platform) {
        case 'darwin':
            openCmd = 'open';
            break;
        case 'win32':
            openCmd = 'start';
            break;
        case 'linux':
            openCmd = 'xdg-open';
            break;
        default:
            // Default to xdg-open, which appears to be supported for the less popular operating systems.
            openCmd = 'xdg-open';
            debugLogger.warn(`Unknown platform: ${process.platform}. Attempting to open URLs with: ${openCmd}.`);
            break;
    }
    return openCmd;
};
//# sourceMappingURL=commandUtils.js.map