// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

// clang-format off
#include "shim.bwd_kernel_fuse.h"
#include <aotriton/util.h>
#include <tuple>
#include "iface.op_attn_bwd.h"

namespace AOTRITON_NS::v3::flash {

#if 1
using AOTRITON_NS::v3::flash::OpAttnBwdParams;
#endif

#define CAST(x) const_cast<void*>(static_cast<const void*>(x))
typedef std::vector<void*>(*PP_FUNC)(const OpAttnBwdParams& context, hipDeviceptr_t*);

namespace {
extern PP_FUNC prepare_arguments[ 6 ];
}

int64_t BwdKernelFuseContext::godel_number() const
{
    int64_t sum = 0;
    const auto& args = *params;
    {
        int64_t number = 0;
        if (args.Q->dtype() == DType::kFloat16) number = 0 ;
        if (args.Q->dtype() == DType::kBFloat16) number = 1 ;
        if (args.Q->dtype() == DType::kFloat32) number = 2 ;
        sum += number * 176;
    }
    {
        int64_t number = 0;
        if (args.BLOCK_DMODEL == 16) number = 0 ;
        if (args.BLOCK_DMODEL == 32) number = 1 ;
        if (args.BLOCK_DMODEL == 48) number = 2 ;
        if (args.BLOCK_DMODEL == 64) number = 3 ;
        if (args.BLOCK_DMODEL == 80) number = 4 ;
        if (args.BLOCK_DMODEL == 96) number = 5 ;
        if (args.BLOCK_DMODEL == 128) number = 6 ;
        if (args.BLOCK_DMODEL == 160) number = 7 ;
        if (args.BLOCK_DMODEL == 192) number = 8 ;
        if (args.BLOCK_DMODEL == 224) number = 9 ;
        if (args.BLOCK_DMODEL == 256) number = 10 ;
        sum += number * 16;
    }
    {
        int64_t number = 0;
        if (args.CAUSAL_TYPE == 0) number = 0 ;
        if (args.CAUSAL_TYPE == 3) number = 1 ;
        sum += number * 8;
    }
    {
        int64_t number = 0;
        if (args.ENABLE_DROPOUT == false) number = 0 ;
        if (args.ENABLE_DROPOUT == true) number = 1 ;
        sum += number * 4;
    }
    {
        int64_t number = 0;
        if (args.PADDED_HEAD == false) number = 0 ;
        if (args.PADDED_HEAD == true) number = 1 ;
        sum += number * 2;
    }
    {
        int64_t number = 0;
        if (args.BIAS_TYPE == 0) number = 0 ;
        if (args.BIAS_TYPE == 1) number = 1 ;
        sum += number * 1;
    }

    return sum;
}

hipError_t
BwdKernelFuseContext::lookup_optimal(Gpu gpu) {
    auto [arch_number, mod_number] = get_archmod_number(gpu);
    if (arch_number < 0) {
        return hipErrorNoBinaryForGpu;
    }
    kernel_on_device = nullptr;
    auto tune_func = autotune_table[arch_number][godel_number()];
    if (!tune_func)
        return hipErrorProfilerNotInitialized;
    tune_func(*this, mod_number);
    if (!kernel_on_device)
        return hipErrorSharedObjectSymbolNotFound;
    return hipSuccess;
}

hipError_t
BwdKernelFuseContext::launch(hipStream_t stream) const {
    constexpr std::string_view triton_kernel_name { "bwd_kernel_fuse" };
    hipDeviceptr_t global_scratch = 0;
    auto args = prepare_arguments[pp_args_index](*this->params, &global_scratch);
    dim3 grid;
    if (custom_grid_calculator) {
        grid = custom_grid_calculator(*this);
    } else {
        grid = grid_calculator();
    }
#if AOTRITON_BUILD_FOR_TUNING
    return kernel_on_device->invoke(triton_kernel_name,
                                    package_path,
                                    func_name,
                                    arch_name,
                                    grid,
                                    args,
                                    peek_kernel_image,
                                    stream);
#else
    return kernel_on_device->invoke(triton_kernel_name,
                                    package_path,
                                    func_name,
                                    arch_name,
                                    grid,
                                    args,
                                    stream);
#endif
}

std::tuple<int, int>
BwdKernelFuseContext::get_archmod_number(Gpu gpu) {
    if (gpu == GPU_AMD_ARCH_GFX950_MOD0) return { 0, 0 };
    if (gpu == GPU_AMD_ARCH_GFX1151_MOD0) return { 1, 0 };
    if (gpu == GPU_AMD_ARCH_GFX1150_MOD0) return { 2, 0 };
    if (gpu == GPU_AMD_ARCH_GFX1201_MOD0) return { 3, 0 };
    if (gpu == GPU_AMD_ARCH_GFX1200_MOD0) return { 4, 0 };
    // TODO: print warning about tuning for this GPU mod is not built.
    // Note: if some mod does not have tuning info in the database at all, the
    //       getGpuFromStream should not return that mod from beginning.
    return std::make_tuple(-1, 0);
}


static std::vector<void*>
bwd_kernel_fuse_pp_args_0(const OpAttnBwdParams& params,
                          hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           CAST(&params.sm_scale), // sm_scale
           params.Out->kparam_data_ptr(), // Out
           params.DO->kparam_data_ptr(), // DO
           params.DK->kparam_data_ptr(), // DK
           params.DV->kparam_data_ptr(), // DV
           params.DQ->kparam_data_ptr(), // DQ
           params.DB->kparam_data_ptr(), // DB
           params.L->kparam_data_ptr(), // L
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bk as constexpr 0
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.DO->kparam_stride(0), // stride_doz
           params.DO->kparam_stride(1), // stride_doh
           params.DO->kparam_stride(2), // stride_dom
           params.DK->kparam_stride(0), // stride_dkz
           params.DK->kparam_stride(1), // stride_dkh
           params.DK->kparam_stride(2), // stride_dkn
           params.DV->kparam_stride(0), // stride_dvz
           params.DV->kparam_stride(1), // stride_dvh
           params.DV->kparam_stride(2), // stride_dvk
           params.DQ->kparam_stride(0), // stride_dqz
           params.DQ->kparam_stride(1), // stride_dqh
           params.DQ->kparam_stride(2), // stride_dqm
           // params.DB->kparam_stride(0), // stride_dbz as constexpr 0
           // params.DB->kparam_stride(1), // stride_dbh as constexpr 0
           // params.DB->kparam_stride(2), // stride_dbm as constexpr 0
           CAST(&params.num_head_q), // num_head_q
           CAST(&params.num_head_k), // num_head_k
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.num_seqlens), // num_seqlens
           CAST(&params.max_seqlen_q), // max_seqlen_q
           CAST(&params.max_seqlen_k), // max_seqlen_k
           CAST(&params.head_dim), // head_dim
           // CAST(&params.dropout_p), // dropout_p as constexpr 0
           // params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr as constexpr 0
           // params.philox_offset1->kparam_data_ptr(), // philox_offset1 as constexpr 0
           // CAST(&params.philox_offset2), // philox_offset2 as constexpr 0
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           CAST(global_scratch)
         };
}
static std::vector<void*>
bwd_kernel_fuse_pp_args_1(const OpAttnBwdParams& params,
                          hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           CAST(&params.sm_scale), // sm_scale
           params.Out->kparam_data_ptr(), // Out
           params.DO->kparam_data_ptr(), // DO
           params.DK->kparam_data_ptr(), // DK
           params.DV->kparam_data_ptr(), // DV
           params.DQ->kparam_data_ptr(), // DQ
           params.DB->kparam_data_ptr(), // DB
           params.L->kparam_data_ptr(), // L
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.B->kparam_stride(0), // stride_bz
           params.B->kparam_stride(1), // stride_bh
           params.B->kparam_stride(2), // stride_bk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.DO->kparam_stride(0), // stride_doz
           params.DO->kparam_stride(1), // stride_doh
           params.DO->kparam_stride(2), // stride_dom
           params.DK->kparam_stride(0), // stride_dkz
           params.DK->kparam_stride(1), // stride_dkh
           params.DK->kparam_stride(2), // stride_dkn
           params.DV->kparam_stride(0), // stride_dvz
           params.DV->kparam_stride(1), // stride_dvh
           params.DV->kparam_stride(2), // stride_dvk
           params.DQ->kparam_stride(0), // stride_dqz
           params.DQ->kparam_stride(1), // stride_dqh
           params.DQ->kparam_stride(2), // stride_dqm
           params.DB->kparam_stride(0), // stride_dbz
           params.DB->kparam_stride(1), // stride_dbh
           params.DB->kparam_stride(2), // stride_dbm
           CAST(&params.num_head_q), // num_head_q
           CAST(&params.num_head_k), // num_head_k
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.num_seqlens), // num_seqlens
           CAST(&params.max_seqlen_q), // max_seqlen_q
           CAST(&params.max_seqlen_k), // max_seqlen_k
           CAST(&params.head_dim), // head_dim
           // CAST(&params.dropout_p), // dropout_p as constexpr 0
           // params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr as constexpr 0
           // params.philox_offset1->kparam_data_ptr(), // philox_offset1 as constexpr 0
           // CAST(&params.philox_offset2), // philox_offset2 as constexpr 0
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           CAST(global_scratch)
         };
}
static std::vector<void*>
bwd_kernel_fuse_pp_args_2(const OpAttnBwdParams& params,
                          hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           CAST(&params.sm_scale), // sm_scale
           params.Out->kparam_data_ptr(), // Out
           params.DO->kparam_data_ptr(), // DO
           params.DK->kparam_data_ptr(), // DK
           params.DV->kparam_data_ptr(), // DV
           params.DQ->kparam_data_ptr(), // DQ
           params.DB->kparam_data_ptr(), // DB
           params.L->kparam_data_ptr(), // L
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bk as constexpr 0
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.DO->kparam_stride(0), // stride_doz
           params.DO->kparam_stride(1), // stride_doh
           params.DO->kparam_stride(2), // stride_dom
           params.DK->kparam_stride(0), // stride_dkz
           params.DK->kparam_stride(1), // stride_dkh
           params.DK->kparam_stride(2), // stride_dkn
           params.DV->kparam_stride(0), // stride_dvz
           params.DV->kparam_stride(1), // stride_dvh
           params.DV->kparam_stride(2), // stride_dvk
           params.DQ->kparam_stride(0), // stride_dqz
           params.DQ->kparam_stride(1), // stride_dqh
           params.DQ->kparam_stride(2), // stride_dqm
           // params.DB->kparam_stride(0), // stride_dbz as constexpr 0
           // params.DB->kparam_stride(1), // stride_dbh as constexpr 0
           // params.DB->kparam_stride(2), // stride_dbm as constexpr 0
           CAST(&params.num_head_q), // num_head_q
           CAST(&params.num_head_k), // num_head_k
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.num_seqlens), // num_seqlens
           CAST(&params.max_seqlen_q), // max_seqlen_q
           CAST(&params.max_seqlen_k), // max_seqlen_k
           CAST(&params.head_dim), // head_dim
           CAST(&params.dropout_p), // dropout_p
           params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr
           params.philox_offset1->kparam_data_ptr(), // philox_offset1
           CAST(&params.philox_offset2), // philox_offset2
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           CAST(global_scratch)
         };
}
static std::vector<void*>
bwd_kernel_fuse_pp_args_3(const OpAttnBwdParams& params,
                          hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           CAST(&params.sm_scale), // sm_scale
           params.Out->kparam_data_ptr(), // Out
           params.DO->kparam_data_ptr(), // DO
           params.DK->kparam_data_ptr(), // DK
           params.DV->kparam_data_ptr(), // DV
           params.DQ->kparam_data_ptr(), // DQ
           params.DB->kparam_data_ptr(), // DB
           params.L->kparam_data_ptr(), // L
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.B->kparam_stride(0), // stride_bz
           params.B->kparam_stride(1), // stride_bh
           params.B->kparam_stride(2), // stride_bk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.DO->kparam_stride(0), // stride_doz
           params.DO->kparam_stride(1), // stride_doh
           params.DO->kparam_stride(2), // stride_dom
           params.DK->kparam_stride(0), // stride_dkz
           params.DK->kparam_stride(1), // stride_dkh
           params.DK->kparam_stride(2), // stride_dkn
           params.DV->kparam_stride(0), // stride_dvz
           params.DV->kparam_stride(1), // stride_dvh
           params.DV->kparam_stride(2), // stride_dvk
           params.DQ->kparam_stride(0), // stride_dqz
           params.DQ->kparam_stride(1), // stride_dqh
           params.DQ->kparam_stride(2), // stride_dqm
           params.DB->kparam_stride(0), // stride_dbz
           params.DB->kparam_stride(1), // stride_dbh
           params.DB->kparam_stride(2), // stride_dbm
           CAST(&params.num_head_q), // num_head_q
           CAST(&params.num_head_k), // num_head_k
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.num_seqlens), // num_seqlens
           CAST(&params.max_seqlen_q), // max_seqlen_q
           CAST(&params.max_seqlen_k), // max_seqlen_k
           CAST(&params.head_dim), // head_dim
           CAST(&params.dropout_p), // dropout_p
           params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr
           params.philox_offset1->kparam_data_ptr(), // philox_offset1
           CAST(&params.philox_offset2), // philox_offset2
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           CAST(global_scratch)
         };
}
static std::vector<void*>
bwd_kernel_fuse_pp_args_4(const OpAttnBwdParams& params,
                          hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           CAST(&params.sm_scale), // sm_scale
           params.Out->kparam_data_ptr(), // Out
           params.DO->kparam_data_ptr(), // DO
           params.DK->kparam_data_ptr(), // DK
           params.DV->kparam_data_ptr(), // DV
           params.DQ->kparam_data_ptr(), // DQ
           params.DB->kparam_data_ptr(), // DB
           params.L->kparam_data_ptr(), // L
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bk as constexpr 0
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.DO->kparam_stride(0), // stride_doz
           params.DO->kparam_stride(1), // stride_doh
           params.DO->kparam_stride(2), // stride_dom
           params.DK->kparam_stride(0), // stride_dkz
           params.DK->kparam_stride(1), // stride_dkh
           params.DK->kparam_stride(2), // stride_dkn
           params.DV->kparam_stride(0), // stride_dvz
           params.DV->kparam_stride(1), // stride_dvh
           params.DV->kparam_stride(2), // stride_dvk
           params.DQ->kparam_stride(0), // stride_dqz
           params.DQ->kparam_stride(1), // stride_dqh
           params.DQ->kparam_stride(2), // stride_dqm
           // params.DB->kparam_stride(0), // stride_dbz as constexpr 0
           // params.DB->kparam_stride(1), // stride_dbh as constexpr 0
           // params.DB->kparam_stride(2), // stride_dbm as constexpr 0
           CAST(&params.num_head_q), // num_head_q
           CAST(&params.num_head_k), // num_head_k
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.num_seqlens), // num_seqlens
           CAST(&params.max_seqlen_q), // max_seqlen_q
           CAST(&params.max_seqlen_k), // max_seqlen_k
           CAST(&params.head_dim), // head_dim
           // CAST(&params.dropout_p), // dropout_p as constexpr 0
           // params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr as constexpr 0
           // params.philox_offset1->kparam_data_ptr(), // philox_offset1 as constexpr 0
           // CAST(&params.philox_offset2), // philox_offset2 as constexpr 0
           CAST(&params.Window_left), // Window_left
           CAST(&params.Window_right), // Window_right
           CAST(global_scratch)
         };
}
static std::vector<void*>
bwd_kernel_fuse_pp_args_5(const OpAttnBwdParams& params,
                          hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           CAST(&params.sm_scale), // sm_scale
           params.Out->kparam_data_ptr(), // Out
           params.DO->kparam_data_ptr(), // DO
           params.DK->kparam_data_ptr(), // DK
           params.DV->kparam_data_ptr(), // DV
           params.DQ->kparam_data_ptr(), // DQ
           params.DB->kparam_data_ptr(), // DB
           params.L->kparam_data_ptr(), // L
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bk as constexpr 0
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.DO->kparam_stride(0), // stride_doz
           params.DO->kparam_stride(1), // stride_doh
           params.DO->kparam_stride(2), // stride_dom
           params.DK->kparam_stride(0), // stride_dkz
           params.DK->kparam_stride(1), // stride_dkh
           params.DK->kparam_stride(2), // stride_dkn
           params.DV->kparam_stride(0), // stride_dvz
           params.DV->kparam_stride(1), // stride_dvh
           params.DV->kparam_stride(2), // stride_dvk
           params.DQ->kparam_stride(0), // stride_dqz
           params.DQ->kparam_stride(1), // stride_dqh
           params.DQ->kparam_stride(2), // stride_dqm
           // params.DB->kparam_stride(0), // stride_dbz as constexpr 0
           // params.DB->kparam_stride(1), // stride_dbh as constexpr 0
           // params.DB->kparam_stride(2), // stride_dbm as constexpr 0
           CAST(&params.num_head_q), // num_head_q
           CAST(&params.num_head_k), // num_head_k
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.num_seqlens), // num_seqlens
           CAST(&params.max_seqlen_q), // max_seqlen_q
           CAST(&params.max_seqlen_k), // max_seqlen_k
           CAST(&params.head_dim), // head_dim
           CAST(&params.dropout_p), // dropout_p
           params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr
           params.philox_offset1->kparam_data_ptr(), // philox_offset1
           CAST(&params.philox_offset2), // philox_offset2
           CAST(&params.Window_left), // Window_left
           CAST(&params.Window_right), // Window_right
           CAST(global_scratch)
         };
}

namespace {
PP_FUNC prepare_arguments[ 6 ] = {
  bwd_kernel_fuse_pp_args_0,
  bwd_kernel_fuse_pp_args_1,
  bwd_kernel_fuse_pp_args_2,
  bwd_kernel_fuse_pp_args_3,
  bwd_kernel_fuse_pp_args_4,
  bwd_kernel_fuse_pp_args_5
};
}


const std::vector<std::string>& BwdKernelFuseMetadata::get_Q_choices()
{
    static const std::vector<std::string> choices = { "*fp16:16", "*bf16:16", "*fp32:16" };
    return choices;
}

const std::vector<std::string>& BwdKernelFuseMetadata::get_sm_scale_choices()
{
    static const std::vector<std::string> choices = { "fp32" };
    return choices;
}

const std::vector<std::string>& BwdKernelFuseMetadata::get_L_choices()
{
    static const std::vector<std::string> choices = { "*fp32:16" };
    return choices;
}

const std::vector<std::string>& BwdKernelFuseMetadata::get_num_head_q_choices()
{
    static const std::vector<std::string> choices = { "i32" };
    return choices;
}

const std::vector<std::string>& BwdKernelFuseMetadata::get_cu_seqlens_q_choices()
{
    static const std::vector<std::string> choices = { "*i32:16" };
    return choices;
}

const std::vector<std::string>& BwdKernelFuseMetadata::get_num_seqlens_choices()
{
    static const std::vector<std::string> choices = { "i32" };
    return choices;
}

const std::vector<int>& BwdKernelFuseMetadata::get_BLOCK_DMODEL_choices()
{
    static const std::vector<int> choices = { 16, 32, 48, 64, 80, 96, 128, 160, 192, 224, 256 };
    return choices;
}

const std::vector<int>& BwdKernelFuseMetadata::get_CAUSAL_TYPE_choices()
{
    static const std::vector<int> choices = { 0, 3 };
    return choices;
}

const std::vector<bool>& BwdKernelFuseMetadata::get_ENABLE_DROPOUT_choices()
{
    static const std::vector<bool> choices = { false, true };
    return choices;
}

const std::vector<bool>& BwdKernelFuseMetadata::get_PADDED_HEAD_choices()
{
    static const std::vector<bool> choices = { false, true };
    return choices;
}

const std::vector<int>& BwdKernelFuseMetadata::get_BIAS_TYPE_choices()
{
    static const std::vector<int> choices = { 0, 1 };
    return choices;
}

namespace autotune {

const char bwd_kernel_fuse_packed_string[] =
"16_16\0"
"wave2_warp2_stg1\0"
"32_16\0"
"wave3_warp4_stg1\0"
"wave4_warp4_stg1\0"
"32_32\0"
"wave2_warp4_stg1\0"
"wave4_warp2_stg1\0"
"64_16\0"
"wave3_warp2_stg1\0"
"64_32\0"
"64_64\0"
"wave1_warp4_stg1\0"
"wave1_warp2_stg1\0";

int bwd_kernel_fuse__lut_lambda__0 (const OpAttnBwdParams& params, int mod_number, int8_t lut[1][7][7]) {
    auto max_seqlen_q_binned_index = [] (int x) {
        if (x <= 16) return 0;
        if (x <= 32) return 1;
        if (x <= 64) return 2;
        if (x <= 128) return 3;
        if (x <= 256) return 4;
        if (x <= 512) return 5;
        if (x <= 1024) return 6;
        return 6;
    }(params.max_seqlen_q);
    auto max_seqlen_k_binned_index = [] (int x) {
        if (x <= 16) return 0;
        if (x <= 32) return 1;
        if (x <= 64) return 2;
        if (x <= 128) return 3;
        if (x <= 256) return 4;
        if (x <= 512) return 5;
        if (x <= 1024) return 6;
        return 6;
    }(params.max_seqlen_k);
    return lut[mod_number][max_seqlen_q_binned_index][max_seqlen_k_binned_index];
};

} // namespace autotune

BwdKernelFuseContext::AutoTuneTableEntry
BwdKernelFuseContext::autotune_table[][ 528 ] = {
    {
        &autotune::Autotune_bwd_kernel_fuse__A0__F0,
        &autotune::Autotune_bwd_kernel_fuse__A0__F1,
        &autotune::Autotune_bwd_kernel_fuse__A0__F2,
        &autotune::Autotune_bwd_kernel_fuse__A0__F3,
        &autotune::Autotune_bwd_kernel_fuse__A0__F4,
        &autotune::Autotune_bwd_kernel_fuse__A0__F5,
        &autotune::Autotune_bwd_kernel_fuse__A0__F6,
        &autotune::Autotune_bwd_kernel_fuse__A0__F7,
        &autotune::Autotune_bwd_kernel_fuse__A0__F8,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F10,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F12,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F14,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F16,
        &autotune::Autotune_bwd_kernel_fuse__A0__F17,
        &autotune::Autotune_bwd_kernel_fuse__A0__F18,
        &autotune::Autotune_bwd_kernel_fuse__A0__F19,
        &autotune::Autotune_bwd_kernel_fuse__A0__F20,
        &autotune::Autotune_bwd_kernel_fuse__A0__F21,
        &autotune::Autotune_bwd_kernel_fuse__A0__F22,
        &autotune::Autotune_bwd_kernel_fuse__A0__F23,
        &autotune::Autotune_bwd_kernel_fuse__A0__F24,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F26,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F28,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F30,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F48,
        &autotune::Autotune_bwd_kernel_fuse__A0__F49,
        &autotune::Autotune_bwd_kernel_fuse__A0__F50,
        &autotune::Autotune_bwd_kernel_fuse__A0__F51,
        &autotune::Autotune_bwd_kernel_fuse__A0__F52,
        &autotune::Autotune_bwd_kernel_fuse__A0__F53,
        &autotune::Autotune_bwd_kernel_fuse__A0__F54,
        &autotune::Autotune_bwd_kernel_fuse__A0__F55,
        &autotune::Autotune_bwd_kernel_fuse__A0__F56,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F58,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F60,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F62,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F80,
        &autotune::Autotune_bwd_kernel_fuse__A0__F81,
        &autotune::Autotune_bwd_kernel_fuse__A0__F82,
        &autotune::Autotune_bwd_kernel_fuse__A0__F83,
        &autotune::Autotune_bwd_kernel_fuse__A0__F84,
        &autotune::Autotune_bwd_kernel_fuse__A0__F85,
        &autotune::Autotune_bwd_kernel_fuse__A0__F86,
        &autotune::Autotune_bwd_kernel_fuse__A0__F87,
        &autotune::Autotune_bwd_kernel_fuse__A0__F88,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F90,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F92,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F94,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F96,
        &autotune::Autotune_bwd_kernel_fuse__A0__F97,
        &autotune::Autotune_bwd_kernel_fuse__A0__F98,
        &autotune::Autotune_bwd_kernel_fuse__A0__F99,
        &autotune::Autotune_bwd_kernel_fuse__A0__F100,
        &autotune::Autotune_bwd_kernel_fuse__A0__F101,
        &autotune::Autotune_bwd_kernel_fuse__A0__F102,
        &autotune::Autotune_bwd_kernel_fuse__A0__F103,
        &autotune::Autotune_bwd_kernel_fuse__A0__F104,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F106,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F108,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F110,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F112,
        &autotune::Autotune_bwd_kernel_fuse__A0__F113,
        &autotune::Autotune_bwd_kernel_fuse__A0__F114,
        &autotune::Autotune_bwd_kernel_fuse__A0__F115,
        &autotune::Autotune_bwd_kernel_fuse__A0__F116,
        &autotune::Autotune_bwd_kernel_fuse__A0__F117,
        &autotune::Autotune_bwd_kernel_fuse__A0__F118,
        &autotune::Autotune_bwd_kernel_fuse__A0__F119,
        &autotune::Autotune_bwd_kernel_fuse__A0__F120,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F122,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F124,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F126,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F128,
        &autotune::Autotune_bwd_kernel_fuse__A0__F129,
        &autotune::Autotune_bwd_kernel_fuse__A0__F130,
        &autotune::Autotune_bwd_kernel_fuse__A0__F131,
        &autotune::Autotune_bwd_kernel_fuse__A0__F132,
        &autotune::Autotune_bwd_kernel_fuse__A0__F133,
        &autotune::Autotune_bwd_kernel_fuse__A0__F134,
        &autotune::Autotune_bwd_kernel_fuse__A0__F135,
        &autotune::Autotune_bwd_kernel_fuse__A0__F136,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F138,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F140,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F142,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F144,
        &autotune::Autotune_bwd_kernel_fuse__A0__F145,
        &autotune::Autotune_bwd_kernel_fuse__A0__F146,
        &autotune::Autotune_bwd_kernel_fuse__A0__F147,
        &autotune::Autotune_bwd_kernel_fuse__A0__F148,
        &autotune::Autotune_bwd_kernel_fuse__A0__F149,
        &autotune::Autotune_bwd_kernel_fuse__A0__F150,
        &autotune::Autotune_bwd_kernel_fuse__A0__F151,
        &autotune::Autotune_bwd_kernel_fuse__A0__F152,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F154,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F156,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F158,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F160,
        &autotune::Autotune_bwd_kernel_fuse__A0__F161,
        &autotune::Autotune_bwd_kernel_fuse__A0__F162,
        &autotune::Autotune_bwd_kernel_fuse__A0__F163,
        &autotune::Autotune_bwd_kernel_fuse__A0__F164,
        &autotune::Autotune_bwd_kernel_fuse__A0__F165,
        &autotune::Autotune_bwd_kernel_fuse__A0__F166,
        &autotune::Autotune_bwd_kernel_fuse__A0__F167,
        &autotune::Autotune_bwd_kernel_fuse__A0__F168,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F170,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F172,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F174,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F176,
        &autotune::Autotune_bwd_kernel_fuse__A0__F177,
        &autotune::Autotune_bwd_kernel_fuse__A0__F178,
        &autotune::Autotune_bwd_kernel_fuse__A0__F179,
        &autotune::Autotune_bwd_kernel_fuse__A0__F180,
        &autotune::Autotune_bwd_kernel_fuse__A0__F181,
        &autotune::Autotune_bwd_kernel_fuse__A0__F182,
        &autotune::Autotune_bwd_kernel_fuse__A0__F183,
        &autotune::Autotune_bwd_kernel_fuse__A0__F184,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F186,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F188,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F190,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F192,
        &autotune::Autotune_bwd_kernel_fuse__A0__F193,
        &autotune::Autotune_bwd_kernel_fuse__A0__F194,
        &autotune::Autotune_bwd_kernel_fuse__A0__F195,
        &autotune::Autotune_bwd_kernel_fuse__A0__F196,
        &autotune::Autotune_bwd_kernel_fuse__A0__F197,
        &autotune::Autotune_bwd_kernel_fuse__A0__F198,
        &autotune::Autotune_bwd_kernel_fuse__A0__F199,
        &autotune::Autotune_bwd_kernel_fuse__A0__F200,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F202,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F204,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F206,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F224,
        &autotune::Autotune_bwd_kernel_fuse__A0__F225,
        &autotune::Autotune_bwd_kernel_fuse__A0__F226,
        &autotune::Autotune_bwd_kernel_fuse__A0__F227,
        &autotune::Autotune_bwd_kernel_fuse__A0__F228,
        &autotune::Autotune_bwd_kernel_fuse__A0__F229,
        &autotune::Autotune_bwd_kernel_fuse__A0__F230,
        &autotune::Autotune_bwd_kernel_fuse__A0__F231,
        &autotune::Autotune_bwd_kernel_fuse__A0__F232,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F234,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F236,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F238,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F256,
        &autotune::Autotune_bwd_kernel_fuse__A0__F257,
        &autotune::Autotune_bwd_kernel_fuse__A0__F258,
        &autotune::Autotune_bwd_kernel_fuse__A0__F259,
        &autotune::Autotune_bwd_kernel_fuse__A0__F260,
        &autotune::Autotune_bwd_kernel_fuse__A0__F261,
        &autotune::Autotune_bwd_kernel_fuse__A0__F262,
        &autotune::Autotune_bwd_kernel_fuse__A0__F263,
        &autotune::Autotune_bwd_kernel_fuse__A0__F264,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F266,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F268,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F270,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F272,
        &autotune::Autotune_bwd_kernel_fuse__A0__F273,
        &autotune::Autotune_bwd_kernel_fuse__A0__F274,
        &autotune::Autotune_bwd_kernel_fuse__A0__F275,
        &autotune::Autotune_bwd_kernel_fuse__A0__F276,
        &autotune::Autotune_bwd_kernel_fuse__A0__F277,
        &autotune::Autotune_bwd_kernel_fuse__A0__F278,
        &autotune::Autotune_bwd_kernel_fuse__A0__F279,
        &autotune::Autotune_bwd_kernel_fuse__A0__F280,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F282,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F284,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F286,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F288,
        &autotune::Autotune_bwd_kernel_fuse__A0__F289,
        &autotune::Autotune_bwd_kernel_fuse__A0__F290,
        &autotune::Autotune_bwd_kernel_fuse__A0__F291,
        &autotune::Autotune_bwd_kernel_fuse__A0__F292,
        &autotune::Autotune_bwd_kernel_fuse__A0__F293,
        &autotune::Autotune_bwd_kernel_fuse__A0__F294,
        &autotune::Autotune_bwd_kernel_fuse__A0__F295,
        &autotune::Autotune_bwd_kernel_fuse__A0__F296,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F298,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F300,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F302,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F304,
        &autotune::Autotune_bwd_kernel_fuse__A0__F305,
        &autotune::Autotune_bwd_kernel_fuse__A0__F306,
        &autotune::Autotune_bwd_kernel_fuse__A0__F307,
        &autotune::Autotune_bwd_kernel_fuse__A0__F308,
        &autotune::Autotune_bwd_kernel_fuse__A0__F309,
        &autotune::Autotune_bwd_kernel_fuse__A0__F310,
        &autotune::Autotune_bwd_kernel_fuse__A0__F311,
        &autotune::Autotune_bwd_kernel_fuse__A0__F312,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F314,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F316,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F318,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F320,
        &autotune::Autotune_bwd_kernel_fuse__A0__F321,
        &autotune::Autotune_bwd_kernel_fuse__A0__F322,
        &autotune::Autotune_bwd_kernel_fuse__A0__F323,
        &autotune::Autotune_bwd_kernel_fuse__A0__F324,
        &autotune::Autotune_bwd_kernel_fuse__A0__F325,
        &autotune::Autotune_bwd_kernel_fuse__A0__F326,
        &autotune::Autotune_bwd_kernel_fuse__A0__F327,
        &autotune::Autotune_bwd_kernel_fuse__A0__F328,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F330,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F332,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F334,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F336,
        &autotune::Autotune_bwd_kernel_fuse__A0__F337,
        &autotune::Autotune_bwd_kernel_fuse__A0__F338,
        &autotune::Autotune_bwd_kernel_fuse__A0__F339,
        &autotune::Autotune_bwd_kernel_fuse__A0__F340,
        &autotune::Autotune_bwd_kernel_fuse__A0__F341,
        &autotune::Autotune_bwd_kernel_fuse__A0__F342,
        &autotune::Autotune_bwd_kernel_fuse__A0__F343,
        &autotune::Autotune_bwd_kernel_fuse__A0__F344,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F346,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F348,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F350,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F352,
        &autotune::Autotune_bwd_kernel_fuse__A0__F353,
        &autotune::Autotune_bwd_kernel_fuse__A0__F354,
        &autotune::Autotune_bwd_kernel_fuse__A0__F355,
        &autotune::Autotune_bwd_kernel_fuse__A0__F356,
        &autotune::Autotune_bwd_kernel_fuse__A0__F357,
        &autotune::Autotune_bwd_kernel_fuse__A0__F358,
        &autotune::Autotune_bwd_kernel_fuse__A0__F359,
        &autotune::Autotune_bwd_kernel_fuse__A0__F360,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F362,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F364,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F366,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F368,
        &autotune::Autotune_bwd_kernel_fuse__A0__F369,
        &autotune::Autotune_bwd_kernel_fuse__A0__F370,
        &autotune::Autotune_bwd_kernel_fuse__A0__F371,
        &autotune::Autotune_bwd_kernel_fuse__A0__F372,
        &autotune::Autotune_bwd_kernel_fuse__A0__F373,
        &autotune::Autotune_bwd_kernel_fuse__A0__F374,
        &autotune::Autotune_bwd_kernel_fuse__A0__F375,
        &autotune::Autotune_bwd_kernel_fuse__A0__F376,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F378,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F380,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F382,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F400,
        &autotune::Autotune_bwd_kernel_fuse__A0__F401,
        &autotune::Autotune_bwd_kernel_fuse__A0__F402,
        &autotune::Autotune_bwd_kernel_fuse__A0__F403,
        &autotune::Autotune_bwd_kernel_fuse__A0__F404,
        &autotune::Autotune_bwd_kernel_fuse__A0__F405,
        &autotune::Autotune_bwd_kernel_fuse__A0__F406,
        &autotune::Autotune_bwd_kernel_fuse__A0__F407,
        &autotune::Autotune_bwd_kernel_fuse__A0__F408,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F410,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F412,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F414,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F432,
        &autotune::Autotune_bwd_kernel_fuse__A0__F433,
        &autotune::Autotune_bwd_kernel_fuse__A0__F434,
        &autotune::Autotune_bwd_kernel_fuse__A0__F435,
        &autotune::Autotune_bwd_kernel_fuse__A0__F436,
        &autotune::Autotune_bwd_kernel_fuse__A0__F437,
        &autotune::Autotune_bwd_kernel_fuse__A0__F438,
        &autotune::Autotune_bwd_kernel_fuse__A0__F439,
        &autotune::Autotune_bwd_kernel_fuse__A0__F440,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F442,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F444,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F446,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F448,
        &autotune::Autotune_bwd_kernel_fuse__A0__F449,
        &autotune::Autotune_bwd_kernel_fuse__A0__F450,
        &autotune::Autotune_bwd_kernel_fuse__A0__F451,
        &autotune::Autotune_bwd_kernel_fuse__A0__F452,
        &autotune::Autotune_bwd_kernel_fuse__A0__F453,
        &autotune::Autotune_bwd_kernel_fuse__A0__F454,
        &autotune::Autotune_bwd_kernel_fuse__A0__F455,
        &autotune::Autotune_bwd_kernel_fuse__A0__F456,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F458,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F460,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F462,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F464,
        &autotune::Autotune_bwd_kernel_fuse__A0__F465,
        &autotune::Autotune_bwd_kernel_fuse__A0__F466,
        &autotune::Autotune_bwd_kernel_fuse__A0__F467,
        &autotune::Autotune_bwd_kernel_fuse__A0__F468,
        &autotune::Autotune_bwd_kernel_fuse__A0__F469,
        &autotune::Autotune_bwd_kernel_fuse__A0__F470,
        &autotune::Autotune_bwd_kernel_fuse__A0__F471,
        &autotune::Autotune_bwd_kernel_fuse__A0__F472,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F474,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F476,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F478,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F480,
        &autotune::Autotune_bwd_kernel_fuse__A0__F481,
        &autotune::Autotune_bwd_kernel_fuse__A0__F482,
        &autotune::Autotune_bwd_kernel_fuse__A0__F483,
        &autotune::Autotune_bwd_kernel_fuse__A0__F484,
        &autotune::Autotune_bwd_kernel_fuse__A0__F485,
        &autotune::Autotune_bwd_kernel_fuse__A0__F486,
        &autotune::Autotune_bwd_kernel_fuse__A0__F487,
        &autotune::Autotune_bwd_kernel_fuse__A0__F488,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F490,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F492,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F494,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F496,
        &autotune::Autotune_bwd_kernel_fuse__A0__F497,
        &autotune::Autotune_bwd_kernel_fuse__A0__F498,
        &autotune::Autotune_bwd_kernel_fuse__A0__F499,
        &autotune::Autotune_bwd_kernel_fuse__A0__F500,
        &autotune::Autotune_bwd_kernel_fuse__A0__F501,
        &autotune::Autotune_bwd_kernel_fuse__A0__F502,
        &autotune::Autotune_bwd_kernel_fuse__A0__F503,
        &autotune::Autotune_bwd_kernel_fuse__A0__F504,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F506,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F508,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F510,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F512,
        &autotune::Autotune_bwd_kernel_fuse__A0__F513,
        &autotune::Autotune_bwd_kernel_fuse__A0__F514,
        &autotune::Autotune_bwd_kernel_fuse__A0__F515,
        &autotune::Autotune_bwd_kernel_fuse__A0__F516,
        &autotune::Autotune_bwd_kernel_fuse__A0__F517,
        &autotune::Autotune_bwd_kernel_fuse__A0__F518,
        &autotune::Autotune_bwd_kernel_fuse__A0__F519,
        &autotune::Autotune_bwd_kernel_fuse__A0__F520,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F522,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F524,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A0__F526,
        nullptr,
    },
    {
        &autotune::Autotune_bwd_kernel_fuse__A1__F0,
        &autotune::Autotune_bwd_kernel_fuse__A1__F1,
        &autotune::Autotune_bwd_kernel_fuse__A1__F2,
        &autotune::Autotune_bwd_kernel_fuse__A1__F3,
        &autotune::Autotune_bwd_kernel_fuse__A1__F4,
        &autotune::Autotune_bwd_kernel_fuse__A1__F5,
        &autotune::Autotune_bwd_kernel_fuse__A1__F6,
        &autotune::Autotune_bwd_kernel_fuse__A1__F7,
        &autotune::Autotune_bwd_kernel_fuse__A1__F8,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F10,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F12,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F14,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F16,
        &autotune::Autotune_bwd_kernel_fuse__A1__F17,
        &autotune::Autotune_bwd_kernel_fuse__A1__F18,
        &autotune::Autotune_bwd_kernel_fuse__A1__F19,
        &autotune::Autotune_bwd_kernel_fuse__A1__F20,
        &autotune::Autotune_bwd_kernel_fuse__A1__F21,
        &autotune::Autotune_bwd_kernel_fuse__A1__F22,
        &autotune::Autotune_bwd_kernel_fuse__A1__F23,
        &autotune::Autotune_bwd_kernel_fuse__A1__F24,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F26,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F28,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F30,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F32,
        &autotune::Autotune_bwd_kernel_fuse__A1__F33,
        &autotune::Autotune_bwd_kernel_fuse__A1__F34,
        &autotune::Autotune_bwd_kernel_fuse__A1__F35,
        &autotune::Autotune_bwd_kernel_fuse__A1__F36,
        &autotune::Autotune_bwd_kernel_fuse__A1__F37,
        &autotune::Autotune_bwd_kernel_fuse__A1__F38,
        &autotune::Autotune_bwd_kernel_fuse__A1__F39,
        &autotune::Autotune_bwd_kernel_fuse__A1__F40,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F42,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F44,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F46,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F48,
        &autotune::Autotune_bwd_kernel_fuse__A1__F49,
        &autotune::Autotune_bwd_kernel_fuse__A1__F50,
        &autotune::Autotune_bwd_kernel_fuse__A1__F51,
        &autotune::Autotune_bwd_kernel_fuse__A1__F52,
        &autotune::Autotune_bwd_kernel_fuse__A1__F53,
        &autotune::Autotune_bwd_kernel_fuse__A1__F54,
        &autotune::Autotune_bwd_kernel_fuse__A1__F55,
        &autotune::Autotune_bwd_kernel_fuse__A1__F56,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F58,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F60,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F62,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F64,
        &autotune::Autotune_bwd_kernel_fuse__A1__F65,
        &autotune::Autotune_bwd_kernel_fuse__A1__F66,
        &autotune::Autotune_bwd_kernel_fuse__A1__F67,
        &autotune::Autotune_bwd_kernel_fuse__A1__F68,
        &autotune::Autotune_bwd_kernel_fuse__A1__F69,
        &autotune::Autotune_bwd_kernel_fuse__A1__F70,
        &autotune::Autotune_bwd_kernel_fuse__A1__F71,
        &autotune::Autotune_bwd_kernel_fuse__A1__F72,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F74,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F76,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F78,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F80,
        &autotune::Autotune_bwd_kernel_fuse__A1__F81,
        &autotune::Autotune_bwd_kernel_fuse__A1__F82,
        &autotune::Autotune_bwd_kernel_fuse__A1__F83,
        &autotune::Autotune_bwd_kernel_fuse__A1__F84,
        &autotune::Autotune_bwd_kernel_fuse__A1__F85,
        &autotune::Autotune_bwd_kernel_fuse__A1__F86,
        &autotune::Autotune_bwd_kernel_fuse__A1__F87,
        &autotune::Autotune_bwd_kernel_fuse__A1__F88,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F90,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F92,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F94,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F96,
        &autotune::Autotune_bwd_kernel_fuse__A1__F97,
        &autotune::Autotune_bwd_kernel_fuse__A1__F98,
        &autotune::Autotune_bwd_kernel_fuse__A1__F99,
        &autotune::Autotune_bwd_kernel_fuse__A1__F100,
        &autotune::Autotune_bwd_kernel_fuse__A1__F101,
        &autotune::Autotune_bwd_kernel_fuse__A1__F102,
        &autotune::Autotune_bwd_kernel_fuse__A1__F103,
        &autotune::Autotune_bwd_kernel_fuse__A1__F104,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F106,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F108,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F110,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F112,
        &autotune::Autotune_bwd_kernel_fuse__A1__F113,
        &autotune::Autotune_bwd_kernel_fuse__A1__F114,
        &autotune::Autotune_bwd_kernel_fuse__A1__F115,
        &autotune::Autotune_bwd_kernel_fuse__A1__F116,
        &autotune::Autotune_bwd_kernel_fuse__A1__F117,
        &autotune::Autotune_bwd_kernel_fuse__A1__F118,
        &autotune::Autotune_bwd_kernel_fuse__A1__F119,
        &autotune::Autotune_bwd_kernel_fuse__A1__F120,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F122,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F124,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F126,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F128,
        &autotune::Autotune_bwd_kernel_fuse__A1__F129,
        &autotune::Autotune_bwd_kernel_fuse__A1__F130,
        &autotune::Autotune_bwd_kernel_fuse__A1__F131,
        &autotune::Autotune_bwd_kernel_fuse__A1__F132,
        &autotune::Autotune_bwd_kernel_fuse__A1__F133,
        &autotune::Autotune_bwd_kernel_fuse__A1__F134,
        &autotune::Autotune_bwd_kernel_fuse__A1__F135,
        &autotune::Autotune_bwd_kernel_fuse__A1__F136,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F138,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F140,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F142,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F144,
        &autotune::Autotune_bwd_kernel_fuse__A1__F145,
        &autotune::Autotune_bwd_kernel_fuse__A1__F146,
        &autotune::Autotune_bwd_kernel_fuse__A1__F147,
        &autotune::Autotune_bwd_kernel_fuse__A1__F148,
        &autotune::Autotune_bwd_kernel_fuse__A1__F149,
        &autotune::Autotune_bwd_kernel_fuse__A1__F150,
        &autotune::Autotune_bwd_kernel_fuse__A1__F151,
        &autotune::Autotune_bwd_kernel_fuse__A1__F152,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F154,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F156,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F158,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F160,
        &autotune::Autotune_bwd_kernel_fuse__A1__F161,
        &autotune::Autotune_bwd_kernel_fuse__A1__F162,
        &autotune::Autotune_bwd_kernel_fuse__A1__F163,
        &autotune::Autotune_bwd_kernel_fuse__A1__F164,
        &autotune::Autotune_bwd_kernel_fuse__A1__F165,
        &autotune::Autotune_bwd_kernel_fuse__A1__F166,
        &autotune::Autotune_bwd_kernel_fuse__A1__F167,
        &autotune::Autotune_bwd_kernel_fuse__A1__F168,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F170,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F172,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F174,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F176,
        &autotune::Autotune_bwd_kernel_fuse__A1__F177,
        &autotune::Autotune_bwd_kernel_fuse__A1__F178,
        &autotune::Autotune_bwd_kernel_fuse__A1__F179,
        &autotune::Autotune_bwd_kernel_fuse__A1__F180,
        &autotune::Autotune_bwd_kernel_fuse__A1__F181,
        &autotune::Autotune_bwd_kernel_fuse__A1__F182,
        &autotune::Autotune_bwd_kernel_fuse__A1__F183,
        &autotune::Autotune_bwd_kernel_fuse__A1__F184,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F186,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F188,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F190,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F192,
        &autotune::Autotune_bwd_kernel_fuse__A1__F193,
        &autotune::Autotune_bwd_kernel_fuse__A1__F194,
        &autotune::Autotune_bwd_kernel_fuse__A1__F195,
        &autotune::Autotune_bwd_kernel_fuse__A1__F196,
        &autotune::Autotune_bwd_kernel_fuse__A1__F197,
        &autotune::Autotune_bwd_kernel_fuse__A1__F198,
        &autotune::Autotune_bwd_kernel_fuse__A1__F199,
        &autotune::Autotune_bwd_kernel_fuse__A1__F200,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F202,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F204,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F206,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F208,
        &autotune::Autotune_bwd_kernel_fuse__A1__F209,
        &autotune::Autotune_bwd_kernel_fuse__A1__F210,
        &autotune::Autotune_bwd_kernel_fuse__A1__F211,
        &autotune::Autotune_bwd_kernel_fuse__A1__F212,
        &autotune::Autotune_bwd_kernel_fuse__A1__F213,
        &autotune::Autotune_bwd_kernel_fuse__A1__F214,
        &autotune::Autotune_bwd_kernel_fuse__A1__F215,
        &autotune::Autotune_bwd_kernel_fuse__A1__F216,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F218,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F220,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F222,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F224,
        &autotune::Autotune_bwd_kernel_fuse__A1__F225,
        &autotune::Autotune_bwd_kernel_fuse__A1__F226,
        &autotune::Autotune_bwd_kernel_fuse__A1__F227,
        &autotune::Autotune_bwd_kernel_fuse__A1__F228,
        &autotune::Autotune_bwd_kernel_fuse__A1__F229,
        &autotune::Autotune_bwd_kernel_fuse__A1__F230,
        &autotune::Autotune_bwd_kernel_fuse__A1__F231,
        &autotune::Autotune_bwd_kernel_fuse__A1__F232,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F234,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F236,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F238,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F240,
        &autotune::Autotune_bwd_kernel_fuse__A1__F241,
        &autotune::Autotune_bwd_kernel_fuse__A1__F242,
        &autotune::Autotune_bwd_kernel_fuse__A1__F243,
        &autotune::Autotune_bwd_kernel_fuse__A1__F244,
        &autotune::Autotune_bwd_kernel_fuse__A1__F245,
        &autotune::Autotune_bwd_kernel_fuse__A1__F246,
        &autotune::Autotune_bwd_kernel_fuse__A1__F247,
        &autotune::Autotune_bwd_kernel_fuse__A1__F248,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F250,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F252,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F254,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F256,
        &autotune::Autotune_bwd_kernel_fuse__A1__F257,
        &autotune::Autotune_bwd_kernel_fuse__A1__F258,
        &autotune::Autotune_bwd_kernel_fuse__A1__F259,
        &autotune::Autotune_bwd_kernel_fuse__A1__F260,
        &autotune::Autotune_bwd_kernel_fuse__A1__F261,
        &autotune::Autotune_bwd_kernel_fuse__A1__F262,
        &autotune::Autotune_bwd_kernel_fuse__A1__F263,
        &autotune::Autotune_bwd_kernel_fuse__A1__F264,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F266,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F268,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F270,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F272,
        &autotune::Autotune_bwd_kernel_fuse__A1__F273,
        &autotune::Autotune_bwd_kernel_fuse__A1__F274,
        &autotune::Autotune_bwd_kernel_fuse__A1__F275,
        &autotune::Autotune_bwd_kernel_fuse__A1__F276,
        &autotune::Autotune_bwd_kernel_fuse__A1__F277,
        &autotune::Autotune_bwd_kernel_fuse__A1__F278,
        &autotune::Autotune_bwd_kernel_fuse__A1__F279,
        &autotune::Autotune_bwd_kernel_fuse__A1__F280,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F282,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F284,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F286,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F288,
        &autotune::Autotune_bwd_kernel_fuse__A1__F289,
        &autotune::Autotune_bwd_kernel_fuse__A1__F290,
        &autotune::Autotune_bwd_kernel_fuse__A1__F291,
        &autotune::Autotune_bwd_kernel_fuse__A1__F292,
        &autotune::Autotune_bwd_kernel_fuse__A1__F293,
        &autotune::Autotune_bwd_kernel_fuse__A1__F294,
        &autotune::Autotune_bwd_kernel_fuse__A1__F295,
        &autotune::Autotune_bwd_kernel_fuse__A1__F296,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F298,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F300,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F302,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F304,
        &autotune::Autotune_bwd_kernel_fuse__A1__F305,
        &autotune::Autotune_bwd_kernel_fuse__A1__F306,
        &autotune::Autotune_bwd_kernel_fuse__A1__F307,
        &autotune::Autotune_bwd_kernel_fuse__A1__F308,
        &autotune::Autotune_bwd_kernel_fuse__A1__F309,
        &autotune::Autotune_bwd_kernel_fuse__A1__F310,
        &autotune::Autotune_bwd_kernel_fuse__A1__F311,
        &autotune::Autotune_bwd_kernel_fuse__A1__F312,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F314,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F316,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F318,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F320,
        &autotune::Autotune_bwd_kernel_fuse__A1__F321,
        &autotune::Autotune_bwd_kernel_fuse__A1__F322,
        &autotune::Autotune_bwd_kernel_fuse__A1__F323,
        &autotune::Autotune_bwd_kernel_fuse__A1__F324,
        &autotune::Autotune_bwd_kernel_fuse__A1__F325,
        &autotune::Autotune_bwd_kernel_fuse__A1__F326,
        &autotune::Autotune_bwd_kernel_fuse__A1__F327,
        &autotune::Autotune_bwd_kernel_fuse__A1__F328,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F330,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F332,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F334,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F336,
        &autotune::Autotune_bwd_kernel_fuse__A1__F337,
        &autotune::Autotune_bwd_kernel_fuse__A1__F338,
        &autotune::Autotune_bwd_kernel_fuse__A1__F339,
        &autotune::Autotune_bwd_kernel_fuse__A1__F340,
        &autotune::Autotune_bwd_kernel_fuse__A1__F341,
        &autotune::Autotune_bwd_kernel_fuse__A1__F342,
        &autotune::Autotune_bwd_kernel_fuse__A1__F343,
        &autotune::Autotune_bwd_kernel_fuse__A1__F344,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F346,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F348,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F350,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F352,
        &autotune::Autotune_bwd_kernel_fuse__A1__F353,
        &autotune::Autotune_bwd_kernel_fuse__A1__F354,
        &autotune::Autotune_bwd_kernel_fuse__A1__F355,
        &autotune::Autotune_bwd_kernel_fuse__A1__F356,
        &autotune::Autotune_bwd_kernel_fuse__A1__F357,
        &autotune::Autotune_bwd_kernel_fuse__A1__F358,
        &autotune::Autotune_bwd_kernel_fuse__A1__F359,
        &autotune::Autotune_bwd_kernel_fuse__A1__F360,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F362,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F364,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F366,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F368,
        &autotune::Autotune_bwd_kernel_fuse__A1__F369,
        &autotune::Autotune_bwd_kernel_fuse__A1__F370,
        &autotune::Autotune_bwd_kernel_fuse__A1__F371,
        &autotune::Autotune_bwd_kernel_fuse__A1__F372,
        &autotune::Autotune_bwd_kernel_fuse__A1__F373,
        &autotune::Autotune_bwd_kernel_fuse__A1__F374,
        &autotune::Autotune_bwd_kernel_fuse__A1__F375,
        &autotune::Autotune_bwd_kernel_fuse__A1__F376,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F378,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F380,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F382,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F384,
        &autotune::Autotune_bwd_kernel_fuse__A1__F385,
        &autotune::Autotune_bwd_kernel_fuse__A1__F386,
        &autotune::Autotune_bwd_kernel_fuse__A1__F387,
        &autotune::Autotune_bwd_kernel_fuse__A1__F388,
        &autotune::Autotune_bwd_kernel_fuse__A1__F389,
        &autotune::Autotune_bwd_kernel_fuse__A1__F390,
        &autotune::Autotune_bwd_kernel_fuse__A1__F391,
        &autotune::Autotune_bwd_kernel_fuse__A1__F392,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F394,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F396,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F398,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F400,
        &autotune::Autotune_bwd_kernel_fuse__A1__F401,
        &autotune::Autotune_bwd_kernel_fuse__A1__F402,
        &autotune::Autotune_bwd_kernel_fuse__A1__F403,
        &autotune::Autotune_bwd_kernel_fuse__A1__F404,
        &autotune::Autotune_bwd_kernel_fuse__A1__F405,
        &autotune::Autotune_bwd_kernel_fuse__A1__F406,
        &autotune::Autotune_bwd_kernel_fuse__A1__F407,
        &autotune::Autotune_bwd_kernel_fuse__A1__F408,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F410,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F412,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F414,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F416,
        &autotune::Autotune_bwd_kernel_fuse__A1__F417,
        &autotune::Autotune_bwd_kernel_fuse__A1__F418,
        &autotune::Autotune_bwd_kernel_fuse__A1__F419,
        &autotune::Autotune_bwd_kernel_fuse__A1__F420,
        &autotune::Autotune_bwd_kernel_fuse__A1__F421,
        &autotune::Autotune_bwd_kernel_fuse__A1__F422,
        &autotune::Autotune_bwd_kernel_fuse__A1__F423,
        &autotune::Autotune_bwd_kernel_fuse__A1__F424,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F426,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F428,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F430,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F432,
        &autotune::Autotune_bwd_kernel_fuse__A1__F433,
        &autotune::Autotune_bwd_kernel_fuse__A1__F434,
        &autotune::Autotune_bwd_kernel_fuse__A1__F435,
        &autotune::Autotune_bwd_kernel_fuse__A1__F436,
        &autotune::Autotune_bwd_kernel_fuse__A1__F437,
        &autotune::Autotune_bwd_kernel_fuse__A1__F438,
        &autotune::Autotune_bwd_kernel_fuse__A1__F439,
        &autotune::Autotune_bwd_kernel_fuse__A1__F440,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F442,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F444,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F446,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F448,
        &autotune::Autotune_bwd_kernel_fuse__A1__F449,
        &autotune::Autotune_bwd_kernel_fuse__A1__F450,
        &autotune::Autotune_bwd_kernel_fuse__A1__F451,
        &autotune::Autotune_bwd_kernel_fuse__A1__F452,
        &autotune::Autotune_bwd_kernel_fuse__A1__F453,
        &autotune::Autotune_bwd_kernel_fuse__A1__F454,
        &autotune::Autotune_bwd_kernel_fuse__A1__F455,
        &autotune::Autotune_bwd_kernel_fuse__A1__F456,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F458,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F460,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F462,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F464,
        &autotune::Autotune_bwd_kernel_fuse__A1__F465,
        &autotune::Autotune_bwd_kernel_fuse__A1__F466,
        &autotune::Autotune_bwd_kernel_fuse__A1__F467,
        &autotune::Autotune_bwd_kernel_fuse__A1__F468,
        &autotune::Autotune_bwd_kernel_fuse__A1__F469,
        &autotune::Autotune_bwd_kernel_fuse__A1__F470,
        &autotune::Autotune_bwd_kernel_fuse__A1__F471,
        &autotune::Autotune_bwd_kernel_fuse__A1__F472,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F474,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F476,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F478,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F480,
        &autotune::Autotune_bwd_kernel_fuse__A1__F481,
        &autotune::Autotune_bwd_kernel_fuse__A1__F482,
        &autotune::Autotune_bwd_kernel_fuse__A1__F483,
        &autotune::Autotune_bwd_kernel_fuse__A1__F484,
        &autotune::Autotune_bwd_kernel_fuse__A1__F485,
        &autotune::Autotune_bwd_kernel_fuse__A1__F486,
        &autotune::Autotune_bwd_kernel_fuse__A1__F487,
        &autotune::Autotune_bwd_kernel_fuse__A1__F488,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F490,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F492,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F494,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F496,
        &autotune::Autotune_bwd_kernel_fuse__A1__F497,
        &autotune::Autotune_bwd_kernel_fuse__A1__F498,
        &autotune::Autotune_bwd_kernel_fuse__A1__F499,
        &autotune::Autotune_bwd_kernel_fuse__A1__F500,
        &autotune::Autotune_bwd_kernel_fuse__A1__F501,
        &autotune::Autotune_bwd_kernel_fuse__A1__F502,
        &autotune::Autotune_bwd_kernel_fuse__A1__F503,
        &autotune::Autotune_bwd_kernel_fuse__A1__F504,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F506,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F508,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F510,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F512,
        &autotune::Autotune_bwd_kernel_fuse__A1__F513,
        &autotune::Autotune_bwd_kernel_fuse__A1__F514,
        &autotune::Autotune_bwd_kernel_fuse__A1__F515,
        &autotune::Autotune_bwd_kernel_fuse__A1__F516,
        &autotune::Autotune_bwd_kernel_fuse__A1__F517,
        &autotune::Autotune_bwd_kernel_fuse__A1__F518,
        &autotune::Autotune_bwd_kernel_fuse__A1__F519,
        &autotune::Autotune_bwd_kernel_fuse__A1__F520,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F522,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F524,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A1__F526,
        nullptr,
    },
    {
        &autotune::Autotune_bwd_kernel_fuse__A2__F0,
        &autotune::Autotune_bwd_kernel_fuse__A2__F1,
        &autotune::Autotune_bwd_kernel_fuse__A2__F2,
        &autotune::Autotune_bwd_kernel_fuse__A2__F3,
        &autotune::Autotune_bwd_kernel_fuse__A2__F4,
        &autotune::Autotune_bwd_kernel_fuse__A2__F5,
        &autotune::Autotune_bwd_kernel_fuse__A2__F6,
        &autotune::Autotune_bwd_kernel_fuse__A2__F7,
        &autotune::Autotune_bwd_kernel_fuse__A2__F8,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F10,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F12,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F14,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F16,
        &autotune::Autotune_bwd_kernel_fuse__A2__F17,
        &autotune::Autotune_bwd_kernel_fuse__A2__F18,
        &autotune::Autotune_bwd_kernel_fuse__A2__F19,
        &autotune::Autotune_bwd_kernel_fuse__A2__F20,
        &autotune::Autotune_bwd_kernel_fuse__A2__F21,
        &autotune::Autotune_bwd_kernel_fuse__A2__F22,
        &autotune::Autotune_bwd_kernel_fuse__A2__F23,
        &autotune::Autotune_bwd_kernel_fuse__A2__F24,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F26,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F28,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F30,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F32,
        &autotune::Autotune_bwd_kernel_fuse__A2__F33,
        &autotune::Autotune_bwd_kernel_fuse__A2__F34,
        &autotune::Autotune_bwd_kernel_fuse__A2__F35,
        &autotune::Autotune_bwd_kernel_fuse__A2__F36,
        &autotune::Autotune_bwd_kernel_fuse__A2__F37,
        &autotune::Autotune_bwd_kernel_fuse__A2__F38,
        &autotune::Autotune_bwd_kernel_fuse__A2__F39,
        &autotune::Autotune_bwd_kernel_fuse__A2__F40,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F42,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F44,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F46,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F48,
        &autotune::Autotune_bwd_kernel_fuse__A2__F49,
        &autotune::Autotune_bwd_kernel_fuse__A2__F50,
        &autotune::Autotune_bwd_kernel_fuse__A2__F51,
        &autotune::Autotune_bwd_kernel_fuse__A2__F52,
        &autotune::Autotune_bwd_kernel_fuse__A2__F53,
        &autotune::Autotune_bwd_kernel_fuse__A2__F54,
        &autotune::Autotune_bwd_kernel_fuse__A2__F55,
        &autotune::Autotune_bwd_kernel_fuse__A2__F56,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F58,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F60,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F62,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F64,
        &autotune::Autotune_bwd_kernel_fuse__A2__F65,
        &autotune::Autotune_bwd_kernel_fuse__A2__F66,
        &autotune::Autotune_bwd_kernel_fuse__A2__F67,
        &autotune::Autotune_bwd_kernel_fuse__A2__F68,
        &autotune::Autotune_bwd_kernel_fuse__A2__F69,
        &autotune::Autotune_bwd_kernel_fuse__A2__F70,
        &autotune::Autotune_bwd_kernel_fuse__A2__F71,
        &autotune::Autotune_bwd_kernel_fuse__A2__F72,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F74,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F76,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F78,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F80,
        &autotune::Autotune_bwd_kernel_fuse__A2__F81,
        &autotune::Autotune_bwd_kernel_fuse__A2__F82,
        &autotune::Autotune_bwd_kernel_fuse__A2__F83,
        &autotune::Autotune_bwd_kernel_fuse__A2__F84,
        &autotune::Autotune_bwd_kernel_fuse__A2__F85,
        &autotune::Autotune_bwd_kernel_fuse__A2__F86,
        &autotune::Autotune_bwd_kernel_fuse__A2__F87,
        &autotune::Autotune_bwd_kernel_fuse__A2__F88,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F90,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F92,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F94,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F96,
        &autotune::Autotune_bwd_kernel_fuse__A2__F97,
        &autotune::Autotune_bwd_kernel_fuse__A2__F98,
        &autotune::Autotune_bwd_kernel_fuse__A2__F99,
        &autotune::Autotune_bwd_kernel_fuse__A2__F100,
        &autotune::Autotune_bwd_kernel_fuse__A2__F101,
        &autotune::Autotune_bwd_kernel_fuse__A2__F102,
        &autotune::Autotune_bwd_kernel_fuse__A2__F103,
        &autotune::Autotune_bwd_kernel_fuse__A2__F104,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F106,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F108,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F110,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F112,
        &autotune::Autotune_bwd_kernel_fuse__A2__F113,
        &autotune::Autotune_bwd_kernel_fuse__A2__F114,
        &autotune::Autotune_bwd_kernel_fuse__A2__F115,
        &autotune::Autotune_bwd_kernel_fuse__A2__F116,
        &autotune::Autotune_bwd_kernel_fuse__A2__F117,
        &autotune::Autotune_bwd_kernel_fuse__A2__F118,
        &autotune::Autotune_bwd_kernel_fuse__A2__F119,
        &autotune::Autotune_bwd_kernel_fuse__A2__F120,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F122,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F124,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F126,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F128,
        &autotune::Autotune_bwd_kernel_fuse__A2__F129,
        &autotune::Autotune_bwd_kernel_fuse__A2__F130,
        &autotune::Autotune_bwd_kernel_fuse__A2__F131,
        &autotune::Autotune_bwd_kernel_fuse__A2__F132,
        &autotune::Autotune_bwd_kernel_fuse__A2__F133,
        &autotune::Autotune_bwd_kernel_fuse__A2__F134,
        &autotune::Autotune_bwd_kernel_fuse__A2__F135,
        &autotune::Autotune_bwd_kernel_fuse__A2__F136,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F138,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F140,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F142,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F144,
        &autotune::Autotune_bwd_kernel_fuse__A2__F145,
        &autotune::Autotune_bwd_kernel_fuse__A2__F146,
        &autotune::Autotune_bwd_kernel_fuse__A2__F147,
        &autotune::Autotune_bwd_kernel_fuse__A2__F148,
        &autotune::Autotune_bwd_kernel_fuse__A2__F149,
        &autotune::Autotune_bwd_kernel_fuse__A2__F150,
        &autotune::Autotune_bwd_kernel_fuse__A2__F151,
        &autotune::Autotune_bwd_kernel_fuse__A2__F152,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F154,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F156,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F158,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F160,
        &autotune::Autotune_bwd_kernel_fuse__A2__F161,
        &autotune::Autotune_bwd_kernel_fuse__A2__F162,
        &autotune::Autotune_bwd_kernel_fuse__A2__F163,
        &autotune::Autotune_bwd_kernel_fuse__A2__F164,
        &autotune::Autotune_bwd_kernel_fuse__A2__F165,
        &autotune::Autotune_bwd_kernel_fuse__A2__F166,
        &autotune::Autotune_bwd_kernel_fuse__A2__F167,
        &autotune::Autotune_bwd_kernel_fuse__A2__F168,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F170,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F172,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F174,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F176,
        &autotune::Autotune_bwd_kernel_fuse__A2__F177,
        &autotune::Autotune_bwd_kernel_fuse__A2__F178,
        &autotune::Autotune_bwd_kernel_fuse__A2__F179,
        &autotune::Autotune_bwd_kernel_fuse__A2__F180,
        &autotune::Autotune_bwd_kernel_fuse__A2__F181,
        &autotune::Autotune_bwd_kernel_fuse__A2__F182,
        &autotune::Autotune_bwd_kernel_fuse__A2__F183,
        &autotune::Autotune_bwd_kernel_fuse__A2__F184,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F186,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F188,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F190,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F192,
        &autotune::Autotune_bwd_kernel_fuse__A2__F193,
        &autotune::Autotune_bwd_kernel_fuse__A2__F194,
        &autotune::Autotune_bwd_kernel_fuse__A2__F195,
        &autotune::Autotune_bwd_kernel_fuse__A2__F196,
        &autotune::Autotune_bwd_kernel_fuse__A2__F197,
        &autotune::Autotune_bwd_kernel_fuse__A2__F198,
        &autotune::Autotune_bwd_kernel_fuse__A2__F199,
        &autotune::Autotune_bwd_kernel_fuse__A2__F200,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F202,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F204,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F206,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F208,
        &autotune::Autotune_bwd_kernel_fuse__A2__F209,
        &autotune::Autotune_bwd_kernel_fuse__A2__F210,
        &autotune::Autotune_bwd_kernel_fuse__A2__F211,
        &autotune::Autotune_bwd_kernel_fuse__A2__F212,
        &autotune::Autotune_bwd_kernel_fuse__A2__F213,
        &autotune::Autotune_bwd_kernel_fuse__A2__F214,
        &autotune::Autotune_bwd_kernel_fuse__A2__F215,
        &autotune::Autotune_bwd_kernel_fuse__A2__F216,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F218,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F220,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F222,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F224,
        &autotune::Autotune_bwd_kernel_fuse__A2__F225,
        &autotune::Autotune_bwd_kernel_fuse__A2__F226,
        &autotune::Autotune_bwd_kernel_fuse__A2__F227,
        &autotune::Autotune_bwd_kernel_fuse__A2__F228,
        &autotune::Autotune_bwd_kernel_fuse__A2__F229,
        &autotune::Autotune_bwd_kernel_fuse__A2__F230,
        &autotune::Autotune_bwd_kernel_fuse__A2__F231,
        &autotune::Autotune_bwd_kernel_fuse__A2__F232,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F234,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F236,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F238,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F240,
        &autotune::Autotune_bwd_kernel_fuse__A2__F241,
        &autotune::Autotune_bwd_kernel_fuse__A2__F242,
        &autotune::Autotune_bwd_kernel_fuse__A2__F243,
        &autotune::Autotune_bwd_kernel_fuse__A2__F244,
        &autotune::Autotune_bwd_kernel_fuse__A2__F245,
        &autotune::Autotune_bwd_kernel_fuse__A2__F246,
        &autotune::Autotune_bwd_kernel_fuse__A2__F247,
        &autotune::Autotune_bwd_kernel_fuse__A2__F248,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F250,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F252,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F254,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F256,
        &autotune::Autotune_bwd_kernel_fuse__A2__F257,
        &autotune::Autotune_bwd_kernel_fuse__A2__F258,
        &autotune::Autotune_bwd_kernel_fuse__A2__F259,
        &autotune::Autotune_bwd_kernel_fuse__A2__F260,
        &autotune::Autotune_bwd_kernel_fuse__A2__F261,
        &autotune::Autotune_bwd_kernel_fuse__A2__F262,
        &autotune::Autotune_bwd_kernel_fuse__A2__F263,
        &autotune::Autotune_bwd_kernel_fuse__A2__F264,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F266,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F268,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F270,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F272,
        &autotune::Autotune_bwd_kernel_fuse__A2__F273,
        &autotune::Autotune_bwd_kernel_fuse__A2__F274,
        &autotune::Autotune_bwd_kernel_fuse__A2__F275,
        &autotune::Autotune_bwd_kernel_fuse__A2__F276,
        &autotune::Autotune_bwd_kernel_fuse__A2__F277,
        &autotune::Autotune_bwd_kernel_fuse__A2__F278,
        &autotune::Autotune_bwd_kernel_fuse__A2__F279,
        &autotune::Autotune_bwd_kernel_fuse__A2__F280,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F282,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F284,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F286,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F288,
        &autotune::Autotune_bwd_kernel_fuse__A2__F289,
        &autotune::Autotune_bwd_kernel_fuse__A2__F290,
        &autotune::Autotune_bwd_kernel_fuse__A2__F291,
        &autotune::Autotune_bwd_kernel_fuse__A2__F292,
        &autotune::Autotune_bwd_kernel_fuse__A2__F293,
        &autotune::Autotune_bwd_kernel_fuse__A2__F294,
        &autotune::Autotune_bwd_kernel_fuse__A2__F295,
        &autotune::Autotune_bwd_kernel_fuse__A2__F296,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F298,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F300,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F302,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F304,
        &autotune::Autotune_bwd_kernel_fuse__A2__F305,
        &autotune::Autotune_bwd_kernel_fuse__A2__F306,
        &autotune::Autotune_bwd_kernel_fuse__A2__F307,
        &autotune::Autotune_bwd_kernel_fuse__A2__F308,
        &autotune::Autotune_bwd_kernel_fuse__A2__F309,
        &autotune::Autotune_bwd_kernel_fuse__A2__F310,
        &autotune::Autotune_bwd_kernel_fuse__A2__F311,
        &autotune::Autotune_bwd_kernel_fuse__A2__F312,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F314,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F316,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F318,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F320,
        &autotune::Autotune_bwd_kernel_fuse__A2__F321,
        &autotune::Autotune_bwd_kernel_fuse__A2__F322,
        &autotune::Autotune_bwd_kernel_fuse__A2__F323,
        &autotune::Autotune_bwd_kernel_fuse__A2__F324,
        &autotune::Autotune_bwd_kernel_fuse__A2__F325,
        &autotune::Autotune_bwd_kernel_fuse__A2__F326,
        &autotune::Autotune_bwd_kernel_fuse__A2__F327,
        &autotune::Autotune_bwd_kernel_fuse__A2__F328,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F330,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F332,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F334,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F336,
        &autotune::Autotune_bwd_kernel_fuse__A2__F337,
        &autotune::Autotune_bwd_kernel_fuse__A2__F338,
        &autotune::Autotune_bwd_kernel_fuse__A2__F339,
        &autotune::Autotune_bwd_kernel_fuse__A2__F340,
        &autotune::Autotune_bwd_kernel_fuse__A2__F341,
        &autotune::Autotune_bwd_kernel_fuse__A2__F342,
        &autotune::Autotune_bwd_kernel_fuse__A2__F343,
        &autotune::Autotune_bwd_kernel_fuse__A2__F344,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F346,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F348,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F350,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F352,
        &autotune::Autotune_bwd_kernel_fuse__A2__F353,
        &autotune::Autotune_bwd_kernel_fuse__A2__F354,
        &autotune::Autotune_bwd_kernel_fuse__A2__F355,
        &autotune::Autotune_bwd_kernel_fuse__A2__F356,
        &autotune::Autotune_bwd_kernel_fuse__A2__F357,
        &autotune::Autotune_bwd_kernel_fuse__A2__F358,
        &autotune::Autotune_bwd_kernel_fuse__A2__F359,
        &autotune::Autotune_bwd_kernel_fuse__A2__F360,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F362,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F364,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F366,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F368,
        &autotune::Autotune_bwd_kernel_fuse__A2__F369,
        &autotune::Autotune_bwd_kernel_fuse__A2__F370,
        &autotune::Autotune_bwd_kernel_fuse__A2__F371,
        &autotune::Autotune_bwd_kernel_fuse__A2__F372,
        &autotune::Autotune_bwd_kernel_fuse__A2__F373,
        &autotune::Autotune_bwd_kernel_fuse__A2__F374,
        &autotune::Autotune_bwd_kernel_fuse__A2__F375,
        &autotune::Autotune_bwd_kernel_fuse__A2__F376,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F378,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F380,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F382,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F384,
        &autotune::Autotune_bwd_kernel_fuse__A2__F385,
        &autotune::Autotune_bwd_kernel_fuse__A2__F386,
        &autotune::Autotune_bwd_kernel_fuse__A2__F387,
        &autotune::Autotune_bwd_kernel_fuse__A2__F388,
        &autotune::Autotune_bwd_kernel_fuse__A2__F389,
        &autotune::Autotune_bwd_kernel_fuse__A2__F390,
        &autotune::Autotune_bwd_kernel_fuse__A2__F391,
        &autotune::Autotune_bwd_kernel_fuse__A2__F392,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F394,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F396,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F398,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F400,
        &autotune::Autotune_bwd_kernel_fuse__A2__F401,
        &autotune::Autotune_bwd_kernel_fuse__A2__F402,
        &autotune::Autotune_bwd_kernel_fuse__A2__F403,
        &autotune::Autotune_bwd_kernel_fuse__A2__F404,
        &autotune::Autotune_bwd_kernel_fuse__A2__F405,
        &autotune::Autotune_bwd_kernel_fuse__A2__F406,
        &autotune::Autotune_bwd_kernel_fuse__A2__F407,
        &autotune::Autotune_bwd_kernel_fuse__A2__F408,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F410,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F412,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F414,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F416,
        &autotune::Autotune_bwd_kernel_fuse__A2__F417,
        &autotune::Autotune_bwd_kernel_fuse__A2__F418,
        &autotune::Autotune_bwd_kernel_fuse__A2__F419,
        &autotune::Autotune_bwd_kernel_fuse__A2__F420,
        &autotune::Autotune_bwd_kernel_fuse__A2__F421,
        &autotune::Autotune_bwd_kernel_fuse__A2__F422,
        &autotune::Autotune_bwd_kernel_fuse__A2__F423,
        &autotune::Autotune_bwd_kernel_fuse__A2__F424,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F426,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F428,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F430,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F432,
        &autotune::Autotune_bwd_kernel_fuse__A2__F433,
        &autotune::Autotune_bwd_kernel_fuse__A2__F434,
        &autotune::Autotune_bwd_kernel_fuse__A2__F435,
        &autotune::Autotune_bwd_kernel_fuse__A2__F436,
        &autotune::Autotune_bwd_kernel_fuse__A2__F437,
        &autotune::Autotune_bwd_kernel_fuse__A2__F438,
        &autotune::Autotune_bwd_kernel_fuse__A2__F439,
        &autotune::Autotune_bwd_kernel_fuse__A2__F440,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F442,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F444,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F446,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F448,
        &autotune::Autotune_bwd_kernel_fuse__A2__F449,
        &autotune::Autotune_bwd_kernel_fuse__A2__F450,
        &autotune::Autotune_bwd_kernel_fuse__A2__F451,
        &autotune::Autotune_bwd_kernel_fuse__A2__F452,
        &autotune::Autotune_bwd_kernel_fuse__A2__F453,
        &autotune::Autotune_bwd_kernel_fuse__A2__F454,
        &autotune::Autotune_bwd_kernel_fuse__A2__F455,
        &autotune::Autotune_bwd_kernel_fuse__A2__F456,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F458,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F460,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F462,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F464,
        &autotune::Autotune_bwd_kernel_fuse__A2__F465,
        &autotune::Autotune_bwd_kernel_fuse__A2__F466,
        &autotune::Autotune_bwd_kernel_fuse__A2__F467,
        &autotune::Autotune_bwd_kernel_fuse__A2__F468,
        &autotune::Autotune_bwd_kernel_fuse__A2__F469,
        &autotune::Autotune_bwd_kernel_fuse__A2__F470,
        &autotune::Autotune_bwd_kernel_fuse__A2__F471,
        &autotune::Autotune_bwd_kernel_fuse__A2__F472,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F474,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F476,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F478,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F480,
        &autotune::Autotune_bwd_kernel_fuse__A2__F481,
        &autotune::Autotune_bwd_kernel_fuse__A2__F482,
        &autotune::Autotune_bwd_kernel_fuse__A2__F483,
        &autotune::Autotune_bwd_kernel_fuse__A2__F484,
        &autotune::Autotune_bwd_kernel_fuse__A2__F485,
        &autotune::Autotune_bwd_kernel_fuse__A2__F486,
        &autotune::Autotune_bwd_kernel_fuse__A2__F487,
        &autotune::Autotune_bwd_kernel_fuse__A2__F488,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F490,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F492,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F494,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F496,
        &autotune::Autotune_bwd_kernel_fuse__A2__F497,
        &autotune::Autotune_bwd_kernel_fuse__A2__F498,
        &autotune::Autotune_bwd_kernel_fuse__A2__F499,
        &autotune::Autotune_bwd_kernel_fuse__A2__F500,
        &autotune::Autotune_bwd_kernel_fuse__A2__F501,
        &autotune::Autotune_bwd_kernel_fuse__A2__F502,
        &autotune::Autotune_bwd_kernel_fuse__A2__F503,
        &autotune::Autotune_bwd_kernel_fuse__A2__F504,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F506,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F508,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F510,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F512,
        &autotune::Autotune_bwd_kernel_fuse__A2__F513,
        &autotune::Autotune_bwd_kernel_fuse__A2__F514,
        &autotune::Autotune_bwd_kernel_fuse__A2__F515,
        &autotune::Autotune_bwd_kernel_fuse__A2__F516,
        &autotune::Autotune_bwd_kernel_fuse__A2__F517,
        &autotune::Autotune_bwd_kernel_fuse__A2__F518,
        &autotune::Autotune_bwd_kernel_fuse__A2__F519,
        &autotune::Autotune_bwd_kernel_fuse__A2__F520,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F522,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F524,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A2__F526,
        nullptr,
    },
    {
        &autotune::Autotune_bwd_kernel_fuse__A3__F0,
        &autotune::Autotune_bwd_kernel_fuse__A3__F1,
        &autotune::Autotune_bwd_kernel_fuse__A3__F2,
        &autotune::Autotune_bwd_kernel_fuse__A3__F3,
        &autotune::Autotune_bwd_kernel_fuse__A3__F4,
        &autotune::Autotune_bwd_kernel_fuse__A3__F5,
        &autotune::Autotune_bwd_kernel_fuse__A3__F6,
        &autotune::Autotune_bwd_kernel_fuse__A3__F7,
        &autotune::Autotune_bwd_kernel_fuse__A3__F8,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F10,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F12,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F14,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F16,
        &autotune::Autotune_bwd_kernel_fuse__A3__F17,
        &autotune::Autotune_bwd_kernel_fuse__A3__F18,
        &autotune::Autotune_bwd_kernel_fuse__A3__F19,
        &autotune::Autotune_bwd_kernel_fuse__A3__F20,
        &autotune::Autotune_bwd_kernel_fuse__A3__F21,
        &autotune::Autotune_bwd_kernel_fuse__A3__F22,
        &autotune::Autotune_bwd_kernel_fuse__A3__F23,
        &autotune::Autotune_bwd_kernel_fuse__A3__F24,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F26,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F28,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F30,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F32,
        &autotune::Autotune_bwd_kernel_fuse__A3__F33,
        &autotune::Autotune_bwd_kernel_fuse__A3__F34,
        &autotune::Autotune_bwd_kernel_fuse__A3__F35,
        &autotune::Autotune_bwd_kernel_fuse__A3__F36,
        &autotune::Autotune_bwd_kernel_fuse__A3__F37,
        &autotune::Autotune_bwd_kernel_fuse__A3__F38,
        &autotune::Autotune_bwd_kernel_fuse__A3__F39,
        &autotune::Autotune_bwd_kernel_fuse__A3__F40,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F42,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F44,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F46,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F48,
        &autotune::Autotune_bwd_kernel_fuse__A3__F49,
        &autotune::Autotune_bwd_kernel_fuse__A3__F50,
        &autotune::Autotune_bwd_kernel_fuse__A3__F51,
        &autotune::Autotune_bwd_kernel_fuse__A3__F52,
        &autotune::Autotune_bwd_kernel_fuse__A3__F53,
        &autotune::Autotune_bwd_kernel_fuse__A3__F54,
        &autotune::Autotune_bwd_kernel_fuse__A3__F55,
        &autotune::Autotune_bwd_kernel_fuse__A3__F56,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F58,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F60,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F62,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F64,
        &autotune::Autotune_bwd_kernel_fuse__A3__F65,
        &autotune::Autotune_bwd_kernel_fuse__A3__F66,
        &autotune::Autotune_bwd_kernel_fuse__A3__F67,
        &autotune::Autotune_bwd_kernel_fuse__A3__F68,
        &autotune::Autotune_bwd_kernel_fuse__A3__F69,
        &autotune::Autotune_bwd_kernel_fuse__A3__F70,
        &autotune::Autotune_bwd_kernel_fuse__A3__F71,
        &autotune::Autotune_bwd_kernel_fuse__A3__F72,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F74,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F76,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F78,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F80,
        &autotune::Autotune_bwd_kernel_fuse__A3__F81,
        &autotune::Autotune_bwd_kernel_fuse__A3__F82,
        &autotune::Autotune_bwd_kernel_fuse__A3__F83,
        &autotune::Autotune_bwd_kernel_fuse__A3__F84,
        &autotune::Autotune_bwd_kernel_fuse__A3__F85,
        &autotune::Autotune_bwd_kernel_fuse__A3__F86,
        &autotune::Autotune_bwd_kernel_fuse__A3__F87,
        &autotune::Autotune_bwd_kernel_fuse__A3__F88,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F90,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F92,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F94,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F96,
        &autotune::Autotune_bwd_kernel_fuse__A3__F97,
        &autotune::Autotune_bwd_kernel_fuse__A3__F98,
        &autotune::Autotune_bwd_kernel_fuse__A3__F99,
        &autotune::Autotune_bwd_kernel_fuse__A3__F100,
        &autotune::Autotune_bwd_kernel_fuse__A3__F101,
        &autotune::Autotune_bwd_kernel_fuse__A3__F102,
        &autotune::Autotune_bwd_kernel_fuse__A3__F103,
        &autotune::Autotune_bwd_kernel_fuse__A3__F104,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F106,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F108,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F110,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F112,
        &autotune::Autotune_bwd_kernel_fuse__A3__F113,
        &autotune::Autotune_bwd_kernel_fuse__A3__F114,
        &autotune::Autotune_bwd_kernel_fuse__A3__F115,
        &autotune::Autotune_bwd_kernel_fuse__A3__F116,
        &autotune::Autotune_bwd_kernel_fuse__A3__F117,
        &autotune::Autotune_bwd_kernel_fuse__A3__F118,
        &autotune::Autotune_bwd_kernel_fuse__A3__F119,
        &autotune::Autotune_bwd_kernel_fuse__A3__F120,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F122,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F124,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F126,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F128,
        &autotune::Autotune_bwd_kernel_fuse__A3__F129,
        &autotune::Autotune_bwd_kernel_fuse__A3__F130,
        &autotune::Autotune_bwd_kernel_fuse__A3__F131,
        &autotune::Autotune_bwd_kernel_fuse__A3__F132,
        &autotune::Autotune_bwd_kernel_fuse__A3__F133,
        &autotune::Autotune_bwd_kernel_fuse__A3__F134,
        &autotune::Autotune_bwd_kernel_fuse__A3__F135,
        &autotune::Autotune_bwd_kernel_fuse__A3__F136,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F138,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F140,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F142,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F144,
        &autotune::Autotune_bwd_kernel_fuse__A3__F145,
        &autotune::Autotune_bwd_kernel_fuse__A3__F146,
        &autotune::Autotune_bwd_kernel_fuse__A3__F147,
        &autotune::Autotune_bwd_kernel_fuse__A3__F148,
        &autotune::Autotune_bwd_kernel_fuse__A3__F149,
        &autotune::Autotune_bwd_kernel_fuse__A3__F150,
        &autotune::Autotune_bwd_kernel_fuse__A3__F151,
        &autotune::Autotune_bwd_kernel_fuse__A3__F152,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F154,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F156,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F158,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F160,
        &autotune::Autotune_bwd_kernel_fuse__A3__F161,
        &autotune::Autotune_bwd_kernel_fuse__A3__F162,
        &autotune::Autotune_bwd_kernel_fuse__A3__F163,
        &autotune::Autotune_bwd_kernel_fuse__A3__F164,
        &autotune::Autotune_bwd_kernel_fuse__A3__F165,
        &autotune::Autotune_bwd_kernel_fuse__A3__F166,
        &autotune::Autotune_bwd_kernel_fuse__A3__F167,
        &autotune::Autotune_bwd_kernel_fuse__A3__F168,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F170,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F172,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F174,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F176,
        &autotune::Autotune_bwd_kernel_fuse__A3__F177,
        &autotune::Autotune_bwd_kernel_fuse__A3__F178,
        &autotune::Autotune_bwd_kernel_fuse__A3__F179,
        &autotune::Autotune_bwd_kernel_fuse__A3__F180,
        &autotune::Autotune_bwd_kernel_fuse__A3__F181,
        &autotune::Autotune_bwd_kernel_fuse__A3__F182,
        &autotune::Autotune_bwd_kernel_fuse__A3__F183,
        &autotune::Autotune_bwd_kernel_fuse__A3__F184,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F186,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F188,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F190,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F192,
        &autotune::Autotune_bwd_kernel_fuse__A3__F193,
        &autotune::Autotune_bwd_kernel_fuse__A3__F194,
        &autotune::Autotune_bwd_kernel_fuse__A3__F195,
        &autotune::Autotune_bwd_kernel_fuse__A3__F196,
        &autotune::Autotune_bwd_kernel_fuse__A3__F197,
        &autotune::Autotune_bwd_kernel_fuse__A3__F198,
        &autotune::Autotune_bwd_kernel_fuse__A3__F199,
        &autotune::Autotune_bwd_kernel_fuse__A3__F200,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F202,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F204,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F206,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F208,
        &autotune::Autotune_bwd_kernel_fuse__A3__F209,
        &autotune::Autotune_bwd_kernel_fuse__A3__F210,
        &autotune::Autotune_bwd_kernel_fuse__A3__F211,
        &autotune::Autotune_bwd_kernel_fuse__A3__F212,
        &autotune::Autotune_bwd_kernel_fuse__A3__F213,
        &autotune::Autotune_bwd_kernel_fuse__A3__F214,
        &autotune::Autotune_bwd_kernel_fuse__A3__F215,
        &autotune::Autotune_bwd_kernel_fuse__A3__F216,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F218,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F220,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F222,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F224,
        &autotune::Autotune_bwd_kernel_fuse__A3__F225,
        &autotune::Autotune_bwd_kernel_fuse__A3__F226,
        &autotune::Autotune_bwd_kernel_fuse__A3__F227,
        &autotune::Autotune_bwd_kernel_fuse__A3__F228,
        &autotune::Autotune_bwd_kernel_fuse__A3__F229,
        &autotune::Autotune_bwd_kernel_fuse__A3__F230,
        &autotune::Autotune_bwd_kernel_fuse__A3__F231,
        &autotune::Autotune_bwd_kernel_fuse__A3__F232,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F234,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F236,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F238,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F240,
        &autotune::Autotune_bwd_kernel_fuse__A3__F241,
        &autotune::Autotune_bwd_kernel_fuse__A3__F242,
        &autotune::Autotune_bwd_kernel_fuse__A3__F243,
        &autotune::Autotune_bwd_kernel_fuse__A3__F244,
        &autotune::Autotune_bwd_kernel_fuse__A3__F245,
        &autotune::Autotune_bwd_kernel_fuse__A3__F246,
        &autotune::Autotune_bwd_kernel_fuse__A3__F247,
        &autotune::Autotune_bwd_kernel_fuse__A3__F248,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F250,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F252,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F254,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F256,
        &autotune::Autotune_bwd_kernel_fuse__A3__F257,
        &autotune::Autotune_bwd_kernel_fuse__A3__F258,
        &autotune::Autotune_bwd_kernel_fuse__A3__F259,
        &autotune::Autotune_bwd_kernel_fuse__A3__F260,
        &autotune::Autotune_bwd_kernel_fuse__A3__F261,
        &autotune::Autotune_bwd_kernel_fuse__A3__F262,
        &autotune::Autotune_bwd_kernel_fuse__A3__F263,
        &autotune::Autotune_bwd_kernel_fuse__A3__F264,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F266,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F268,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F270,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F272,
        &autotune::Autotune_bwd_kernel_fuse__A3__F273,
        &autotune::Autotune_bwd_kernel_fuse__A3__F274,
        &autotune::Autotune_bwd_kernel_fuse__A3__F275,
        &autotune::Autotune_bwd_kernel_fuse__A3__F276,
        &autotune::Autotune_bwd_kernel_fuse__A3__F277,
        &autotune::Autotune_bwd_kernel_fuse__A3__F278,
        &autotune::Autotune_bwd_kernel_fuse__A3__F279,
        &autotune::Autotune_bwd_kernel_fuse__A3__F280,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F282,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F284,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F286,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F288,
        &autotune::Autotune_bwd_kernel_fuse__A3__F289,
        &autotune::Autotune_bwd_kernel_fuse__A3__F290,
        &autotune::Autotune_bwd_kernel_fuse__A3__F291,
        &autotune::Autotune_bwd_kernel_fuse__A3__F292,
        &autotune::Autotune_bwd_kernel_fuse__A3__F293,
        &autotune::Autotune_bwd_kernel_fuse__A3__F294,
        &autotune::Autotune_bwd_kernel_fuse__A3__F295,
        &autotune::Autotune_bwd_kernel_fuse__A3__F296,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F298,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F300,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F302,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F304,
        &autotune::Autotune_bwd_kernel_fuse__A3__F305,
        &autotune::Autotune_bwd_kernel_fuse__A3__F306,
        &autotune::Autotune_bwd_kernel_fuse__A3__F307,
        &autotune::Autotune_bwd_kernel_fuse__A3__F308,
        &autotune::Autotune_bwd_kernel_fuse__A3__F309,
        &autotune::Autotune_bwd_kernel_fuse__A3__F310,
        &autotune::Autotune_bwd_kernel_fuse__A3__F311,
        &autotune::Autotune_bwd_kernel_fuse__A3__F312,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F314,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F316,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F318,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F320,
        &autotune::Autotune_bwd_kernel_fuse__A3__F321,
        &autotune::Autotune_bwd_kernel_fuse__A3__F322,
        &autotune::Autotune_bwd_kernel_fuse__A3__F323,
        &autotune::Autotune_bwd_kernel_fuse__A3__F324,
        &autotune::Autotune_bwd_kernel_fuse__A3__F325,
        &autotune::Autotune_bwd_kernel_fuse__A3__F326,
        &autotune::Autotune_bwd_kernel_fuse__A3__F327,
        &autotune::Autotune_bwd_kernel_fuse__A3__F328,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F330,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F332,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F334,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F336,
        &autotune::Autotune_bwd_kernel_fuse__A3__F337,
        &autotune::Autotune_bwd_kernel_fuse__A3__F338,
        &autotune::Autotune_bwd_kernel_fuse__A3__F339,
        &autotune::Autotune_bwd_kernel_fuse__A3__F340,
        &autotune::Autotune_bwd_kernel_fuse__A3__F341,
        &autotune::Autotune_bwd_kernel_fuse__A3__F342,
        &autotune::Autotune_bwd_kernel_fuse__A3__F343,
        &autotune::Autotune_bwd_kernel_fuse__A3__F344,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F346,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F348,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F350,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F352,
        &autotune::Autotune_bwd_kernel_fuse__A3__F353,
        &autotune::Autotune_bwd_kernel_fuse__A3__F354,
        &autotune::Autotune_bwd_kernel_fuse__A3__F355,
        &autotune::Autotune_bwd_kernel_fuse__A3__F356,
        &autotune::Autotune_bwd_kernel_fuse__A3__F357,
        &autotune::Autotune_bwd_kernel_fuse__A3__F358,
        &autotune::Autotune_bwd_kernel_fuse__A3__F359,
        &autotune::Autotune_bwd_kernel_fuse__A3__F360,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F362,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F364,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F366,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F368,
        &autotune::Autotune_bwd_kernel_fuse__A3__F369,
        &autotune::Autotune_bwd_kernel_fuse__A3__F370,
        &autotune::Autotune_bwd_kernel_fuse__A3__F371,
        &autotune::Autotune_bwd_kernel_fuse__A3__F372,
        &autotune::Autotune_bwd_kernel_fuse__A3__F373,
        &autotune::Autotune_bwd_kernel_fuse__A3__F374,
        &autotune::Autotune_bwd_kernel_fuse__A3__F375,
        &autotune::Autotune_bwd_kernel_fuse__A3__F376,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F378,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F380,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F382,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F384,
        &autotune::Autotune_bwd_kernel_fuse__A3__F385,
        &autotune::Autotune_bwd_kernel_fuse__A3__F386,
        &autotune::Autotune_bwd_kernel_fuse__A3__F387,
        &autotune::Autotune_bwd_kernel_fuse__A3__F388,
        &autotune::Autotune_bwd_kernel_fuse__A3__F389,
        &autotune::Autotune_bwd_kernel_fuse__A3__F390,
        &autotune::Autotune_bwd_kernel_fuse__A3__F391,
        &autotune::Autotune_bwd_kernel_fuse__A3__F392,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F394,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F396,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F398,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F400,
        &autotune::Autotune_bwd_kernel_fuse__A3__F401,
        &autotune::Autotune_bwd_kernel_fuse__A3__F402,
        &autotune::Autotune_bwd_kernel_fuse__A3__F403,
        &autotune::Autotune_bwd_kernel_fuse__A3__F404,
        &autotune::Autotune_bwd_kernel_fuse__A3__F405,
        &autotune::Autotune_bwd_kernel_fuse__A3__F406,
        &autotune::Autotune_bwd_kernel_fuse__A3__F407,
        &autotune::Autotune_bwd_kernel_fuse__A3__F408,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F410,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F412,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F414,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F416,
        &autotune::Autotune_bwd_kernel_fuse__A3__F417,
        &autotune::Autotune_bwd_kernel_fuse__A3__F418,
        &autotune::Autotune_bwd_kernel_fuse__A3__F419,
        &autotune::Autotune_bwd_kernel_fuse__A3__F420,
        &autotune::Autotune_bwd_kernel_fuse__A3__F421,
        &autotune::Autotune_bwd_kernel_fuse__A3__F422,
        &autotune::Autotune_bwd_kernel_fuse__A3__F423,
        &autotune::Autotune_bwd_kernel_fuse__A3__F424,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F426,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F428,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F430,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F432,
        &autotune::Autotune_bwd_kernel_fuse__A3__F433,
        &autotune::Autotune_bwd_kernel_fuse__A3__F434,
        &autotune::Autotune_bwd_kernel_fuse__A3__F435,
        &autotune::Autotune_bwd_kernel_fuse__A3__F436,
        &autotune::Autotune_bwd_kernel_fuse__A3__F437,
        &autotune::Autotune_bwd_kernel_fuse__A3__F438,
        &autotune::Autotune_bwd_kernel_fuse__A3__F439,
        &autotune::Autotune_bwd_kernel_fuse__A3__F440,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F442,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F444,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F446,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F448,
        &autotune::Autotune_bwd_kernel_fuse__A3__F449,
        &autotune::Autotune_bwd_kernel_fuse__A3__F450,
        &autotune::Autotune_bwd_kernel_fuse__A3__F451,
        &autotune::Autotune_bwd_kernel_fuse__A3__F452,
        &autotune::Autotune_bwd_kernel_fuse__A3__F453,
        &autotune::Autotune_bwd_kernel_fuse__A3__F454,
        &autotune::Autotune_bwd_kernel_fuse__A3__F455,
        &autotune::Autotune_bwd_kernel_fuse__A3__F456,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F458,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F460,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F462,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F464,
        &autotune::Autotune_bwd_kernel_fuse__A3__F465,
        &autotune::Autotune_bwd_kernel_fuse__A3__F466,
        &autotune::Autotune_bwd_kernel_fuse__A3__F467,
        &autotune::Autotune_bwd_kernel_fuse__A3__F468,
        &autotune::Autotune_bwd_kernel_fuse__A3__F469,
        &autotune::Autotune_bwd_kernel_fuse__A3__F470,
        &autotune::Autotune_bwd_kernel_fuse__A3__F471,
        &autotune::Autotune_bwd_kernel_fuse__A3__F472,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F474,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F476,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F478,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F480,
        &autotune::Autotune_bwd_kernel_fuse__A3__F481,
        &autotune::Autotune_bwd_kernel_fuse__A3__F482,
        &autotune::Autotune_bwd_kernel_fuse__A3__F483,
        &autotune::Autotune_bwd_kernel_fuse__A3__F484,
        &autotune::Autotune_bwd_kernel_fuse__A3__F485,
        &autotune::Autotune_bwd_kernel_fuse__A3__F486,
        &autotune::Autotune_bwd_kernel_fuse__A3__F487,
        &autotune::Autotune_bwd_kernel_fuse__A3__F488,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F490,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F492,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F494,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F496,
        &autotune::Autotune_bwd_kernel_fuse__A3__F497,
        &autotune::Autotune_bwd_kernel_fuse__A3__F498,
        &autotune::Autotune_bwd_kernel_fuse__A3__F499,
        &autotune::Autotune_bwd_kernel_fuse__A3__F500,
        &autotune::Autotune_bwd_kernel_fuse__A3__F501,
        &autotune::Autotune_bwd_kernel_fuse__A3__F502,
        &autotune::Autotune_bwd_kernel_fuse__A3__F503,
        &autotune::Autotune_bwd_kernel_fuse__A3__F504,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F506,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F508,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F510,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F512,
        &autotune::Autotune_bwd_kernel_fuse__A3__F513,
        &autotune::Autotune_bwd_kernel_fuse__A3__F514,
        &autotune::Autotune_bwd_kernel_fuse__A3__F515,
        &autotune::Autotune_bwd_kernel_fuse__A3__F516,
        &autotune::Autotune_bwd_kernel_fuse__A3__F517,
        &autotune::Autotune_bwd_kernel_fuse__A3__F518,
        &autotune::Autotune_bwd_kernel_fuse__A3__F519,
        &autotune::Autotune_bwd_kernel_fuse__A3__F520,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F522,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F524,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A3__F526,
        nullptr,
    },
    {
        &autotune::Autotune_bwd_kernel_fuse__A4__F0,
        &autotune::Autotune_bwd_kernel_fuse__A4__F1,
        &autotune::Autotune_bwd_kernel_fuse__A4__F2,
        &autotune::Autotune_bwd_kernel_fuse__A4__F3,
        &autotune::Autotune_bwd_kernel_fuse__A4__F4,
        &autotune::Autotune_bwd_kernel_fuse__A4__F5,
        &autotune::Autotune_bwd_kernel_fuse__A4__F6,
        &autotune::Autotune_bwd_kernel_fuse__A4__F7,
        &autotune::Autotune_bwd_kernel_fuse__A4__F8,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F10,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F12,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F14,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F16,
        &autotune::Autotune_bwd_kernel_fuse__A4__F17,
        &autotune::Autotune_bwd_kernel_fuse__A4__F18,
        &autotune::Autotune_bwd_kernel_fuse__A4__F19,
        &autotune::Autotune_bwd_kernel_fuse__A4__F20,
        &autotune::Autotune_bwd_kernel_fuse__A4__F21,
        &autotune::Autotune_bwd_kernel_fuse__A4__F22,
        &autotune::Autotune_bwd_kernel_fuse__A4__F23,
        &autotune::Autotune_bwd_kernel_fuse__A4__F24,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F26,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F28,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F30,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F32,
        &autotune::Autotune_bwd_kernel_fuse__A4__F33,
        &autotune::Autotune_bwd_kernel_fuse__A4__F34,
        &autotune::Autotune_bwd_kernel_fuse__A4__F35,
        &autotune::Autotune_bwd_kernel_fuse__A4__F36,
        &autotune::Autotune_bwd_kernel_fuse__A4__F37,
        &autotune::Autotune_bwd_kernel_fuse__A4__F38,
        &autotune::Autotune_bwd_kernel_fuse__A4__F39,
        &autotune::Autotune_bwd_kernel_fuse__A4__F40,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F42,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F44,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F46,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F48,
        &autotune::Autotune_bwd_kernel_fuse__A4__F49,
        &autotune::Autotune_bwd_kernel_fuse__A4__F50,
        &autotune::Autotune_bwd_kernel_fuse__A4__F51,
        &autotune::Autotune_bwd_kernel_fuse__A4__F52,
        &autotune::Autotune_bwd_kernel_fuse__A4__F53,
        &autotune::Autotune_bwd_kernel_fuse__A4__F54,
        &autotune::Autotune_bwd_kernel_fuse__A4__F55,
        &autotune::Autotune_bwd_kernel_fuse__A4__F56,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F58,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F60,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F62,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F64,
        &autotune::Autotune_bwd_kernel_fuse__A4__F65,
        &autotune::Autotune_bwd_kernel_fuse__A4__F66,
        &autotune::Autotune_bwd_kernel_fuse__A4__F67,
        &autotune::Autotune_bwd_kernel_fuse__A4__F68,
        &autotune::Autotune_bwd_kernel_fuse__A4__F69,
        &autotune::Autotune_bwd_kernel_fuse__A4__F70,
        &autotune::Autotune_bwd_kernel_fuse__A4__F71,
        &autotune::Autotune_bwd_kernel_fuse__A4__F72,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F74,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F76,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F78,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F80,
        &autotune::Autotune_bwd_kernel_fuse__A4__F81,
        &autotune::Autotune_bwd_kernel_fuse__A4__F82,
        &autotune::Autotune_bwd_kernel_fuse__A4__F83,
        &autotune::Autotune_bwd_kernel_fuse__A4__F84,
        &autotune::Autotune_bwd_kernel_fuse__A4__F85,
        &autotune::Autotune_bwd_kernel_fuse__A4__F86,
        &autotune::Autotune_bwd_kernel_fuse__A4__F87,
        &autotune::Autotune_bwd_kernel_fuse__A4__F88,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F90,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F92,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F94,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F96,
        &autotune::Autotune_bwd_kernel_fuse__A4__F97,
        &autotune::Autotune_bwd_kernel_fuse__A4__F98,
        &autotune::Autotune_bwd_kernel_fuse__A4__F99,
        &autotune::Autotune_bwd_kernel_fuse__A4__F100,
        &autotune::Autotune_bwd_kernel_fuse__A4__F101,
        &autotune::Autotune_bwd_kernel_fuse__A4__F102,
        &autotune::Autotune_bwd_kernel_fuse__A4__F103,
        &autotune::Autotune_bwd_kernel_fuse__A4__F104,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F106,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F108,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F110,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F112,
        &autotune::Autotune_bwd_kernel_fuse__A4__F113,
        &autotune::Autotune_bwd_kernel_fuse__A4__F114,
        &autotune::Autotune_bwd_kernel_fuse__A4__F115,
        &autotune::Autotune_bwd_kernel_fuse__A4__F116,
        &autotune::Autotune_bwd_kernel_fuse__A4__F117,
        &autotune::Autotune_bwd_kernel_fuse__A4__F118,
        &autotune::Autotune_bwd_kernel_fuse__A4__F119,
        &autotune::Autotune_bwd_kernel_fuse__A4__F120,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F122,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F124,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F126,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F128,
        &autotune::Autotune_bwd_kernel_fuse__A4__F129,
        &autotune::Autotune_bwd_kernel_fuse__A4__F130,
        &autotune::Autotune_bwd_kernel_fuse__A4__F131,
        &autotune::Autotune_bwd_kernel_fuse__A4__F132,
        &autotune::Autotune_bwd_kernel_fuse__A4__F133,
        &autotune::Autotune_bwd_kernel_fuse__A4__F134,
        &autotune::Autotune_bwd_kernel_fuse__A4__F135,
        &autotune::Autotune_bwd_kernel_fuse__A4__F136,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F138,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F140,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F142,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F144,
        &autotune::Autotune_bwd_kernel_fuse__A4__F145,
        &autotune::Autotune_bwd_kernel_fuse__A4__F146,
        &autotune::Autotune_bwd_kernel_fuse__A4__F147,
        &autotune::Autotune_bwd_kernel_fuse__A4__F148,
        &autotune::Autotune_bwd_kernel_fuse__A4__F149,
        &autotune::Autotune_bwd_kernel_fuse__A4__F150,
        &autotune::Autotune_bwd_kernel_fuse__A4__F151,
        &autotune::Autotune_bwd_kernel_fuse__A4__F152,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F154,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F156,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F158,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F160,
        &autotune::Autotune_bwd_kernel_fuse__A4__F161,
        &autotune::Autotune_bwd_kernel_fuse__A4__F162,
        &autotune::Autotune_bwd_kernel_fuse__A4__F163,
        &autotune::Autotune_bwd_kernel_fuse__A4__F164,
        &autotune::Autotune_bwd_kernel_fuse__A4__F165,
        &autotune::Autotune_bwd_kernel_fuse__A4__F166,
        &autotune::Autotune_bwd_kernel_fuse__A4__F167,
        &autotune::Autotune_bwd_kernel_fuse__A4__F168,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F170,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F172,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F174,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F176,
        &autotune::Autotune_bwd_kernel_fuse__A4__F177,
        &autotune::Autotune_bwd_kernel_fuse__A4__F178,
        &autotune::Autotune_bwd_kernel_fuse__A4__F179,
        &autotune::Autotune_bwd_kernel_fuse__A4__F180,
        &autotune::Autotune_bwd_kernel_fuse__A4__F181,
        &autotune::Autotune_bwd_kernel_fuse__A4__F182,
        &autotune::Autotune_bwd_kernel_fuse__A4__F183,
        &autotune::Autotune_bwd_kernel_fuse__A4__F184,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F186,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F188,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F190,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F192,
        &autotune::Autotune_bwd_kernel_fuse__A4__F193,
        &autotune::Autotune_bwd_kernel_fuse__A4__F194,
        &autotune::Autotune_bwd_kernel_fuse__A4__F195,
        &autotune::Autotune_bwd_kernel_fuse__A4__F196,
        &autotune::Autotune_bwd_kernel_fuse__A4__F197,
        &autotune::Autotune_bwd_kernel_fuse__A4__F198,
        &autotune::Autotune_bwd_kernel_fuse__A4__F199,
        &autotune::Autotune_bwd_kernel_fuse__A4__F200,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F202,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F204,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F206,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F208,
        &autotune::Autotune_bwd_kernel_fuse__A4__F209,
        &autotune::Autotune_bwd_kernel_fuse__A4__F210,
        &autotune::Autotune_bwd_kernel_fuse__A4__F211,
        &autotune::Autotune_bwd_kernel_fuse__A4__F212,
        &autotune::Autotune_bwd_kernel_fuse__A4__F213,
        &autotune::Autotune_bwd_kernel_fuse__A4__F214,
        &autotune::Autotune_bwd_kernel_fuse__A4__F215,
        &autotune::Autotune_bwd_kernel_fuse__A4__F216,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F218,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F220,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F222,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F224,
        &autotune::Autotune_bwd_kernel_fuse__A4__F225,
        &autotune::Autotune_bwd_kernel_fuse__A4__F226,
        &autotune::Autotune_bwd_kernel_fuse__A4__F227,
        &autotune::Autotune_bwd_kernel_fuse__A4__F228,
        &autotune::Autotune_bwd_kernel_fuse__A4__F229,
        &autotune::Autotune_bwd_kernel_fuse__A4__F230,
        &autotune::Autotune_bwd_kernel_fuse__A4__F231,
        &autotune::Autotune_bwd_kernel_fuse__A4__F232,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F234,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F236,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F238,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F240,
        &autotune::Autotune_bwd_kernel_fuse__A4__F241,
        &autotune::Autotune_bwd_kernel_fuse__A4__F242,
        &autotune::Autotune_bwd_kernel_fuse__A4__F243,
        &autotune::Autotune_bwd_kernel_fuse__A4__F244,
        &autotune::Autotune_bwd_kernel_fuse__A4__F245,
        &autotune::Autotune_bwd_kernel_fuse__A4__F246,
        &autotune::Autotune_bwd_kernel_fuse__A4__F247,
        &autotune::Autotune_bwd_kernel_fuse__A4__F248,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F250,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F252,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F254,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F256,
        &autotune::Autotune_bwd_kernel_fuse__A4__F257,
        &autotune::Autotune_bwd_kernel_fuse__A4__F258,
        &autotune::Autotune_bwd_kernel_fuse__A4__F259,
        &autotune::Autotune_bwd_kernel_fuse__A4__F260,
        &autotune::Autotune_bwd_kernel_fuse__A4__F261,
        &autotune::Autotune_bwd_kernel_fuse__A4__F262,
        &autotune::Autotune_bwd_kernel_fuse__A4__F263,
        &autotune::Autotune_bwd_kernel_fuse__A4__F264,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F266,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F268,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F270,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F272,
        &autotune::Autotune_bwd_kernel_fuse__A4__F273,
        &autotune::Autotune_bwd_kernel_fuse__A4__F274,
        &autotune::Autotune_bwd_kernel_fuse__A4__F275,
        &autotune::Autotune_bwd_kernel_fuse__A4__F276,
        &autotune::Autotune_bwd_kernel_fuse__A4__F277,
        &autotune::Autotune_bwd_kernel_fuse__A4__F278,
        &autotune::Autotune_bwd_kernel_fuse__A4__F279,
        &autotune::Autotune_bwd_kernel_fuse__A4__F280,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F282,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F284,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F286,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F288,
        &autotune::Autotune_bwd_kernel_fuse__A4__F289,
        &autotune::Autotune_bwd_kernel_fuse__A4__F290,
        &autotune::Autotune_bwd_kernel_fuse__A4__F291,
        &autotune::Autotune_bwd_kernel_fuse__A4__F292,
        &autotune::Autotune_bwd_kernel_fuse__A4__F293,
        &autotune::Autotune_bwd_kernel_fuse__A4__F294,
        &autotune::Autotune_bwd_kernel_fuse__A4__F295,
        &autotune::Autotune_bwd_kernel_fuse__A4__F296,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F298,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F300,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F302,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F304,
        &autotune::Autotune_bwd_kernel_fuse__A4__F305,
        &autotune::Autotune_bwd_kernel_fuse__A4__F306,
        &autotune::Autotune_bwd_kernel_fuse__A4__F307,
        &autotune::Autotune_bwd_kernel_fuse__A4__F308,
        &autotune::Autotune_bwd_kernel_fuse__A4__F309,
        &autotune::Autotune_bwd_kernel_fuse__A4__F310,
        &autotune::Autotune_bwd_kernel_fuse__A4__F311,
        &autotune::Autotune_bwd_kernel_fuse__A4__F312,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F314,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F316,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F318,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F320,
        &autotune::Autotune_bwd_kernel_fuse__A4__F321,
        &autotune::Autotune_bwd_kernel_fuse__A4__F322,
        &autotune::Autotune_bwd_kernel_fuse__A4__F323,
        &autotune::Autotune_bwd_kernel_fuse__A4__F324,
        &autotune::Autotune_bwd_kernel_fuse__A4__F325,
        &autotune::Autotune_bwd_kernel_fuse__A4__F326,
        &autotune::Autotune_bwd_kernel_fuse__A4__F327,
        &autotune::Autotune_bwd_kernel_fuse__A4__F328,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F330,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F332,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F334,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F336,
        &autotune::Autotune_bwd_kernel_fuse__A4__F337,
        &autotune::Autotune_bwd_kernel_fuse__A4__F338,
        &autotune::Autotune_bwd_kernel_fuse__A4__F339,
        &autotune::Autotune_bwd_kernel_fuse__A4__F340,
        &autotune::Autotune_bwd_kernel_fuse__A4__F341,
        &autotune::Autotune_bwd_kernel_fuse__A4__F342,
        &autotune::Autotune_bwd_kernel_fuse__A4__F343,
        &autotune::Autotune_bwd_kernel_fuse__A4__F344,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F346,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F348,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F350,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F352,
        &autotune::Autotune_bwd_kernel_fuse__A4__F353,
        &autotune::Autotune_bwd_kernel_fuse__A4__F354,
        &autotune::Autotune_bwd_kernel_fuse__A4__F355,
        &autotune::Autotune_bwd_kernel_fuse__A4__F356,
        &autotune::Autotune_bwd_kernel_fuse__A4__F357,
        &autotune::Autotune_bwd_kernel_fuse__A4__F358,
        &autotune::Autotune_bwd_kernel_fuse__A4__F359,
        &autotune::Autotune_bwd_kernel_fuse__A4__F360,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F362,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F364,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F366,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F368,
        &autotune::Autotune_bwd_kernel_fuse__A4__F369,
        &autotune::Autotune_bwd_kernel_fuse__A4__F370,
        &autotune::Autotune_bwd_kernel_fuse__A4__F371,
        &autotune::Autotune_bwd_kernel_fuse__A4__F372,
        &autotune::Autotune_bwd_kernel_fuse__A4__F373,
        &autotune::Autotune_bwd_kernel_fuse__A4__F374,
        &autotune::Autotune_bwd_kernel_fuse__A4__F375,
        &autotune::Autotune_bwd_kernel_fuse__A4__F376,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F378,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F380,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F382,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F384,
        &autotune::Autotune_bwd_kernel_fuse__A4__F385,
        &autotune::Autotune_bwd_kernel_fuse__A4__F386,
        &autotune::Autotune_bwd_kernel_fuse__A4__F387,
        &autotune::Autotune_bwd_kernel_fuse__A4__F388,
        &autotune::Autotune_bwd_kernel_fuse__A4__F389,
        &autotune::Autotune_bwd_kernel_fuse__A4__F390,
        &autotune::Autotune_bwd_kernel_fuse__A4__F391,
        &autotune::Autotune_bwd_kernel_fuse__A4__F392,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F394,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F396,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F398,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F400,
        &autotune::Autotune_bwd_kernel_fuse__A4__F401,
        &autotune::Autotune_bwd_kernel_fuse__A4__F402,
        &autotune::Autotune_bwd_kernel_fuse__A4__F403,
        &autotune::Autotune_bwd_kernel_fuse__A4__F404,
        &autotune::Autotune_bwd_kernel_fuse__A4__F405,
        &autotune::Autotune_bwd_kernel_fuse__A4__F406,
        &autotune::Autotune_bwd_kernel_fuse__A4__F407,
        &autotune::Autotune_bwd_kernel_fuse__A4__F408,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F410,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F412,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F414,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F416,
        &autotune::Autotune_bwd_kernel_fuse__A4__F417,
        &autotune::Autotune_bwd_kernel_fuse__A4__F418,
        &autotune::Autotune_bwd_kernel_fuse__A4__F419,
        &autotune::Autotune_bwd_kernel_fuse__A4__F420,
        &autotune::Autotune_bwd_kernel_fuse__A4__F421,
        &autotune::Autotune_bwd_kernel_fuse__A4__F422,
        &autotune::Autotune_bwd_kernel_fuse__A4__F423,
        &autotune::Autotune_bwd_kernel_fuse__A4__F424,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F426,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F428,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F430,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F432,
        &autotune::Autotune_bwd_kernel_fuse__A4__F433,
        &autotune::Autotune_bwd_kernel_fuse__A4__F434,
        &autotune::Autotune_bwd_kernel_fuse__A4__F435,
        &autotune::Autotune_bwd_kernel_fuse__A4__F436,
        &autotune::Autotune_bwd_kernel_fuse__A4__F437,
        &autotune::Autotune_bwd_kernel_fuse__A4__F438,
        &autotune::Autotune_bwd_kernel_fuse__A4__F439,
        &autotune::Autotune_bwd_kernel_fuse__A4__F440,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F442,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F444,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F446,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F448,
        &autotune::Autotune_bwd_kernel_fuse__A4__F449,
        &autotune::Autotune_bwd_kernel_fuse__A4__F450,
        &autotune::Autotune_bwd_kernel_fuse__A4__F451,
        &autotune::Autotune_bwd_kernel_fuse__A4__F452,
        &autotune::Autotune_bwd_kernel_fuse__A4__F453,
        &autotune::Autotune_bwd_kernel_fuse__A4__F454,
        &autotune::Autotune_bwd_kernel_fuse__A4__F455,
        &autotune::Autotune_bwd_kernel_fuse__A4__F456,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F458,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F460,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F462,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F464,
        &autotune::Autotune_bwd_kernel_fuse__A4__F465,
        &autotune::Autotune_bwd_kernel_fuse__A4__F466,
        &autotune::Autotune_bwd_kernel_fuse__A4__F467,
        &autotune::Autotune_bwd_kernel_fuse__A4__F468,
        &autotune::Autotune_bwd_kernel_fuse__A4__F469,
        &autotune::Autotune_bwd_kernel_fuse__A4__F470,
        &autotune::Autotune_bwd_kernel_fuse__A4__F471,
        &autotune::Autotune_bwd_kernel_fuse__A4__F472,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F474,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F476,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F478,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F480,
        &autotune::Autotune_bwd_kernel_fuse__A4__F481,
        &autotune::Autotune_bwd_kernel_fuse__A4__F482,
        &autotune::Autotune_bwd_kernel_fuse__A4__F483,
        &autotune::Autotune_bwd_kernel_fuse__A4__F484,
        &autotune::Autotune_bwd_kernel_fuse__A4__F485,
        &autotune::Autotune_bwd_kernel_fuse__A4__F486,
        &autotune::Autotune_bwd_kernel_fuse__A4__F487,
        &autotune::Autotune_bwd_kernel_fuse__A4__F488,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F490,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F492,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F494,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F496,
        &autotune::Autotune_bwd_kernel_fuse__A4__F497,
        &autotune::Autotune_bwd_kernel_fuse__A4__F498,
        &autotune::Autotune_bwd_kernel_fuse__A4__F499,
        &autotune::Autotune_bwd_kernel_fuse__A4__F500,
        &autotune::Autotune_bwd_kernel_fuse__A4__F501,
        &autotune::Autotune_bwd_kernel_fuse__A4__F502,
        &autotune::Autotune_bwd_kernel_fuse__A4__F503,
        &autotune::Autotune_bwd_kernel_fuse__A4__F504,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F506,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F508,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F510,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F512,
        &autotune::Autotune_bwd_kernel_fuse__A4__F513,
        &autotune::Autotune_bwd_kernel_fuse__A4__F514,
        &autotune::Autotune_bwd_kernel_fuse__A4__F515,
        &autotune::Autotune_bwd_kernel_fuse__A4__F516,
        &autotune::Autotune_bwd_kernel_fuse__A4__F517,
        &autotune::Autotune_bwd_kernel_fuse__A4__F518,
        &autotune::Autotune_bwd_kernel_fuse__A4__F519,
        &autotune::Autotune_bwd_kernel_fuse__A4__F520,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F522,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F524,
        nullptr,
        &autotune::Autotune_bwd_kernel_fuse__A4__F526,
        nullptr,
    },
};

}

// vim: set fileencoding=utf-8

