// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A0__F386

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_16_0_F_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xa6aa001au, 0xdb1c4b20u, 0, 155 }, // a6aa001adb1c4b20 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__16_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x55008942u, 0xd30b90b9u, 0, 46 }, // 55008942d30b90b9 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x7b31a724u, 0x8999386cu, 0, 103 }, // 7b31a7248999386c = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x3dda70f3u, 0x2d29c352u, 0, 23 }, // 3dda70f32d29c352 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__16_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0xce5b49ecu, 0xa8d6c0a6u, 40, 126 }, // ce5b49eca8d6c0a6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xe27682b1u, 0x3eba502du, 40, 6 }, // e27682b13eba502d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x067d0918u, 0x7d375b61u, 40, 46 }, // 067d09187d375b61 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x324efc42u, 0x080c8d9au, 40, 23 }, // 324efc42080c8d9a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__32_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0x8b265ccau, 0x0896837eu, 80, 6 }, // 8b265cca0896837e = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__32_32__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x851bcbb6u, 0x41c37eabu, 80, 86 }, // 851bcbb641c37eab = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__32_32__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x73a0ebe1u, 0x8ebe5059u, 80, 46 }, // 73a0ebe18ebe5059 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__32_32__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xfaf8052fu, 0x89cff09bu, 80, 103 }, // faf8052f89cff09b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__32_32__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xb6e813acu, 0x071df83fu, 80, 23 }, // b6e813ac071df83f = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_0_F_T_0___gfx950__P__32_32__CO__wave4_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{11, 0, 7,12,10,12,12,12,12, 8},
 { 5, 3, 1,12, 8,12,12,12,12,12},
 { 9, 2, 3, 8,12,12,12,12,12,12},
 { 8, 4, 4, 8, 8,10, 8,10,12, 8},
 { 6, 7, 5, 5,12,12,12,12,12,12},
 { 4, 6, 6, 5, 7, 5, 5, 6, 6, 5},
 { 6, 4, 4, 4, 4, 6, 5, 7, 5, 6},
 { 6, 5, 5, 6, 5, 7, 5, 7, 5, 7},
 { 6, 7, 5, 6, 6, 5, 6, 5, 6, 7},
 { 6, 5, 6, 5, 5, 6, 6, 6, 6, 7}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A0__F386 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 0;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 0;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 0

// vim: set fileencoding=utf-8

