/* src/services/mod.rs
 *
 * Copyright 2025 Mission Center Developers
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

mod openrc;
mod systemd;

use magpie_platform::services::Service;
use std::num::NonZero;

enum ServiceCacheKind {
    None,
    SystemD(systemd::SystemD),
    OpenRC(openrc::OpenRC),
}

enum ServiceManagerKind {
    None,
    SystemD(systemd::ServiceManager),
    OpenRC(openrc::ServiceManager),
}

pub struct ServiceCache {
    kind: ServiceCacheKind,
    services: Vec<Service>,
}

pub struct ServiceManager {
    kind: ServiceManagerKind,
}

impl magpie_platform::services::ServiceCache for ServiceCache {
    fn new() -> Self
    where
        Self: Sized,
    {
        let kind = match systemd::SystemD::new() {
            Ok(systemd) => ServiceCacheKind::SystemD(systemd),
            Err(e) => {
                log::debug!("Failed to initialize SystemD service cache: {e}");
                match openrc::OpenRC::new() {
                    Ok(openrc) => ServiceCacheKind::OpenRC(openrc),
                    Err(e) => {
                        log::debug!("Failed to initialize OpenRC service cache: {e}");
                        ServiceCacheKind::None
                    }
                }
            }
        };

        Self {
            kind,
            services: Vec::new(),
        }
    }

    fn refresh(&mut self) {
        match &mut self.kind {
            ServiceCacheKind::SystemD(systemd) => {
                self.services = match systemd.list_services() {
                    Ok(services) => services,
                    Err(e) => {
                        log::debug!("Failed to list systemd services: {e}");
                        return;
                    }
                };
            }
            ServiceCacheKind::OpenRC(openrc) => {
                self.services = match openrc.list_services() {
                    Ok(services) => services,
                    Err(e) => {
                        log::debug!("Failed to list OpenRC services: {e}");
                        return;
                    }
                };
            }
            ServiceCacheKind::None => {}
        }
    }

    fn cached_entries(&self) -> &[Service] {
        &self.services
    }
}

impl magpie_platform::services::ServiceManager for ServiceManager {
    fn new() -> Self {
        let kind = match systemd::ServiceManager::new() {
            Ok(systemd) => ServiceManagerKind::SystemD(systemd),
            Err(e) => {
                log::debug!("Failed to initialize SystemD service manager: {e}");
                match openrc::ServiceManager::new() {
                    Ok(openrc) => ServiceManagerKind::OpenRC(openrc),
                    Err(e) => {
                        log::debug!("Failed to initialize OpenRC service manager: {e}");
                        ServiceManagerKind::None
                    }
                }
            }
        };

        Self { kind }
    }

    fn logs(&self, id: &str, pid: Option<NonZero<u32>>) -> Option<String> {
        match &self.kind {
            ServiceManagerKind::SystemD(systemd) => match systemd.service_logs(id, pid) {
                Ok(logs) => Some(logs),
                Err(e) => {
                    log::warn!("Failed to get logs for systemd service {id}: {e}");
                    None
                }
            },
            ServiceManagerKind::OpenRC(_) => None,
            ServiceManagerKind::None => None,
        }
    }

    fn start(&self, id: &str) {
        match &self.kind {
            ServiceManagerKind::SystemD(systemd) => match systemd.start_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to start systemd service {id}: {e}"),
            },
            ServiceManagerKind::OpenRC(openrc) => match openrc.start_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to start OpenRC service {id}: {e}"),
            },
            ServiceManagerKind::None => {}
        }
    }

    fn stop(&self, id: &str) {
        match &self.kind {
            ServiceManagerKind::SystemD(systemd) => match systemd.stop_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to stop systemd service {id}: {e}"),
            },
            ServiceManagerKind::OpenRC(openrc) => match openrc.stop_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to stop OpenRC service {id}: {e}"),
            },
            ServiceManagerKind::None => {}
        }
    }

    fn restart(&self, id: &str) {
        match &self.kind {
            ServiceManagerKind::SystemD(systemd) => match systemd.restart_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to restart systemd service {id}: {e}"),
            },
            ServiceManagerKind::OpenRC(openrc) => match openrc.restart_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to restart OpenRC service {id}: {e}"),
            },
            ServiceManagerKind::None => {}
        }
    }

    fn enable(&self, id: &str) {
        match &self.kind {
            ServiceManagerKind::SystemD(systemd) => match systemd.enable_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to enable systemd service {id}: {e}"),
            },
            ServiceManagerKind::OpenRC(openrc) => match openrc.enable_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to enable OpenRC service {id}: {e}"),
            },
            ServiceManagerKind::None => {}
        }
    }

    fn disable(&self, id: &str) {
        match &self.kind {
            ServiceManagerKind::SystemD(systemd) => match systemd.disable_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to disable systemd service {id}: {e}"),
            },
            ServiceManagerKind::OpenRC(openrc) => match openrc.disable_service(id) {
                Ok(_) => {}
                Err(e) => log::warn!("Failed to disable OpenRC service {id}: {e}"),
            },
            ServiceManagerKind::None => {}
        }
    }
}
