/* src/disks/mod.rs
 *
 * Copyright 2025 Mission Center Developers
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

use std::collections::HashSet;

use phf::phf_set;

use magpie_platform::disks::Disk;

use crate::{async_runtime, sync, system_bus};

pub use manager::DisksManager;
use stats::Stats;

mod manager;
mod smart_data;
mod stats;
mod util;

static IGNORED_DISK_PREFIXES: phf::Set<&'static str> = phf_set! {
    "loop",
    "ram",
    "zram",
    "fd",
    "md",
    "dm",
    "zd",
};

pub struct DisksCache {
    disks: Vec<Disk>,
    stats: Vec<Stats>,

    ignored: HashSet<String>,

    udisks2: Option<udisks2::Client>,
}

impl magpie_platform::disks::DisksCache for DisksCache {
    fn new() -> Self
    where
        Self: Sized,
    {
        let bus = match system_bus() {
            Some(bus) => bus.clone(),
            None => {
                log::warn!("Failed to connect to system bus");
                return Self {
                    disks: Vec::new(),
                    stats: Vec::new(),
                    ignored: HashSet::new(),
                    udisks2: None,
                };
            }
        };

        let rt = async_runtime();
        let udisks2 = match sync!(rt, udisks2::Client::new_for_connection(bus)) {
            Ok(udisks2) => Some(udisks2),
            Err(e) => {
                log::warn!("Failed to connect to udisks2: {}", e);
                None
            }
        };

        Self {
            disks: Vec::new(),
            stats: Vec::new(),
            ignored: HashSet::new(),
            udisks2,
        }
    }

    fn refresh(&mut self) {
        let udisks2 = self.udisks2.as_ref();
        let rt = async_runtime();

        let mut prev_disks = std::mem::take(&mut self.disks);
        let mut prev_stats = std::mem::take(&mut self.stats);

        let dir = match std::fs::read_dir("/sys/block") {
            Ok(dir) => dir,
            Err(e) => {
                log::warn!("Failed to read `/sys/block`: {e}");
                return;
            }
        };

        'outer: for entry in dir.filter_map(Result::ok) {
            let file_name = entry.file_name();
            let disk_id = file_name.to_string_lossy();
            let disk_id = disk_id.as_ref();

            if self.ignored.contains(disk_id) {
                continue;
            }

            for i in 2..=disk_id.len().min(4) {
                if IGNORED_DISK_PREFIXES.contains(&disk_id[..i]) {
                    self.ignored.insert(disk_id.to_string());
                    continue 'outer;
                }
            }

            let mut prev_disk_index = None;
            for (i, disk) in prev_disks.iter().enumerate() {
                if disk.id == disk_id {
                    prev_disk_index = Some(i);
                    break;
                }
            }

            let mut block_proxy = None;
            let mut drive_obj = None;

            let mut drive_proxy = None;
            let mut drive_ata = None;

            let mut partitions = Vec::new();
            let mut filesystems = Vec::new();

            if let Some((udisks2, object)) =
                udisks2.and_then(|client| util::object(client, disk_id).map(|obj| (client, obj)))
            {
                partitions = util::partitions(rt, udisks2, &object);
                filesystems = util::filesystems(rt, udisks2, &partitions);
                if let Some(bp) = util::block(rt, &object, disk_id) {
                    if let Some(obj) = util::drive(rt, udisks2, &bp, disk_id) {
                        drive_proxy = sync!(rt, obj.drive()).ok();
                        drive_ata = sync!(rt, obj.drive_ata()).ok();
                        drive_obj = Some(obj);
                    }
                    block_proxy = Some(bp);
                }
            }

            let (mut disk, stats) = match prev_disk_index {
                Some(i) => (prev_disks.swap_remove(i), prev_stats.swap_remove(i)),
                None => {
                    let mut disk = Disk::default();

                    let drive_nvme = drive_obj
                        .as_ref()
                        .and_then(|obj| sync!(rt, obj.nvme_controller()).ok());

                    disk.id = disk_id.to_string();
                    disk.model = util::model(disk_id);
                    disk.kind =
                        util::kind(rt, disk_id, drive_proxy.as_ref()).map(|kind| kind.into());
                    disk.smart_interface =
                        util::smart_interface(drive_ata.as_ref(), drive_nvme.as_ref())
                            .map(|kind| kind.into());
                    disk.is_system = util::is_system(disk_id, rt, &filesystems);
                    disk.ejectable = drive_proxy
                        .as_ref()
                        .and_then(|proxy| sync!(rt, proxy.ejectable()).ok())
                        .unwrap_or(false);

                    (disk, Stats::default())
                }
            };

            let mut new_stats = Stats::load(disk_id);
            new_stats.update(&stats, &mut disk);

            disk.capacity_bytes = util::capacity(rt, disk_id, block_proxy.as_ref()).unwrap_or(0);
            disk.formatted_bytes = util::formatted(rt, &partitions, &filesystems);
            disk.temperature_milli_k = util::temperature(disk_id, rt, drive_ata.as_ref());

            self.disks.push(disk);
            self.stats.push(new_stats);
        }
    }

    fn cached_entries(&self) -> &[Disk] {
        &self.disks
    }
}

#[cfg(test)]
mod tests {
    use magpie_platform::disks::DisksCache;

    #[test]
    fn test_disks_cache() {
        let mut cache = super::DisksCache::new();
        cache.refresh();
    }
}
