use std::sync::OnceLock;

pub(crate) fn path() -> &'static [String] {
    static PATH: OnceLock<Vec<String>> = OnceLock::new();

    PATH.get_or_init(|| {
        let path = std::env::var_os("PATH").unwrap_or_else(|| {
            "/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/snap/bin".into()
        });
        path.to_string_lossy()
            .split(':')
            .map(|c| c.to_string())
            .collect::<Vec<_>>()
    })
    .as_slice()
}

pub(crate) fn home() -> &'static str {
    static HOME: OnceLock<String> = OnceLock::new();

    HOME.get_or_init(|| std::env::var("HOME").unwrap_or_else(|_| "/home".into()))
        .as_str()
}

pub(crate) fn xdg_data_dirs() -> &'static [String] {
    static XDG_DATA_DIRS: OnceLock<Vec<String>> = OnceLock::new();

    XDG_DATA_DIRS
        .get_or_init(|| {
            let xdg_data_dirs = std::env::var_os("XDG_DATA_DIRS")
            .unwrap_or_else(|| {
                "/var/lib/flatpak/exports/share:/usr/local/share:/usr/share:/var/lib/snapd/desktop"
                    .into()
            })
            .to_string_lossy()
            .to_string();

            let mut res = xdg_data_dirs
                .split(':')
                .filter_map(|c| {
                    let c = c.trim();
                    if c.is_empty() {
                        None
                    } else {
                        Some(c.to_string())
                    }
                })
                .collect::<Vec<_>>();
            res.push(format!("{}/.local/share", home()));

            res
        })
        .as_slice()
}
